/*
 * Created on Aug 1, 2005
// Per VHA Directive 2004-038, this routine should not be modified.
//+---------------------------------------------------------------+
//| Property of the US Government.                                |
//| No permission to copy or redistribute this software is given. |
//| Use of unreleased versions of this software requires the user |
//| to execute a written test agreement with the VistA Imaging    |
//| Development Office of the Department of Veterans Affairs,     |
//| telephone (301) 734-0100.                                     |
//|                                                               |
//| The Food and Drug Administration classifies this software as  |
//| a medical device.  As such, it may not be changed in any way. |
//| Modifications to this software may result in an adulterated   |
//| medical device under 21CFR820, the use of which is considered |
//| to be a violation of US Federal Statutes.                     |
//+---------------------------------------------------------------+
 */
package gov.va.med.imaging.dicom.scp.qrfactory.factory;

import gov.va.med.imaging.dicom.common.Constants;
import gov.va.med.imaging.dicom.common.interfaces.IDicomDataSet;
import gov.va.med.imaging.dicom.scp.exceptions.QRFactoryTranslationException;
import gov.va.med.imaging.dicom.scp.exceptions.UnsupportedLevelException;
import gov.va.med.imaging.dicom.scp.qrfactory.concreteproducts.QueryRootPatient;
import gov.va.med.imaging.dicom.scp.qrfactory.concreteproducts.QueryRootPatientStudyOnly;
import gov.va.med.imaging.dicom.scp.qrfactory.concreteproducts.QueryRootStudy;
import gov.va.med.imaging.exchange.business.dicom.DicomRequestParameters;

import org.apache.log4j.Logger;

/**
 *
 * This class represents the Factory of the MethodFactory pattern.  This class
 * determines which specialized (concrete) method to used based on the requested 
 * SOP Class Presentation Context.  This Factory currently handles both C-Find-Rq 
 * and C-Move-Rq Dimse messages.  There are three Presentation Contexts for each 
 * Dimse message. It creates the appropriate Hashmap in each case by using the 
 * correct specialized method.
 *
 * @author William Peterson
 *
 */
public class QueryRootFactory {


    public static final int INT_PATIENTROOT_FIND = 200;
    public static final int INT_STUDYROOT_FIND = 201;
    public static final int INT_PATIENTSTUDYONLYROOT_FIND = 202;
    public static final int INT_PATIENTROOT_MOVE = 203;
    public static final int INT_STUDYROOT_MOVE = 204;
    public static final int INT_PATIENTSTUDYONLYROOT_MOVE = 205;

    private static final Logger logger = Logger.getLogger (QueryRootFactory.class);

  
  
    /**
     * Constructor
     */
    public QueryRootFactory() {
        super();
        //
    }
    
    /**
     * Create Hashmap object containing the C-Find-Rq parameters.  This method 
     * determines which Query Root to execute and passes the method parameters 
     * to the specialized method based on the Query Root value received.
     *   
     * @param dds represents the DicomDataSet
     * @param serviceMappingSet represents the MappingSet to build the responses later.
     * @return
     * @throws QRFactoryTranslationException
     * @throws UnsupportedLevelException
     */
    public DicomRequestParameters createQueryRequestParameters(IDicomDataSet dds)
                    throws QRFactoryTranslationException, UnsupportedLevelException{
        //Convert String to int values to represent Root for switch.
        int iQueryRoot = 0;
        if(dds.getAffectedSOPClass().equals(Constants.PATIENTROOT_FIND)) 
            iQueryRoot = INT_PATIENTROOT_FIND;
        if(dds.getAffectedSOPClass().equals(Constants.STUDYROOT_FIND)) 
            iQueryRoot = INT_STUDYROOT_FIND;
        if(dds.getAffectedSOPClass().equals(Constants.PATIENTSTUDYONLYROOT_FIND)) 
            iQueryRoot = INT_PATIENTSTUDYONLYROOT_FIND;
        if(iQueryRoot == 0){
            throw new UnsupportedLevelException();
        }
        
        //read queryRoot and determine proper concrete product
        //Call proper Concrete product
        switch (iQueryRoot){
            case INT_PATIENTROOT_FIND:
                return new QueryRootPatient().createQueryParameters(dds);
            case INT_STUDYROOT_FIND:
                return new QueryRootStudy().createQueryParameters(dds);
            case INT_PATIENTSTUDYONLYROOT_FIND:
                return new QueryRootPatientStudyOnly().createQueryParameters(dds);
            default:
                throw new UnsupportedLevelException();
            }
        
        }

    
    /**
     * Create Hashmap object containing the C-Move-Rq parameters.  This method 
     * determines which Query Root to execute and passes the method parameters 
     * to the specialized method based on the Query Root value received.
    *  
     * @param dds represents the DicomDataSet containing the C-Move Request.
     * @return represents the HashMap object containing the Request.
     * @throws QRFactoryTranslationException
     * @throws UnsupportedLevelException
     */
    public DicomRequestParameters createMoveRequestParameters(IDicomDataSet dds)
            throws QRFactoryTranslationException,
            UnsupportedLevelException{
        
    	
        int iMoveRoot = 0;
        if(dds.getAffectedSOPClass().equals(Constants.PATIENTROOT_MOVE)) 
            iMoveRoot = INT_PATIENTROOT_MOVE;
        if(dds.getAffectedSOPClass().equals(Constants.STUDYROOT_MOVE)) 
            iMoveRoot = INT_STUDYROOT_MOVE;
        if(dds.getAffectedSOPClass().equals(Constants.PATIENTSTUDYONLYROOT_MOVE)) 
            iMoveRoot = INT_PATIENTSTUDYONLYROOT_MOVE;
        if(iMoveRoot == 0){
            throw new UnsupportedLevelException("Unsupported Move Level Exception.");
        }
        
        switch (iMoveRoot){
            case INT_PATIENTROOT_MOVE:
                return new QueryRootPatient().createMoveParameters(dds);
            case INT_STUDYROOT_MOVE:
                return new QueryRootStudy().createMoveParameters(dds);
            case INT_PATIENTSTUDYONLYROOT_MOVE:
                return new QueryRootPatientStudyOnly().createMoveParameters(dds);
            default:
                throw new UnsupportedLevelException("Unsupported Move Level Exception.");
        }

    }

}
    
