/*
 * Created on Mar 15, 2005
//+---------------------------------------------------------------+
//| Property of the US Government.                                |
//| No permission to copy or redistribute this software is given. |
//| Use of unreleased versions of this software requires the user |
//| to execute a written test agreement with the VistA Imaging    |
//| Development Office of the Department of Veterans Affairs,     |
//| telephone (301) 734-0100.                                     |
//|                                                               |
//| The Food and Drug Administration classifies this software as  |
//| a medical device.  As such, it may not be changed in any way. |
//| Modifications to this software may result in an adulterated   |
//| medical device under 21CFR820, the use of which is considered |
//| to be a violation of US Federal Statutes.                     |
//+---------------------------------------------------------------+
 */
package gov.va.med.imaging.dicom.scp;

import gov.va.med.imaging.GUID;

import java.util.HashMap;
/**
 * Handles relationship between incoming InstanceUIDs from the DicomDataset Object 
 * and their respective GUIDs.
 * 
 * 
 * @author William Peterson
 *
 */
public class InstanceList {
    
    /**
     * Add the Study InstanceUID and its correlating GUID to the Map.  This allows the 
     * ability to retrieve the correlating GUID through other method calls by only knowing
     * the Study InstanceUID.
     * 
     * @param studyuid represents Study Instance UID.
     * @param newguid represents GUID to add to the Map.
     */
    public void addStudyGuid(String studyuidkey, GUID nuguid){
        
        //check if studyuidkey is already in hashmap.
        //this.isStudyInstanceUIDListed(studyuidkey);
        
        //add study uid and guid to study hashmap.
        this.studyGuidMap.put(studyuidkey, nuguid);
        
    }
    
    /**
     * Add the Series InstanceUID and its correlating GUID to the Map.  This allows the 
     * ability to retrieve the correlating GUID through other method calls by only knowing
     * the Series Instance UID.
     * 
     * @param seriesuid represents Series Instance UID.
     * @param newguid represents GUID to add to the Map.
     */
    public void addSeriesGuid(String seriesuidkey, GUID nuguid){
 
        //check if seriesuidkey is already in hashmap.
        //this.isSeriesInstanceUIDListed(seriesuidkey);
        
        //add series uid and guid to series hashmap.
        this.seriesGuidMap.put(seriesuidkey, nuguid);
        
    }
    
    /**
     * Find and return the correlating GUID for the Study InstanceUID given.
     * 
     * @param studyuid represents Study Instance UID.
     * @return GUID.  If null, Map does not contain GUID.
     */
    public GUID getStudyGUID(String studyuidkey){
        GUID guid = new GUID();
        
        //search for matching key to get guid.
        guid = (GUID)this.studyGuidMap.get(studyuidkey);
        
        //return study guid.
        return guid;
    }
    
    /**
     * Find and return the correlating GUID for the Series InstanceUID given.
     * 
     * @param seriesuid represents Series Instance UID.
     * @return GUID.  If null, Map does not contain GUID.
     */
    public GUID getSeriesGUID(String seriesuidkey){
        GUID guid = new GUID();
        
        //search for matching key to get guid.
        guid = (GUID)this.seriesGuidMap.get(seriesuidkey);
        
        //return series guid.
        return guid;
    }
    
    /**
     * Determine if the Study Instance UID is in the Map.
     * 
     * @param studyuid represents Study Instance UID.
     * @return boolean
     */
    public boolean isStudyInstanceUIDListed(String studyuidkey){
        boolean result = false;
        //find matching study uid key.
        //if match is found, set return to true.
        if ((this.studyGuidMap.get(studyuidkey)) != null){
            result = true;
        }
            return result;
    }
    
    /**
     * Determine if the Series Instance UID is in the Map.
     * 
     * @param seriesuid represents the Series Instance UID.
     * @return boolean
     */
    public boolean isSeriesInstanceUIDListed(String seriesuidkey){
        boolean result = false;
        //find matching series uid key
        //if match is found, set return to true.
        if ((this.seriesGuidMap.get(seriesuidkey)) != null){
            result = true;
        }
        return result;
    }
    
    private HashMap<String, GUID> studyGuidMap = new HashMap<String, GUID>();
    private HashMap<String, GUID> seriesGuidMap = new HashMap<String, GUID>();
    
}
