/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.memory;

import gov.va.med.imaging.storage.cache.*;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.CacheInitializationException;
import gov.va.med.imaging.storage.cache.exceptions.UnknownEvictionStrategyException;
import gov.va.med.imaging.storage.cache.memento.RegionMemento;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * @author        DNS
 *
 */
public class MemoryRegion 
implements Region
{
	private final String name;
	private final Cache parentCache; 	
	private final GroupFactory groupFactory;
	private final InstanceFactory instanceFactory;
	private final GroupSet groups;
	private boolean initialized;
	private EvictionStrategy[] evictionStrategies;

	/**
	 * @param name
	 * @param evictionStrategyName
	 */
	MemoryRegion(Cache parentCache, String name, EvictionStrategy[] evictionStrategies, GroupFactory groupFactory, InstanceFactory instanceFactory)
	{
		this.parentCache = parentCache;
		this.name = name;
		setEvictionStrategies( evictionStrategies );
		this.groupFactory = groupFactory;
		this.instanceFactory = instanceFactory;
		groups = new GroupSet(getGroupFactory());
	}
	
	/**
	 * @return the parentCache
	 */
	public Cache getParentCache()
	{
		return this.parentCache;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#getName()
	 */
	public String getName()
	{
		return this.name;
	}

	/**
	 * @see gov.va.med.imaging.storage.cache.impl.PersistentRegionMBean#getEvictionStrategyNames()
	 */
	public String[] getEvictionStrategyNames()
	{
		if(evictionStrategies == null)
			return null;
		
		String[] evictionStrategyNames = new String[evictionStrategies.length];
		int evictionStrategyIndex = 0;
		
		for(EvictionStrategy evictionStrategy : evictionStrategies)
			evictionStrategyNames[evictionStrategyIndex++] = evictionStrategy.getName();
		
		return evictionStrategyNames;
	}
	
	/**
	 * @see gov.va.med.imaging.storage.cache.impl.PersistentRegionMBean#setEvictionStrategyNames(java.lang.String[])
	 */
	public void setEvictionStrategyNames(String[] evictionStrategyNames)
	throws UnknownEvictionStrategyException
	{
		if(evictionStrategyNames != null)
		{
			EvictionStrategy[] evictionStrategies = new EvictionStrategy[evictionStrategyNames.length];
			
			int evictionStrategyIndex = 0;
			for(String evictionStrategyName : evictionStrategyNames)
			{
				evictionStrategies[evictionStrategyIndex++] = getParentCache().getEvictionStrategy(evictionStrategyName);
			
				if(evictionStrategies[evictionStrategyIndex-1] == null)
					throw new UnknownEvictionStrategyException("EvictionStrategy '" + evictionStrategyNames + "' is not known to this cache instance.");
			}
			
			setEvictionStrategies( evictionStrategies );
		}
	}
	
	/**
	 * Use the String based eviction strategy name methods externally to this class.
	 * 
	 * @param evictionStrategy
	 */
	private void setEvictionStrategies(EvictionStrategy[] newEvictionStrategies)
	{
		// remove the old eviction strategies if there are any
		if(this.evictionStrategies != null)
		{
			for(EvictionStrategy evictionStrategy : evictionStrategies)
				evictionStrategy.removeRegion(this);
		}
		
		this.evictionStrategies = newEvictionStrategies;
		
		if(newEvictionStrategies != null && newEvictionStrategies.length != 0)
		{
			for(EvictionStrategy evictionStrategy : evictionStrategies)
				evictionStrategy.addRegion(this);
		}
	}

	public EvictionStrategy[] getEvictionStrategies()
	{
		return this.evictionStrategies;
	}
	
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#isInitialized()
	 */
	public Boolean isInitialized()
	{
		return initialized;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#setInitialized(java.lang.Boolean)
	 */
	public void setInitialized(Boolean initialized) 
	throws CacheException
	{
		this.initialized = initialized;
	}

	/**
	 * @return the groupFactory
	 */
	public GroupFactory getGroupFactory()
	{
		return this.groupFactory;
	}

	/**
	 * @return the instanceFactory
	 */
	public InstanceFactory getInstanceFactory()
	{
		return this.instanceFactory;
	}

	/**
	 * @see gov.va.med.imaging.storage.cache.Region#cacheLifecycleEvent(gov.va.med.imaging.storage.cache.CacheLifecycleEvent)
	 */
	public void cacheLifecycleEvent(CacheLifecycleEvent event)
	{

	}

	// ========================================================================================================================
	// Group Management
	// ========================================================================================================================
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#clear()
	 */
	public void clear() 
	throws CacheException
	{
		groups.clear();
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#getGroup(java.lang.String[])
	 */
	public Group getGroup(String[] groupPath) 
	throws CacheException
	{
		return groups.getByName(groupPath);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#getGroups()
	 */
	public Iterator<? extends Group> getGroups() 
	throws CacheException
	{
		return groups.iterator();
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#getOrCreateGroup(java.lang.String[])
	 */
	public Group getOrCreateGroup(String[] groupPath) 
	throws CacheException
	{
		return groups.getOrCreateByName(groupPath);
	}

	public void removeChildGroup(Group childGroup) 
	throws CacheException
	{
		if( groups.remove(childGroup) )
			childGroup.remove();
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#evaluateAndEvictChildGroups(gov.va.med.imaging.storage.cache.EvictionJudge)
	 */
	public int evaluateAndEvictChildGroups(EvictionJudge<Group> judge) 
	throws CacheException
	{
		List<Group> killList = new ArrayList<Group>();
		int evictedCount = 0;

		// kill the lowest progeny first
		for(Group group : groups)
			evictedCount += group.evaluateAndEvictChildGroups(judge);
		
		for(Group group : groups)
			if(judge.isEvictable(group))
				killList.add(group);
		
		for(Group group : killList)
		{
			removeChildGroup(group);
			++evictedCount;
		}
		
		return evictedCount;
	}

	// ========================================================================================================================
	// Instance Management
	// ========================================================================================================================
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#getInstance(java.lang.String[], java.lang.String)
	 */
	public Instance getInstance(String[] path, String key) 
	throws CacheException
	{
		Group group = getGroup(path);
		return group == null ? null : group.getChildInstance(key);
	}

	/**
	 * 
	 */
	public void deleteInstance(String[] path, String key) 
	throws CacheException
	{
		Group group = getGroup(path);
		if(group != null)
			group.deleteChildInstance(key);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#getOrCreateInstance(java.lang.String[], java.lang.String)
	 */
	public Instance getOrCreateInstance(String[] path, String key) 
	throws CacheException
	{
		return getOrCreateGroup(path).getOrCreateChildInstance(key);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#getFreeSpace()
	 */
	public long getFreeSpace()
	{
		Cache parent = getParentCache();
		if(parent instanceof MemoryCache)
			return ((MemoryCache)parent).getFreeSpace();
		else
			return Long.MAX_VALUE;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Region#getTotalSpace()
	 */
	public long getTotalSpace()
	{
		Cache parent = getParentCache();
		if(parent instanceof MemoryCache)
			return ((MemoryCache)parent).getFreeSpace();
		else
			return Long.MAX_VALUE;
	}

	@Override
	public long getUsedSpace() 
	{
		long usedSpace = 0;
		
		try 
		{
			for(Iterator<? extends Group> groupIter = getGroups(); groupIter.hasNext(); )
			{
				Group group = groupIter.next();
				usedSpace += group.getSize();
			}
		} 
		catch (CacheException e) 
		{
			e.printStackTrace();
		}
		
		return usedSpace;
	}

	public RegionMemento createMemento()
	{
		RegionMemento memento = new RegionMemento();
		
		memento.setName(getName());
		memento.setEvictionStrategyNames(getEvictionStrategyNames());
		
		return memento;
	}

	public void setParameters(RegionMemento memento) 
	throws CacheInitializationException
	{
		this.setEvictionStrategyNames( memento.getEvictionStrategyNames() );
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.MutableNamedObject#remove()
	 */
	public void remove() 
	throws CacheException
	{

	}
	
}
