/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.memory;

import gov.va.med.imaging.channels.ChecksumValue;
import gov.va.med.imaging.storage.cache.*;
import gov.va.med.imaging.storage.cache.events.InstanceLifecycleEvent;
import gov.va.med.imaging.storage.cache.events.InstanceLifecycleListener;
import gov.va.med.imaging.storage.cache.events.LifecycleEvent;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.InstanceInvalidStateException;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author        DNS
 *
 */
public class MemoryInstance 
implements Instance, InstanceByteChannelListener
{
	private final Integer id = new Integer( (new Object()).hashCode() );		// a unique ID within the process
	private String name;
	private Date lastAccessed = new Date();
	private ChecksumValue checksumValue;
	private boolean removed = false;
	
	private MemoryInstanceByteChannelFactory byteChannelFactory;
	
	public MemoryInstance(MemoryInstanceByteChannelFactory byteChannelFactory, String name)
	{
		super();
		this.byteChannelFactory = byteChannelFactory;
		this.name = name;
	}

	public String getName()
	{
		return name;
	}

	/**
	 * @return the byteChannelFactory
	 */
	public MemoryInstanceByteChannelFactory getByteChannelFactory()
	{
		return this.byteChannelFactory;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Instance#getChecksumValue()
	 */
	public String getChecksumValue()
	{
		return checksumValue == null ? null : checksumValue.toString();
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Instance#getLastAccessed()
	 */
	public Date getLastAccessed() 
	throws CacheException
	{
		if(isRemoved())
			throw new InstanceInvalidStateException("Instance has been removed from the cache, access date is not accessible");
		
		return lastAccessed;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Instance#getSize()
	 */
	public long getSize() 
	throws CacheException
	{
		if(isRemoved())
			throw new InstanceInvalidStateException("Instance has been removed from the cache, size is not accessible");
		
		return 0;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Instance#getWritableChannel()
	 */
	public InstanceWritableByteChannel getWritableChannel() 
	throws CacheException
	{
		return getWritableChannel(true);
	}
	public InstanceWritableByteChannel getWritableChannelNoWait() 
	throws CacheException
	{
		return getWritableChannel(false);
	}
	private InstanceWritableByteChannel getWritableChannel(boolean wait) 
	throws CacheException
	{
		if(isRemoved())
			throw new InstanceInvalidStateException("Instance has been removed from the cache, channels are not accessible");
		
		lastAccessed = new Date();
		return byteChannelFactory.getInstanceWritableByteChannel(id, this);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Instance#getReadableChannel()
	 */
	public InstanceReadableByteChannel getReadableChannel() 
	throws CacheException
	{
		return getReadableChannel(true);
	}
	public InstanceReadableByteChannel getReadableChannelNoWait() 
	throws CacheException
	{
		return getReadableChannel(false);
	}
	private InstanceReadableByteChannel getReadableChannel(boolean wait) 
	throws CacheException
	{
		if(isRemoved())
			throw new InstanceInvalidStateException("Instance has been removed from the cache, channels are not accessible");
		
		lastAccessed = new Date();
		return byteChannelFactory.getInstanceReadableByteChannel(id, this);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Instance#isPersistent()
	 */
	public boolean isPersistent() 
	throws CacheException
	{
		return false;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Instance#remove()
	 */
	public void remove() 
	throws CacheException
	{
		getByteChannelFactory().remove(this.id);
		notifyListeners(LifecycleEvent.DELETE);
	}

	private boolean isRemoved()
	{
		return removed;
	}
	
	// ==============================================================================================================
	// 
	// ==============================================================================================================
	public void readChannelClose(InstanceReadableByteChannel readable)
	{
	}
	public void readChannelIdleTimeout(InstanceReadableByteChannel readable)
	{
	}
	public void writeChannelClose(InstanceWritableByteChannel writable)
	{
		checksumValue = new ChecksumValue(writable.getChecksum());
	}
	public void writeChannelIdleTimeout(InstanceWritableByteChannel writable)
	{
	}
	
	// ==============================================================================================================
	// 
	// ==============================================================================================================
	
	private List<InstanceLifecycleListener> listeners = new ArrayList<InstanceLifecycleListener>();
	public void registerListener(InstanceLifecycleListener listener)
	{
		listeners.add(listener);
	}
	
	public void unregisterListener(InstanceLifecycleListener listener)
	{
		listeners.remove(listener);
	}
	
	protected void notifyListeners(LifecycleEvent event)
	{
		InstanceLifecycleEvent lifecycleEvent = new InstanceLifecycleEvent(event, getName());
		for(InstanceLifecycleListener listener : listeners)
			listener.notify(lifecycleEvent);
	}

}
