/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.memory;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import gov.va.med.imaging.storage.cache.*;
import gov.va.med.imaging.storage.cache.events.GroupLifecycleEvent;
import gov.va.med.imaging.storage.cache.events.GroupLifecycleListener;
import gov.va.med.imaging.storage.cache.events.LifecycleEvent;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;

/**
 * @author       DNS
 *
 */
public class MemoryGroup 
implements Group
{
	private final String name;
	private final GroupFactory groupFactory;
	private final InstanceFactory instanceFactory;
	private final GroupSet groups;
	private final InstanceSet instanceSet;
	private Date lastAccessed = new Date();

	/**
	 * 
	 * @param name
	 * @param groupFactory
	 * @param instanceFactory
	 */
	public MemoryGroup(String name, GroupFactory groupFactory, InstanceFactory instanceFactory)
	{
		super();
		this.name = name;
		this.groupFactory = groupFactory;
		this.instanceFactory = instanceFactory;
		this.groups = new GroupSet(getGroupFactory());
		this.instanceSet = new InstanceSet(getInstanceFactory());
	}

	public String getName()
	{
		return name;
	}

	/**
	 * @return the groupFactory
	 */
	public GroupFactory getGroupFactory()
	{
		return this.groupFactory;
	}

	/**
	 * @return the instanceFactory
	 */
	public InstanceFactory getInstanceFactory()
	{
		return this.instanceFactory;
	}

	public Date getLastAccessed() 
	throws CacheException
	{
		return lastAccessed;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Group#getSize()
	 */
	public long getSize() 
	throws CacheException
	{
		return 0;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Group#remove()
	 */
	public void remove() 
	throws CacheException
	{
		removeAllChildGroups();
		removeAllChildInstances();
		notifyListeners(LifecycleEvent.DELETE);
	}
	
	// ================================================================================================================
	// Group Management
	// ================================================================================================================
	public Group getChildGroup(String groupName) 
	throws CacheException
	{
		return groups.getByName(groupName);
	}

	public Group getGroup(String[] groupPath) 
	throws CacheException
	{
		return groups.getByName(groupPath);
	}

	public Iterator<? extends Group> getGroups() 
	throws CacheException
	{
		return groups.iterator();
	}

	public Group getOrCreateGroup(String[] groupPath) 
	throws CacheException
	{
		return groups.getOrCreateByName(groupPath);
	}

	public Group getOrCreateChildGroup(String groupName) 
	throws CacheException
	{
		return groups.getOrCreateByName(groupName);
	}

	public void removeAllChildGroups() 
	throws CacheException
	{
		for(Group group : groups)
			group.remove();
		
		while(groups.size() > 0)
			groups.clear();
	}

	public void removeChildGroup(Group childGroup) 
	throws CacheException
	{
		if( groups.remove(childGroup) )
			childGroup.remove();
	}

	public int evaluateAndEvictChildGroups(EvictionJudge<Group> judge) 
	throws CacheException
	{
		List<Group> killList = new ArrayList<Group>();
		int evictedCount = 0;

		// kill the lowest progeny first
		for(Group group : groups)
			evictedCount += group.evaluateAndEvictChildGroups(judge);
		
		for(Group group : groups)
			if(judge.isEvictable(group))
				killList.add(group);
		
		for(Group group : killList)
		{
			removeChildGroup(group);
			++evictedCount;
		}
		
		return evictedCount;
	}

	// ================================================================================================================
	// Instance Management
	// ================================================================================================================
	public Instance getChildInstance(String key) 
	throws CacheException
	{
		return instanceSet.getByName(key);
	}

	public void deleteChildInstance(String key) 
	throws CacheException
	{
		Instance childInstance = instanceSet.getByName(key);
		if(childInstance != null)
			this.removeChildInstance(childInstance);
	}
	
	public Instance getInstance(String[] path, String key) 
	throws CacheException
	{
		return getGroup(path).getChildInstance(key);
	}

	public void deleteInstance(String[] path, String key) 
	throws CacheException
	{
		getGroup(path).deleteChildInstance(key);
	}

	public Iterator<? extends Instance> getInstances() 
	throws CacheException
	{
		return instanceSet.iterator();
	}

	public Instance getOrCreateChildInstance(String key) 
	throws CacheException
	{
		return instanceSet.getOrCreateByName(key);
	}

	public Instance getOrCreateInstance(String[] path, String key) 
	throws CacheException
	{
		return getOrCreateGroup(path).getOrCreateChildInstance(key);
	}

	public void removeAllChildInstances() 
	throws CacheException
	{
		synchronized(instanceSet)
		{
			for(Instance instance : instanceSet)
				instance.remove();
			instanceSet.clear();
		}
	}

	public void removeChildInstance(Instance childInstance) 
	throws CacheException
	{
		instanceSet.remove(childInstance);
		childInstance.remove();
	}

	// ======================================================================================================
	// Listener Management
	// ======================================================================================================
	private List<GroupLifecycleListener> listeners = new ArrayList<GroupLifecycleListener>();
	public void registerListener(GroupLifecycleListener listener)
	{
		listeners.add(listener);
	}
	
	public void unregisterListener(GroupLifecycleListener listener)
	{
		listeners.remove(listener);
	}
	
	protected void notifyListeners(LifecycleEvent event)
	{
		GroupLifecycleEvent lifecycleEvent = new GroupLifecycleEvent(event, getName());
		for(GroupLifecycleListener listener : listeners)
			listener.notify(lifecycleEvent);
	}
	
}
