package gov.va.med.imaging.storage.cache.impl.jcifs;

import gov.va.med.imaging.storage.cache.Instance;
import gov.va.med.imaging.storage.cache.InstanceByteChannelFactory;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.CacheInternalException;
import gov.va.med.imaging.storage.cache.exceptions.PersistenceIOException;
import gov.va.med.imaging.storage.cache.impl.PersistentInstanceSet;

import java.lang.ref.SoftReference;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.Iterator;

import jcifs.smb.SmbException;
import jcifs.smb.SmbFile;
import jcifs.smb.SmbFileFilter;

import org.apache.log4j.Logger;

/**
 * This class encapsulates the collection of child instances in a JcifsCacheGroup.
 * It is the responsibility of this class to ensure that the persistent (filesystem)
 * and the transient (memory) views of the Instances in a group are consistent.
 * It is a requirement that this class NOT keep references to child instances that 
 * would prevent garbage collection of instances that are no longer referenced outside
 * the cache (i.e. if the application does not have a reference then the cache should not
 * prevent garbage collection).
 * This class should be the sole modifier of the instances referenced within.
 * 
 * @author        DNS
 *
 */
class JcifsInstanceSet
extends PersistentInstanceSet
{
	private Logger logger = Logger.getLogger(this.getClass());
	private static final long serialVersionUID = -346790233288183129L;
	private SmbFile rootDirectory = null;			// the directory in which all of our instances reside in persistent storage
	private SmbUtilities smbUtilities = new SmbUtilities();
	
	/**
	 * 
	 * @param rootDirectory
	 * @param byteChannelFactory
	 * @param secondsReadWaitsForWriteCompletion
	 * @param setModificationTimeOnRead
	 */
	JcifsInstanceSet(
		SmbFile rootDirectory, 
		InstanceByteChannelFactory byteChannelFactory,
		int secondsReadWaitsForWriteCompletion,
		boolean setModificationTimeOnRead
	)
	{
		super(byteChannelFactory, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
		this.rootDirectory = rootDirectory;
	}
	
	/**
	 * 
	 * @return
	 */
	public SmbFile getRootDirectory()
	{
		return rootDirectory;
	}
	
	/**
	 * 
	 */
	@Override
	protected Instance getOrCreate(String name, boolean create) 
	throws CacheException
	{
		SmbFile childInstanceFile;
		try
		{
			childInstanceFile = new SmbFile(getRootDirectory().getCanonicalPath() + "/" + name);
			
			Instance child = create ? 
					JcifsInstance.getOrCreateInstance(childInstanceFile, getByteChannelFactory(), getSecondsReadWaitsForWriteCompletion(), isSetModificationTimeOnRead()) :
					JcifsInstance.getInstance(childInstanceFile, getByteChannelFactory(), getSecondsReadWaitsForWriteCompletion(), isSetModificationTimeOnRead());
				
			return child;
		} 
		catch (MalformedURLException x)
		{
			throw new CacheInternalException(
				"The SMB URL built from directory '" + getRootDirectory().getPath() + "' and file name '" + name + "' is invalid.", 
				x);
		} 
		//catch (UnknownHostException x)
		//{
		//	throw new PersistenceIOException("The SMB host computer (file server) is inaccesible.", x);
		//}
		
	}
	
	/**
	 * Assure that the internal represenation of child instances matches
	 * what is in the file system. 
	 */
	@Override
	protected void internalSynchronizeChildren()
	throws CacheException
	{
		String instanceName = null;		// declared out here so tha the error logs can report it
		
		try
		{
			// prune unused references
			for( Iterator<SoftReference<? extends Instance>> iter=iterator();
				iter.hasNext(); )
			{
				SoftReference<? extends Instance> instanceRef = iter.next();
				if( instanceRef.get() == null )
					iter.remove();
			}

			// get a list of all the child files
			for( SmbFile childFile : smbUtilities.getPersistentChildren(getRootDirectory(), false, true) )
			{
				instanceName = childFile.getName();		// the file name and the Instance name are the same
				Instance childInstance = getTransient(instanceName);		// get an existing reference by name
				if(childInstance == null)
				{
					childInstance = getChild(instanceName, false);
					SmbFile instanceFile = new SmbFile(this.getRootDirectory(), instanceName);
					childInstance =  JcifsInstance.getInstance(instanceFile, getByteChannelFactory(), getSecondsReadWaitsForWriteCompletion(), isSetModificationTimeOnRead());
					SoftReference<JcifsInstance> instanceRef = new SoftReference<JcifsInstance>( (JcifsInstance)childInstance);
					add( instanceRef );
				}
			}
		} 
		catch (MalformedURLException x)
		{
			throw new CacheInternalException(
					"The SMB URL built from directory '" + getRootDirectory().getPath() + "' and file name '" + instanceName + "' is invalid.", 
					x);
		} 
		catch (UnknownHostException x)
		{
			throw new PersistenceIOException(
					"The host specified in the SMB URL '" + getRootDirectory().getPath() + "' cannot be contacted.", 
					x);
		}
	}
}
