/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.jcifs;

import gov.va.med.imaging.storage.cache.*;
import gov.va.med.imaging.storage.cache.exceptions.*;
import gov.va.med.imaging.storage.cache.impl.AbstractCacheImpl;
import gov.va.med.imaging.storage.cache.impl.eviction.EvictionStrategyFactory;
import gov.va.med.imaging.storage.cache.impl.eviction.LastAccessedEvictionStrategy;
import gov.va.med.imaging.storage.cache.impl.eviction.LastAccessedEvictionStrategyMemento;
import gov.va.med.imaging.storage.cache.impl.jcifs.memento.JcifsCacheMemento;
import gov.va.med.imaging.storage.cache.impl.memento.PersistentRegionMemento;
import gov.va.med.imaging.storage.cache.memento.CacheMemento;
import gov.va.med.imaging.storage.cache.memento.EvictionStrategyMemento;
import gov.va.med.imaging.storage.cache.memento.RegionMemento;
import gov.va.med.imaging.storage.cache.timer.EvictionTimerImpl;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import jcifs.smb.SmbException;
import jcifs.smb.SmbFile;

import org.apache.log4j.Logger;

/**
 * @author        DNS
 * 
 */
public class JcifsCache 
extends AbstractCacheImpl 
implements Cache
{
	public final static String protocol = "smb";
	private Logger logger = Logger.getLogger(this.getClass());
	private URL smbLocationUrl;		// the root location as a valid SMB formatted URL
	private SmbFile rootDirectory;
	private String host;
	private int port;
	private String userId;
	private String userPwd;
	private String file;

	static
	{
		// register the SMB URL handler
		jcifs.Config.registerSmbURLHandler();
		
		Logger.getLogger(JcifsCache.class).info("Registered protocol handlers are '" + System.getProperty("java.protocol.handler.pkgs") + "'.");
	}
	
	// ===========================================================================================================
	// Create (factory) methods
	// ===========================================================================================================
	/**
	 * 
	 * @param name
	 * @param locationUri
	 * @return
	 * @throws CacheException
	 */
	public static JcifsCache create(String name, URI locationUri) 
	throws CacheException
	{
		JcifsCacheConfigurator configurator = new JcifsCacheConfigurator();
		EvictionTimerImpl evictionTimer = EvictionTimerImpl.create(configurator.getEvictionTimerSweepIntervalMap());

		return new JcifsCache(name, locationUri, evictionTimer, JcifsByteChannelFactory.create());
	}

	/**
	 * 
	 * @param name
	 * @param locationUri
	 * @param evictionTimer
	 * @return
	 * @throws CacheException
	 */
	public static JcifsCache create(String name, URI locationUri, EvictionTimer evictionTimer) 
	throws CacheException
	{
		return new JcifsCache(name, locationUri, evictionTimer, JcifsByteChannelFactory.create());
	}
	
	/**
	 * 
	 * @param memento
	 * @return
	 * @throws CacheException
	 */
	public static JcifsCache create(CacheMemento memento) 
	throws CacheException
	{
		return new JcifsCache(memento);
	}

	// ===========================================================================================================
	// Constructors, note that they are all private, use the factory methods to create instances.
	// ===========================================================================================================
	private JcifsCache(String name, URI locationUri, EvictionTimer evictionTimer, InstanceByteChannelFactory<SmbFile> byteChannelFactory)
	throws CacheException
	{
		super(name, locationUri, evictionTimer, byteChannelFactory);
	}

	private JcifsCache(CacheMemento memento) 
	throws CacheException
	{
		super(memento);

		if (memento instanceof JcifsCacheMemento)
			restoreFromMemento((JcifsCacheMemento) memento);
	}

	@Override
	protected void parseLocationUri(URI locationUri) 
	throws CacheInitializationException
	{
		// The primary reason we parse the locationUri is to have some assurance
		// that it is valid and log some messages as to where the cache lives.
		// We should not log the entire locationUri as it may include
		// the user ID and password.
		try
		{
			smbLocationUrl = locationUri.toURL();
			if(smbLocationUrl.getUserInfo() != null)
			{
				String[] userInfo = smbLocationUrl.getUserInfo().split(":");
				userId  = userInfo.length > 0 ? userInfo[0] : null;
				userPwd  = userInfo.length > 1 ? userInfo[1] : null;
			}
			
			host = smbLocationUrl.getHost();
			port = smbLocationUrl.getPort();
			file = smbLocationUrl.getFile();

			if(host == null)
				throw new CacheInitializationException(
					"The given location URI '" + locationUri.toString() + "' does not include a host name." 
				);
			
			if(file == null)
				throw new CacheInitializationException(
					"The given location URI '" + locationUri.toString() + "' does not include a file(share)." 
				);
			
			Logger.getLogger(JcifsCache.class).info("SMB Location URL parsed as follows: userId='" + userId + 
					"', userPwd is " + (userPwd == null ? "null" : "not null") + 
					" host='" + host + 
					"' port='" + port +
					"' file='" + file + "'");
			
			if( file.lastIndexOf('/') != file.length()-1 )
				throw new CacheInitializationException(
					"The given location URI '" + locationUri.toString() + "' does not end with a share name and it must.  The URI must end with a alash character '/' to be valid." 
				);
			
		} 
		catch (MalformedURLException x)
		{
			logger.error(x);
			throw new CacheInitializationException(
					"Unable to parse the location URI '" + locationUri.toString() + "' into a valid SMB URL.\n" + 
					"The correct syntax is 'smb://[[[domain;]username[:password]@]server[:port]/[[share/[dir/]file]]][?[param=value[param2=value2[...]]]'\n" + 
					"A complete description of the format may be found at http://jcifs.samba.org/src/docs/api/jcifs/smb/SmbFile.html \n",
					x);
		}
	}

	public URL getSmbLocationUrl()
	{
		return this.smbLocationUrl;
	}
	
	public SmbFile getRootDirectory() 
	throws CacheStateException
	{
		// Check whether the rootDirectory has been set rather than whether the cache has been initialized
		// because, internally, this method must return the root directory before initialization is complete.
		// Externally, isInitialized() and the existence of a non-null root directory are nearly synonomous.
		if( rootDirectory == null )
			throw new CacheStateException("JCIFS Cache must be initialized before the root directory is available.");
		
		return rootDirectory;
	}
	
	private void setRootDirectory(SmbFile rootDirectory)
	{
		this.rootDirectory = rootDirectory;
	}
	
	@Override
	public void internalInitialize() 
	throws InitializationException, CacheStateException
	{
		try
		{
			SmbFile root = new SmbFile(getSmbLocationUrl());
			//NtlmPasswordAuthentication ntPassAuth = new NtlmPasswordAuthentication(null, "jcifs", "Raptor999");
			//SmbFile root = new SmbFile("smb://jcifs:Raptor999@Isw-beckeyc/jcifs-cache/");
			
			root.connect();
			
			if( ! root.exists() )
				throw new CacheInitializationException("Cache root '" + getSmbLocationUrl().toExternalForm() + "' does not exist.");
			
			if( ! root.canRead() )
				throw new CacheInitializationException("Cache root '" + getSmbLocationUrl().toExternalForm() + "' exists but is not readable.");
			
			// the directory reports back as un-writable regardless of whether we can write to it
			//if( ! root.canWrite() )
			//	throw new CacheInitializationException("Cache root '" + getSmbLocationUrl().toExternalForm() + "' exists but is not writable.");

			if( ! root.isDirectory() )
				throw new CacheInitializationException("Cache root '" + getSmbLocationUrl().toExternalForm() + "' exists but is not a directory.");

			logger.info( "'" + getSmbLocationUrl().toExternalForm() + " has " + root.getDiskFreeSpace() + " bytes free." );
			
			setRootDirectory(root);		// if everything works then set the root directory
		} 
		catch (SmbException x)
		{
			logger.error(x);
			throw new CacheInitializationException("Unable to access the cache root '" + getSmbLocationUrl().toExternalForm() + "'.", x);
		}
		catch (IOException x)
		{
			logger.error(x);
			throw new CacheInitializationException("Unable to connect to root '" + getSmbLocationUrl().toExternalForm() + "'.", x);
		}
	}

	// ==================================================================================================================
	//
	// ==================================================================================================================

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.imaging.storage.cache.impl.AbstractCacheImpl#createRegion(java.lang.String)
	 */
	@Override
	public Region createRegion(String name, String[] evictionStrategyNames) 
	throws RegionInitializationException
	{
		JcifsRegion region = JcifsRegion.create(this, name, evictionStrategyNames, defaultSecondsReadWaitsForWriteCompletion, defaultSetModificationTimeOnRead);
		
		return region;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.imaging.storage.cache.impl.AbstractCacheImpl#createRegion(gov.va.med.imaging.storage.cache.memento.RegionMemento)
	 */
	@Override
	public Region createRegion(RegionMemento regionMemento) 
	throws RegionInitializationException
	{
		if(regionMemento instanceof PersistentRegionMemento)
			return JcifsRegion.create(this, (PersistentRegionMemento)regionMemento);

		throw new RegionInitializationException("Given region memento of type '" + regionMemento.getClass().getName() + "' cannot be used to construct a JCIFS Region");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.imaging.storage.cache.impl.AbstractCacheImpl#validateRegionType(gov.va.med.imaging.storage.cache.Region)
	 */
	@Override
	protected void validateRegionType(Region region) 
	throws IncompatibleRegionException
	{
		if(! (region instanceof JcifsRegion) )
			throw new IncompatibleRegionException("Region of type '" + region.getClass().getName() + "' is incompatible with cahe of type '"+ this.getClass().getName() + "'.");
	}

	// ==================================================================================================================
	//
	// ==================================================================================================================

	/**
	 * @see gov.va.med.imaging.storage.cache.impl.AbstractCacheImpl#createMemento()
	 */
	@Override
	public CacheMemento createMemento()
	{
		JcifsCacheMemento memento = new JcifsCacheMemento();

		memento.setEvictionTimerMemento(defaultEvictionTimer.createMemento());
		
		memento.setName(getName());
		memento.setLocationUri(getLocationUri().toString());
		memento.setEnabled(isEnabled());
		memento.setInitialized(isInitialized());
		
		if(getInstanceByteChannelFactory() instanceof JcifsByteChannelFactory)
			memento.setByteChannelFactoryMemento(  ((JcifsByteChannelFactory)getInstanceByteChannelFactory()).createMemento() );
		memento.setEvictionStrategyMementos(createEvictionStrategyMementos());
		memento.setRegionMementos(createRegionMementos());

		return memento;
	}

	protected List<LastAccessedEvictionStrategyMemento> createEvictionStrategyMementos()
	{
		List<LastAccessedEvictionStrategyMemento> evictionStrategyMementos = new ArrayList<LastAccessedEvictionStrategyMemento>();
		for(EvictionStrategy evictionStrategy:getEvictionStrategies())
		{
			if(evictionStrategy instanceof LastAccessedEvictionStrategy)
				evictionStrategyMementos.add( ((LastAccessedEvictionStrategy)evictionStrategy).createMemento() );
			else
				logger.error("An eviction strategy of type '" + evictionStrategy.getClass().getName() + "' was encountered.\n" + 
						"Configuration persistence of this class in unknown.");
		}
		return evictionStrategyMementos;
	}
	
	protected List<PersistentRegionMemento> createRegionMementos()
	{
		List<PersistentRegionMemento> regionMementos = new ArrayList<PersistentRegionMemento>();
		for(Region region:getRegions())
			regionMementos.add( (PersistentRegionMemento)region.createMemento() );
		
		return regionMementos;
	}
	
	/**
	 *The cache must be restored from a memento in the following order:
	 * 0.) the default eviction timer
	 * 1.) root directory name
	 * 2.) instance byte channel
	 * 3.) eviction strategies
	 * 4.) regions (needs instance byte channel and eviction strategies)
	 * 5.) initialized flag
	 * 6.) enabled flag
	 * 7.) the START signal may be acted upon, sending the START is the responsibility of
	 *     the FileSystemCacheManager
	 *     
	 * @param memento
	 */
	private void restoreFromMemento(JcifsCacheMemento memento)
	throws CacheException
	{
		// restore the byte channel factory
		setInstanceByteChannelFactory( JcifsByteChannelFactory.create(memento.getByteChannelFactoryMemento()) );
		
		for( EvictionStrategyMemento evictionStrategyMemento:memento.getEvictionStrategyMementos() )
		{
			EvictionStrategy evictionStrategy = 
				EvictionStrategyFactory.getSingleton().createEvictionStrategy(evictionStrategyMemento, defaultEvictionTimer);
			addEvictionStrategy(evictionStrategy);
		}
		
		for(RegionMemento regionMemento:memento.getRegionMementos())
		{
			if(regionMemento instanceof PersistentRegionMemento)
			{
				JcifsRegion region = JcifsRegion.create(
					this, 
					(PersistentRegionMemento)regionMemento
				); 
				addRegion(region);
			}
		}
		
		// note that set initialized to true is much more than a simple bit flip
		if(memento.isInitialized())
			setInitialized(true);
		// setting enabled to true is pretty much a simple bit flip
		if(memento.isEnabled())
			setEnabled(true);
	}
}
