/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.jcifs;


import gov.va.med.imaging.StackTraceAnalyzer;
import gov.va.med.imaging.storage.cache.*;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.CacheStateException;
import gov.va.med.imaging.storage.cache.exceptions.PersistenceIOException;
import gov.va.med.imaging.storage.cache.memento.ByteChannelFactoryMemento;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;

import javax.management.*;
import javax.management.openmbean.*;

import jcifs.smb.SmbFile;

import org.apache.log4j.Logger;

/**
 * @author       DNS
 *
 * A factory class that provides some management of instance file channels.
 * 
 */
public class JcifsByteChannelFactory 
implements InstanceByteChannelFactory<SmbFile>, DynamicMBean, CacheLifecycleListener
{
	public static final long defaultMaxChannelOpenDuration = 300000L;	// long default fore remote jukeboxes
	public static final long defaultSweepTime = 10000L;
	public static final boolean defaultTraceChannelInstantiation = true;
	
	private DateFormat df = new SimpleDateFormat("ddMMMyyyy hh:mm:ss");
	
	private long maxChannelOpenDuration = defaultMaxChannelOpenDuration;
	private long sweepTime = defaultSweepTime;
	private boolean traceChannelInstantiation = defaultTraceChannelInstantiation;
	
	Logger log = Logger.getLogger(this.getClass());
	private ChannelCleanupThread cleanupThread;
	
	// ===================================================================================================================
	// Factory Methods
	// ===================================================================================================================
	public static JcifsByteChannelFactory create()
	{
		return new JcifsByteChannelFactory(defaultMaxChannelOpenDuration, defaultSweepTime);
	}
	
	public static JcifsByteChannelFactory create(ByteChannelFactoryMemento memento)
	{
		return new JcifsByteChannelFactory(memento);
	}
	
	/**
	 * 
	 * @param memento
	 */
	private JcifsByteChannelFactory(ByteChannelFactoryMemento memento)
	{
		this(
			memento == null ? defaultMaxChannelOpenDuration : memento.getMaxChannelOpenDuration(), 
			memento == null ? defaultSweepTime : memento.getSweepTime() 
		);
	}
	
	/**
	 * 
	 * @param maxChannelOpenDuration - the maximum time a channel is allowed to be open
	 * @param sweepTime - the delay in the background thread that looks for open channels
	 */
	private JcifsByteChannelFactory(Long maxChannelOpenDuration, Long sweepTime)
	{
		this.maxChannelOpenDuration = maxChannelOpenDuration.longValue();
		this.sweepTime = sweepTime;
	}

	// ==================================================================================================
	// InstanceByteChannelFactory Implementation
	// Behavioral Modification Methods
	// ==================================================================================================
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.impl.filesystem.InstanceByteChannelFactoryImplMBean#getMaxChannelOpenDuration()
	 */
	public long getMaxChannelOpenDuration()
	{
		return maxChannelOpenDuration;
	}
	public void setMaxChannelOpenDuration(long maxChannelOpenDuration)
	{
		if(maxChannelOpenDuration > 0L)
			this.maxChannelOpenDuration = maxChannelOpenDuration;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.impl.filesystem.InstanceByteChannelFactoryImplMBean#getSweepTime()
	 */
	public long getSweepTime()
	{
		return sweepTime;
	}
	public void setSweepTime(long sweepTime)
	{
		if(sweepTime > 0L)
			this.sweepTime = sweepTime;
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.impl.filesystem.InstanceByteChannelFactoryImplMBean#getCurrentlyOpenReadableByteChannels()
	 */
	public int getCurrentlyOpenReadableByteChannels()
	{
		return openReadChannels.size();
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.impl.filesystem.InstanceByteChannelFactoryImplMBean#getCurrentlyOpenWritableByteChannels()
	 */
	public int getCurrentlyOpenWritableByteChannels()
	{
		return openWriteChannels.size();
	}

	/**
	 * If traceChannelInstantiation is set then the factory will record
	 * the stack trace when a channel is instantiated and report the stack
	 * trace when the channel is closed due to a timeout. 
	 * 
	 * @return
	 */
	public boolean isTraceChannelInstantiation()
	{
		return this.traceChannelInstantiation;
	}

	public void setTraceChannelInstantiation(boolean traceChannelInstantiation)
	{
		this.traceChannelInstantiation = traceChannelInstantiation;
	}

	// ==================================================================================================
	// InstanceByteChannelFactory Implementation
	// Business Methods
	// ==================================================================================================
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.filesystem.InstanceByteChannelFactory#getInstanceReadableByteChannel(java.io.File)
	 */
	public InstanceReadableByteChannel getInstanceReadableByteChannel(SmbFile instanceFile, InstanceByteChannelListener timeoutListener)
	throws PersistenceIOException, CacheException
	{
		try
		{
			InstanceReadableByteChannelImpl readable = new InstanceReadableByteChannelImpl(this, instanceFile);
			if(timeoutListener != null)
				this.putReadableChannel(readable, timeoutListener);
				
			return readable;
		} 
		catch (IOException ioX)
		{
			throw new PersistenceIOException(ioX);
		}
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.filesystem.InstanceByteChannelFactory#getInstanceWritableByteChannel(java.io.File)
	 */
	public InstanceWritableByteChannel getInstanceWritableByteChannel(SmbFile instanceFile, InstanceByteChannelListener timeoutListener) 
	throws PersistenceIOException, CacheException
	{
		try
		{
			InstanceWritableByteChannelImpl writable = new InstanceWritableByteChannelImpl(this, instanceFile );

			if(timeoutListener != null)
				putWritableChannel(writable, timeoutListener);
			
			return writable;
		} 
		catch (IOException ioX)
		{
			throw new PersistenceIOException(ioX);
		}
	}
	
	// =============================================================================================================================
	// Open Read/Write Channels list management.
	// All access to these maps should go through the calls provided below so that correct
	// synchronization may be provided.
	// =============================================================================================================================
	private Map<InstanceWritableByteChannelImpl, InstanceByteChannelListener> openWriteChannels = 
		Collections.synchronizedMap( new HashMap<InstanceWritableByteChannelImpl, InstanceByteChannelListener>() );
	private Map<InstanceReadableByteChannelImpl, InstanceByteChannelListener> openReadChannels = 
		Collections.synchronizedMap( new HashMap<InstanceReadableByteChannelImpl, InstanceByteChannelListener>() );
	
	private void putWritableChannel(InstanceWritableByteChannelImpl writable, InstanceByteChannelListener listener)
	{
		openWriteChannels.put(writable, listener);
	}

	private void putReadableChannel(InstanceReadableByteChannelImpl readable, InstanceByteChannelListener listener)
	{
		openReadChannels.put(readable, listener);
	}
	
	// ==================================================================================================
	// InstanceByteChannelFactory Implementation
	// Statistics Gathering Methods
	// ==================================================================================================
	/**
	 * Look through our internal list of open writable channels and return a
	 * reference to the byte channel that is open on the given file,
	 * or return null if not open.
	 * 
	 * @param instanceFile
	 * @return
	 */
	InstanceWritableByteChannelImpl getOpenWritableByteChannel(SmbFile instanceFile)
	{
		if(instanceFile == null)
			return null;
		
		synchronized(openWriteChannels)
		{
			for(InstanceWritableByteChannelImpl channel: openWriteChannels.keySet() )
			{
				SmbFile channelFile = channel.getFile();
				
				if(instanceFile.equals(channelFile))
					return channel;
			}
		}
		
		return null;
	}
	
	/**
	 * Look through our internal list of open readable channels and return a
	 * reference to the byte channel that is open on the given file,
	 * or return null if not open.
	 * 
	 * @param instanceFile
	 * @return
	 */
	InstanceReadableByteChannelImpl getOpenReadableByteChannel(SmbFile instanceFile)
	{
		if(instanceFile == null)
			return null;
		
		synchronized(openReadChannels)
		{
			for(InstanceReadableByteChannelImpl channel: openReadChannels.keySet() )
			{
				SmbFile channelFile = channel.getFile();
				
				if(instanceFile.equals(channelFile))
					return channel;
			}
		}
		
		return null;
	}
	
	/**
	 * This is NOT an idempotent method, it REMOVES the writable map entry.
	 * 
	 * @param writable
	 */
	private void notifyTimeoutWritableChannelListeners(InstanceWritableByteChannelImpl writable)
	{
		writableByteChannelClosed(writable, true);
	}
	
	/**
	 * This is NOT an idempotent method, it REMOVES the readable map entry.
	 * 
	 * @param readable
	 */
	private void notifyTimeoutReadableChannelListeners(InstanceReadableByteChannelImpl readable)
	{
		readableByteChannelClosed(readable, true);
	}
	
	
	/**
	 * Any should call this to remove the writable byte channel,
	 * else the listeners will get timeout signals. 
	 * 
	 * @param writable
	 */
	void writableByteChannelClosed(InstanceWritableByteChannelImpl writable, boolean errorClose)
	{
		InstanceByteChannelListener listener = openWriteChannels.get(writable);
		openWriteChannels.remove(writable);
		
		if(listener != null)
		{
			if(errorClose)
				listener.writeChannelIdleTimeout(writable);
			else
				listener.writeChannelClose(writable);
		}
	}
	/**
	 * A 'normal' close should call this to remove the readable byte channel,
	 * else the listeners will get timeout signals. 
	 * 
	 * @param readable
	 */
	void readableByteChannelClosed(InstanceReadableByteChannelImpl readable, boolean errorClose)
	{
		InstanceByteChannelListener listener = openReadChannels.get(readable);
		openReadChannels.remove(readable);
		
		if(listener != null)
		{
			if(errorClose)
				listener.readChannelIdleTimeout(readable);
			else
				listener.readChannelClose(readable);
		}
	}
	
	// ==========================================================================================
	// State persistence implementation
	// ==========================================================================================
	/**
	 * Create a Serializable representation of our state that may be
	 * used later to recreate our state.
	 * 
	 * @return
	 */
	public ByteChannelFactoryMemento createMemento()
	{
		ByteChannelFactoryMemento memento = new ByteChannelFactoryMemento();
		
		memento.setMaxChannelOpenDuration(getMaxChannelOpenDuration());
		memento.setSweepTime(getSweepTime());
		
		return memento;
	}

	public void restoreMemento(ByteChannelFactoryMemento memento)
	{
		setMaxChannelOpenDuration( memento.getMaxChannelOpenDuration() );
		setSweepTime( memento.getSweepTime() );
	}

	// ===============================================================================================
	// CacheLifecycleListener 
	// ===============================================================================================
	public void cacheLifecycleEvent(CacheLifecycleEvent event) throws CacheStateException
	{
		if(event == CacheLifecycleEvent.START)
		{
			if(this.sweepTime > 0L && this.maxChannelOpenDuration > 0L)
			{
				cleanupThread = new ChannelCleanupThread(this.sweepTime, this.maxChannelOpenDuration);
				cleanupThread.start();
			}
		}
		else if (event == CacheLifecycleEvent.STOP)
		{
			if(cleanupThread != null)
				cleanupThread.kill();
		}
	}

	// ===============================================================================================
	// JMX (management) related methods (DynamicMBean implementation)
	// ===============================================================================================
	
	/**
	 * @see javax.management.DynamicMBean#getMBeanInfo()
	 */
	private MBeanInfo openMBeanInfo;
	public synchronized MBeanInfo getMBeanInfo()
	{
		if(openMBeanInfo == null)
		{
			openMBeanInfo = new OpenMBeanInfoSupport(
					getClass().getName(),
                    "Byte channel factory for file system based cache implementations.",
                    new OpenMBeanAttributeInfo[] 
                    {
						new OpenMBeanAttributeInfoSupport("maxChannelOpenDuration", "Maximum time in milliseconds between channel usage", SimpleType.LONG, true, true, false),
						new OpenMBeanAttributeInfoSupport("sweepTime", "Time to delay between sweeping open channels", SimpleType.LONG, true, true, false),
						new OpenMBeanAttributeInfoSupport("traceChannelInstantiation", "Trace the code that opened channels", SimpleType.BOOLEAN, true, true, true),
						new OpenMBeanAttributeInfoSupport("openReadableByteChannels", "Number of read channels currently open", SimpleType.INTEGER, true, false, false),
						new OpenMBeanAttributeInfoSupport("openWritableByteChannels", "Number of write channels currently open", SimpleType.INTEGER, true, false, false)
                    },
                    new OpenMBeanConstructorInfo[]{},
                    new OpenMBeanOperationInfo[]{},
                    new MBeanNotificationInfo[]{}
			);
		}
		
		return openMBeanInfo;
	}

	/**
	 * @see javax.management.DynamicMBean#getAttribute(java.lang.String)
	 */
	public Object getAttribute(String attribute) 
	throws AttributeNotFoundException, MBeanException, ReflectionException
	{
		if( "maxChannelOpenDuration".equals(attribute) )
			return new Long(getMaxChannelOpenDuration());
		else if( "sweepTime".equals(attribute) )
			return new Long(getSweepTime());
		else if( "traceChannelInstantiation".equals(attribute) )
			return new Boolean(isTraceChannelInstantiation());
		else if( "openReadableByteChannels".equals(attribute) )
			return new Integer(getCurrentlyOpenReadableByteChannels());
		else if( "openWritableByteChannels".equals(attribute) )
			return new Integer(getCurrentlyOpenWritableByteChannels());
		else
			throw new AttributeNotFoundException("Attribute '" + attribute + "' not found");
	}

	/**
	 * @see javax.management.DynamicMBean#getAttributes(java.lang.String[])
	 */
	public AttributeList getAttributes(String[] attributes)
	{
		AttributeList list = new AttributeList();
		for(String attribute:attributes)
			try
			{
				list.add( new Attribute(attribute, getAttribute(attribute)) );
			} 
			catch (AttributeNotFoundException x)
			{
				x.printStackTrace();
			} 
			catch (MBeanException x)
			{
				x.printStackTrace();
			} 
			catch (ReflectionException x)
			{
				x.printStackTrace();
			}
		return list;
	}

	/**
	 * @see javax.management.DynamicMBean#setAttribute(javax.management.Attribute)
	 */
	public void setAttribute(Attribute attribute) 
	throws AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException
	{
		try
		{
			if( "maxChannelOpenDuration".equals(attribute.getName()) )
				setMaxChannelOpenDuration( (Long)attribute.getValue() );
			else if( "sweepTime".equals(attribute.getName()) )
				setSweepTime( (Long)attribute.getValue() );
			else if( "traceChannelInstantiation".equals(attribute.getName()) )
				setTraceChannelInstantiation( (Boolean)attribute.getValue() );
			else
				throw new AttributeNotFoundException("Attribute '" + attribute + "' not found.");
		} 
		catch (ClassCastException x)
		{
			throw new InvalidAttributeValueException("Attribute '" + attribute + "' values was of incorrect type.");
		}
	}

	/**
	 * @see javax.management.DynamicMBean#setAttributes(javax.management.AttributeList)
	 */
	public AttributeList setAttributes(AttributeList attributes)
	{
		for(int index=0; index < attributes.size(); ++index)
		{
			Attribute attribute = (Attribute)attributes.get(index);
			
			try
			{
				setAttribute(attribute);
			} 
			catch (AttributeNotFoundException x)
			{
				x.printStackTrace();
			} 
			catch (InvalidAttributeValueException x)
			{
				x.printStackTrace();
			} 
			catch (MBeanException x)
			{
				x.printStackTrace();
			} 
			catch (ReflectionException x)
			{
				x.printStackTrace();
			}
		}
		
		return attributes;
	}
	
	/**
	 * @see javax.management.DynamicMBean#invoke(java.lang.String, java.lang.Object[], java.lang.String[])
	 */
	public Object invoke(String actionName, Object[] params, String[] signature) 
	throws MBeanException, ReflectionException
	{
		return null;
	}

	// ===============================================================================================
	// The thread that monitors and cleans up channels that have been left hanging
	// ===============================================================================================
	/**
	 * 
	 * @author       BECKEC
	 *
	 */
	class ChannelCleanupThread 
	extends Thread
	{
		private long sweepInterval;
		private long maxChannelOpenDuration;
		private boolean running = true;
		
		ChannelCleanupThread(long sweepInterval, long maxChannelOpenDuration)
		{
			this.sweepInterval = sweepInterval;
			this.maxChannelOpenDuration = maxChannelOpenDuration;
			
			this.setDaemon(true);
		}
		
		public void kill()
		{
			this.running = false;
			this.interrupt();
		}

		@Override
		public void run()
		{
			while(running)
			{
				long minOpenTime = System.currentTimeMillis() - maxChannelOpenDuration;
				
				log.info("Sweeping write channels open before " + df.format(minOpenTime));
				try
				{
					List<InstanceWritableByteChannelImpl> writeChannelKillList = new ArrayList<InstanceWritableByteChannelImpl>();
					try
					{
						// do the kill/close in two loops to avoid concurrent modification exceptions
						for( InstanceWritableByteChannelImpl writeChannel:openWriteChannels.keySet() )
						{
							if( writeChannel.getLastAccessedTime() < minOpenTime )
							{
								log.warn("Writable Byte Channel " + writeChannel.toString() + " has remained open past the maximum allowable, forcing close!" );
								writeChannelKillList.add(writeChannel);
								
								if(isTraceChannelInstantiation())
								{
									StackTraceElement[] instantiatingStackTrace = writeChannel.getInstantiatingStackTrace();
									warnInstantiatingStackTrace(instantiatingStackTrace);
								}
							}
						}
					}
					catch(ConcurrentModificationException cmX)
					{
						// note that if we get a concurrent modification (i.e. another thread is opening or closing readable channels)
						// then log it but don't fail 'cause we can always close the channel later
						log.info("Concurrent modification exception while iterating open write channels, some overdue channels may not be closed immediately.");
					}
					
					log.info("Closing " + writeChannelKillList.size() + " write channels due to inactivity timeout");
					for(InstanceWritableByteChannelImpl deadChannel:writeChannelKillList)
					{
						try
						{
							deadChannel.error();
						} 
						catch (IOException e)
						{
							log.error(e);
						}
						notifyTimeoutWritableChannelListeners(deadChannel);
						openWriteChannels.remove(deadChannel);
					}
					
					log.info("Sweeping read channels open before " + df.format(minOpenTime));
					List<InstanceReadableByteChannelImpl> readChannelKillList = new ArrayList<InstanceReadableByteChannelImpl>();
					
					// do the kill/close in two loops to avoid concurrent modification exceptions
					try
					{
						for( InstanceReadableByteChannelImpl readChannel:openReadChannels.keySet() )
						{
							if( readChannel.getLastAccessedTime() < minOpenTime )
							{
								log.warn("Readable Byte Channel " + readChannel.toString() + " has remained open past the maximum allowable, notifying listeners" );
								readChannelKillList.add(readChannel);
								
								if(isTraceChannelInstantiation())
								{
									StackTraceElement[] instantiatingStackTrace = readChannel.getInstantiatingStackTrace();
									warnInstantiatingStackTrace(instantiatingStackTrace);
								}
							}
						}
					}
					catch(ConcurrentModificationException cmX)
					{
						// note that if we get a concurrent modification (i.e. another thread is opening or closing readable channels)
						// then log it but don't fail 'cause we can always close the channel later
						log.info("Concurrent modification exception while iterating open read channels, some overdue channels may not be closed immediately.");
					}
					
					log.info("Closing " + readChannelKillList.size() + " read channels due to inactivity timeout");
					for(InstanceReadableByteChannelImpl deadChannel:readChannelKillList)
					{
						try
						{
							deadChannel.close();
						} 
						catch (IOException e)
						{
							log.error(e);
						}
						notifyTimeoutReadableChannelListeners(deadChannel);
						openReadChannels.remove(deadChannel);
					}
					
					sleep(sweepInterval);
				} 
				catch (InterruptedException e)
				{
					// if someone interrupts us then run the thread out, we're done
					log.error(e);
					break;
				}
			}
		}
		
		/*
		 * Log warning about who is leaving channels open, if the info is available, else
		 * jujst log messages that a channel was left open.
		 */
		private void warnInstantiatingStackTrace(StackTraceElement[] instantiatingStackTrace)
		{
			if(instantiatingStackTrace != null)
			{
				StackTraceAnalyzer stAnalyzer = new StackTraceAnalyzer(instantiatingStackTrace);
				StackTraceElement element = stAnalyzer.getFirstElementNotInPackageHierarchy("gov.va.med.imaging.storage.cache");
				if(element != null)
					log.warn("Method '" + element.getClassName() + "." + element.getMethodName() + "' (or something it calls) is opening channels that are not being closed." + 
							"Entire call stack is:\n" + stAnalyzer.toString());
				else
					log.warn("Some method in this stack trace is opening channels that are not being closed:\n" + stAnalyzer.toString());
			}
			else
				log.warn("Stack Trace analysis of channel instantiating method is not available.  Turn TraceChannelInstantiation on to find the offending code.");
			
		}
	}		// end ChannelCleanup
}
