package gov.va.med.imaging.storage.cache.impl;

import gov.va.med.imaging.storage.cache.Group;
import gov.va.med.imaging.storage.cache.Instance;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.InstanceInaccessibleException;

import org.apache.log4j.Logger;

/**
 * A superclass of Group and Region implementations.
 * 
 * This class implements the recursive Group handling of Group and Region instances.
 * Any class that can be a parent of a set of Group instances should derive from this
 * class.
 * A class that can be a parent of both Group and Instance instances should derive
 * from PersistentGroupInstanceParent.
 * 
 * @author  DNS
 *
 */
public abstract class RecursiveGroupParent
{
	@SuppressWarnings("unused")
	private Logger log = Logger.getLogger(this.getClass());
	
	/*
	 * ==================================================================================
	 * These methods must implement the child level (i.e. directly beneath this group) 
	 * concrete class methods
	 * ==================================================================================
	 */
	abstract public Group getOrCreateChildGroup(String groupName)
	throws CacheException;
	
	abstract public Group getChildGroup(String groupName)
	throws CacheException;
	
	/**
	 * Get or create an Instance.  The groupName is an ordered array if the
	 * instances ancestor groups, starting from the progeny of this
	 * group.
	 */
	public Instance getOrCreateInstance(String[] groupName, String key) 
	throws CacheException
	{
		return getOrCreateInstance(groupName, key, true); 
	}

	/**
	 * 
	 * @param groupName
	 * @param key
	 * @return
	 * @throws CacheException
	 */
	public Instance getInstance(String[] groupName, String key) 
	throws CacheException
	{
		return getOrCreateInstance(groupName, key, false); 
	}

	/**
	 * Delete an Instance.  The groupName is an ordered array of the
	 * instances ancestor groups, starting from the progeny of this
	 * group.
	 */
	public void deleteInstance(String[] groupName, String key) 
	throws CacheException
	{
		Group childGroup = getChildGroup(groupName[0]);
		
		// if the instance ancestor groups do not exists, just return
		if(childGroup == null)
			return;
			
		if(groupName.length == 1)					// shortcut if childGroup is the parent of the Instance
		{
			childGroup.deleteChildInstance(key);
			return;
		}
		
		String[] progenyGroupNames = new String[groupName.length-1];
		System.arraycopy(groupName, 1, progenyGroupNames, 0, progenyGroupNames.length);
		childGroup.deleteInstance( progenyGroupNames, key);
		return;
	}
	
	/**
	 * NOTE: this method will NOT create an Instance instance.
	 * The derived class RecursiveGroupAndInstanceParent is called
	 * directly, if it needs to recurse through offspring Groups then
	 * it calls this method to do so.
	 * 
	 * @param groupName
	 * @param key
	 * @param allowCreate
	 * @return
	 * @throws CacheException
	 */
	public Instance getOrCreateInstance(String[] groupName, String key, boolean allowCreate) 
	throws CacheException
	{
		Group childGroup =  allowCreate ?
			getOrCreateChildGroup(groupName[0]) :
			getChildGroup(groupName[0]);
		
		if(childGroup == null)
		{
			if(allowCreate)
				log.error("Unable to create child group '" + groupName[0] + "'.");
			return null;
		}
		
		if(groupName.length == 1)					// shortcut if childGroup is the parent of the Instance
			return allowCreate ? childGroup.getOrCreateChildInstance(key) : childGroup.getChildInstance(key);
		
		String[] progenyGroupNames = new String[groupName.length-1];
		System.arraycopy(groupName, 1, progenyGroupNames, 0, progenyGroupNames.length);
		return allowCreate ? 
			childGroup.getOrCreateInstance( progenyGroupNames, key) :
			childGroup.getInstance( progenyGroupNames, key);
	}

	/*
	 * ==================================================================================
	 * The following methods implement the recursive functions in the Group interface.
	 * These functions delegate the child level (i.e. directly beneath this group) to
	 * concrete class methods.
	 * ==================================================================================
	 */
	public Group getOrCreateGroup(String[] groupName) 
	throws CacheException
	{
		if(groupName == null || groupName.length == 0)
			return null;

		Group childGroup = getOrCreateChildGroup(groupName[0]);
		
		String[] progenyGroupNames = new String[groupName.length-1];
		System.arraycopy(groupName, 1, progenyGroupNames, 0, progenyGroupNames.length);
		return groupName.length == 0 ? childGroup : childGroup.getOrCreateGroup( progenyGroupNames);
	}

	/**
	 * 
	 */
	public Group getGroup(String[] groupName) 
	throws CacheException
	{
		if(groupName == null || groupName.length == 0 || groupName[0] == null)
			return null;
		
		Group childGroup = getChildGroup(groupName[0]);
		
		if(childGroup == null)
			throw new InstanceInaccessibleException("Group '" + groupName[0] + "' not found.");
		
		String[] progenyGroupNames = new String[groupName.length-1];
		System.arraycopy(groupName, 1, progenyGroupNames, 0, progenyGroupNames.length);
		
		return groupName.length == 0 ? childGroup : childGroup.getGroup( progenyGroupNames);
	}

}
