package gov.va.med.imaging.storage.cache;

import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.UnknownEvictionStrategyException;
import gov.va.med.imaging.storage.cache.memento.RegionMemento;

import java.util.Iterator;
import java.util.regex.Pattern;


/**
 * 
 * 
 */
public interface Region
extends CacheLifecycleListener, MutableNamedObject
{
	// A region name must be 1 to 64 chars, start with a letter and contain letters, numbers, dashes and underscores
	public static Pattern NamePattern = Pattern.compile( "[a-zA-Z][a-zA-Z0-9-_]{0,63}" );
	public static final int SIZE_CALCULATION_RETRIES = 5;
	
	/**
	 * Initialization provides the Region the opportunity to 
	 * either initialization itself, or if it throws an exception,
	 * to halt the initialization of the entire cache.
	 * 
	 * @return
	 */
	public abstract Boolean isInitialized();
	public abstract void setInitialized(Boolean initialized)
	throws CacheException;
	
	/**
	 * Notify the region instance that it is starting or stopping.
	 * Depending on the cache implementation, it may be critical that it be
	 * notified of start and stop events.
	 * @param event
	 */
	public abstract void cacheLifecycleEvent(CacheLifecycleEvent event);

	/**
	 * 
	 */
	@Override
	public String getName();
	
	/**
	 * Report the available space (in bytes) for the storage backing this region.
	 * 
	 * @return
	 */
	public long getFreeSpace();
	
	/**
	 * Report the total space (in bytes) for the storage backing this region.
	 * 
	 * @return
	 */
	public long getTotalSpace();
	
	/**
	 * Return a "best effort" estimate of the size of all elements in the Region.
	 * This method is defined intentionally to NOT allow any exceptions.  Implementations
	 * are expected to soldier on and return a best guess with the prevailing conditions.
	 * In particular, groups added/deleted/modified while calculating the used space may be
	 * ignored for purposes of this method.
	 * @return
	 */
	public long getUsedSpace();
	
	public Iterator<? extends Group> getGroups() 
	throws CacheException;

	public Instance getOrCreateInstance(String[] group, String key)
	throws CacheException;

	public Instance getInstance(String[] group, String key) 
	throws CacheException;

	public void deleteInstance(String[] group, String key) 
	throws CacheException;
	
	public Group getOrCreateGroup(String[] group)
	throws CacheException;

	public Group getGroup(String[] group) 
	throws CacheException;
	
	/**
	 * Remove a child group.
	 *
	 */
	public void removeChildGroup(Group childGroup)
	throws CacheException;
	
	/**
	 * @param evictionStrategy
	 * @throws UnknownEvictionStrategyException 
	 */
	public abstract void setEvictionStrategyNames(String[] evictionStrategyName) 
	throws UnknownEvictionStrategyException;
	
	/**
	 * 
	 * @return
	 */
	public abstract String[] getEvictionStrategyNames();
	
	/**
	 * 
	 * @return
	 */
	public abstract EvictionStrategy[] getEvictionStrategies();
	
	/**
	 * 
	 * @param judge
	 * @return
	 * @throws CacheException
	 */
	public int evaluateAndEvictChildGroups(EvictionJudge<Group> judge)
	throws CacheException;
	
	/**
	 * Clear all groups from the regions
	 * @throws CacheException 
	 */
	public abstract void clear() 
	throws CacheException;
	
	/**
	 * @return
	 */
	public abstract RegionMemento createMemento();
}
