package gov.va.med.imaging.storage.cache;

import java.util.Iterator;
import java.util.regex.Pattern;

import gov.va.med.imaging.storage.cache.events.GroupLifecycleListener;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;

/**
 * 
 * 
 */
public interface Group
extends MutableNamedObject
{
	// A group name must be 1 to 64 chars, start with a letter and contain letters, numbers, dashes and underscores
	public static Pattern NamePattern = Pattern.compile( "[a-zA-Z][a-zA-Z0-9-_]{0,63}" );
	
	/**
	 * Return some human readable identifier for this group.
	 * This must be unique within the parent group, not necessarily across all groups.
	 * @return
	 */
	public String getName();
	
	public java.util.Date getLastAccessed() 
	throws CacheException;

	public long getSize()
	throws CacheException;

	public Iterator<? extends Group> getGroups()
	throws CacheException;
	
	public Iterator<? extends Instance> getInstances()
	throws CacheException;

	// i.e. A Group may contain other Group instances, which contain other Group instances,
	// which eventually contain Instance.
	// A call to getInstance(null, key) is equivalent to getInstance(key).
	// A call to createInstance(null, key) is equivalent to createInstance(key).
	/*
	 * =======================================================================
	 * The following methods operate recursively on the ancestors of the current
	 * group.
	 * =======================================================================
	 */
	public Instance getOrCreateInstance(String[] group, String key)
	throws CacheException;

	public Instance getInstance(String[] group, String key) 
	throws CacheException;

	public void deleteInstance(String[] group, String key) 
	throws CacheException;

	public Group getOrCreateGroup(String[] group)
	throws CacheException;

	public Group getGroup(String[] group) 
	throws CacheException;
	
	/*
	 * =======================================================================
	 * The folowing methods operate directly on the children of the current
	 * group, they are not recursive methods.
	 * =======================================================================
	 */
	public Instance getOrCreateChildInstance(String key)
	throws CacheException;

	public Instance getChildInstance(String key) 
	throws CacheException;

	public void deleteChildInstance(String key) 
	throws CacheException;

	public Group getOrCreateChildGroup(String groupname)
	throws CacheException;

	public Group getChildGroup(String groupname) 
	throws CacheException;

	/**
	 * Remove a child group.
	 *
	 */
	public void removeChildGroup(Group childGroup)
	throws CacheException;
	
	/**
	 * Remove a child instance.
	 *
	 */
	public void removeChildInstance(Instance childInstance)
	throws CacheException;

	public void removeAllChildGroups()
	throws CacheException;

	public void removeAllChildInstances()
	throws CacheException;

	/**
	 * Remove whatever persistent copies of ourselves that exist.
	 *
	 */
	public void remove()
	throws CacheException;
	
	public int evaluateAndEvictChildGroups(EvictionJudge<Group> judge)
	throws CacheException;
	
	// ======================================================================================================
	// Listener Management
	// ======================================================================================================
	public abstract void registerListener(GroupLifecycleListener listener);
	public abstract void unregisterListener(GroupLifecycleListener listener);
	
}
