/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Mar 27, 2012
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:        DNS
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone ( DNS
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.imagegear.datasource;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import gov.va.med.imaging.artifactsource.ResolvedArtifactSource;
import gov.va.med.imaging.channels.ByteStreamPump;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.core.interfaces.exceptions.SecurityException;
import gov.va.med.imaging.disclosure.pdf.PdfStatusReturnCode;
import gov.va.med.imaging.disclosure.pdf.ROIMakePDFDebugMode;
import gov.va.med.imaging.exchange.business.Patient;
import gov.va.med.imaging.exchange.enums.ImageFormat;
import gov.va.med.imaging.exchange.storage.ByteBufferBackedImageInputStream;
import gov.va.med.imaging.exchange.storage.DataSourceImageInputStream;
import gov.va.med.imaging.roi.datasource.ImageMergeWriterDataSourceSpi;

/**
 * @author        DNS
 *
 */
public class ImageGearImageMergeWriterDataSourceService
extends AbstractImageGearDataSourceService
implements ImageMergeWriterDataSourceSpi
{
	public final static String SUPPORTED_PROTOCOL = "imagegear";
	public final static float protocolVersion = 1.0F;
	
	public ImageGearImageMergeWriterDataSourceService(ResolvedArtifactSource resolvedArtifactSource, String protocol)
	{
		super(resolvedArtifactSource, protocol);
	}
	
	// to support local data source
	public ImageGearImageMergeWriterDataSourceService(ResolvedArtifactSource resolvedArtifactSource)
	{
		super(resolvedArtifactSource, SUPPORTED_PROTOCOL);
	}

	@Override
	public OutputStream getOutputStream(String groupIdentifier,
			String objectIdentifier, ImageFormat imageFormat, String objectDescription) 
	throws ConnectionException, MethodException
	{
		setDataSourceMethodVersionAndProtocol("getOutputStream", getDataSourceVersion());
		
		File outputDirectory = getGroupDirectory(groupIdentifier);
		outputDirectory.mkdirs();
		String filename = outputDirectory.getAbsolutePath() + File.separatorChar + objectIdentifier + "." + imageFormat.getDefaultFileExtension();
		
		// after the file is written the manifest file should be added with the entry
		// not really able to do after file is written
		appendObjectToInputManifest(groupIdentifier, filename, objectDescription);
		try
		{
			return new FileOutputStream(filename);
		} 
		catch (FileNotFoundException e)
		{
			throw new MethodException("FileNotFoundException creating output file stream, " + e.getMessage());
		}
	}
	
	private void appendObjectToInputManifest(String groupIdentifier, String filename, String objectDescription) 
	throws MethodException
	{
		File manifestFile = getGroupFileList(groupIdentifier);
		
		try
		{
			BufferedWriter writer = new BufferedWriter(new FileWriter(manifestFile, true));
			writer.write(filename + "^" + objectDescription + "\n");
			writer.close();
		} 
		catch (IOException ioX)
		{
			throw new MethodException("Error writing to input manifest file, " + ioX.getMessage(), ioX);
		}	
	}
	
	private File getGroupFileList(String groupIdentifier)
	{
		File outputDirectory = getGroupDirectory(groupIdentifier);
		return new File( outputDirectory.getAbsolutePath() + File.separatorChar + groupIdentifier + ".txt");
	}
	
	private File getGroupDirectory(String groupIdentifier)
	{
		String groupOutputDirectory = getImageGearConfiguration().getGroupOutputDirectory();
		groupOutputDirectory = groupOutputDirectory + File.separatorChar + groupIdentifier;
		return new File(groupOutputDirectory);
	}

	@Override
	public DataSourceImageInputStream mergeObjects(String groupIdentifier, Patient patient)
	throws ConnectionException, MethodException
	{
		getStatistics().incrementDisclosureWriteRequests();
		setDataSourceMethodVersionAndProtocol("mergeObjects", getDataSourceVersion());
		
		File groupDirectory = getGroupDirectory(groupIdentifier);
		File groupFile = getGroupFileList(groupIdentifier); // list of input files
		File manifestDirectory = new File(groupDirectory.getAbsolutePath() + File.separatorChar + "output");
		
		String debugOutputFilename = groupDirectory.getAbsolutePath() + File.separatorChar + "pdf_disclosure.log";
		
		int retVal = -1;
		try
		{
			Process process = createExecutingProcess(groupIdentifier, groupFile, 
					groupDirectory, manifestDirectory, patient);
			
			InputStream is = process.getInputStream();
			InputStreamReader isr = new InputStreamReader(is);
		    BufferedReader br = new BufferedReader(isr);
		    BufferedWriter debugWriter = new BufferedWriter(new FileWriter(debugOutputFilename, false));
		    String line = null;
		    
		    while ((line = br.readLine()) != null) 
		    {
		         debugWriter.write(line + "\n");
		    }
		    
		    CountDownLatch countdownLatch = new CountDownLatch(1);
		    long timeout = getImageGearConfiguration().getThreadTimeoutMs();
		    ImageGearWorkerThread thread = new ImageGearWorkerThread(process, countdownLatch, "PDF Merge [" + groupIdentifier + "]");
		    thread.start();
		    try
			{
		    	getLogger().debug("Waiting '" + timeout + "' ms for PDF thread to complete.");
				countdownLatch.await(timeout, 
						TimeUnit.MILLISECONDS);
			}
			catch (InterruptedException x)
			{
				String msg = "InterruptedException waiting for thread to generate PDF, " + x.getMessage();
				getLogger().error(msg, x);
				throw new MethodException(msg, x);
			}
		    
			if(countdownLatch.getCount() > 0)
				throw new MethodException("Unable to create PDF in allowed time '" + timeout + " ms.");
			retVal = thread.getExitCode();
		    getLogger().debug("PDF process completed with exit code '" + retVal + "'.");
			//retVal = process.waitFor();
			debugWriter.flush();
			debugWriter.close();
			/*
			retVal = ROIMakePDF.MakePdfDoc(groupIdentifier, debugMode.getValue(), roiOfficeName,
				groupFile.getAbsolutePath(), groupDirectory.getAbsolutePath(), manifestDirectory.getName(), 
				patient.getPatientName(), 
				patient.getPatientIcn(), ssn, dobFormat.format(patient.getDob()), 
				dateFormat.format(now.getTime()), timeFormat.format(now.getTime()));
				*/
		}
		catch(Exception ex)
		{
			getLogger().error("Exception thrown generating PDF disclosure, " + ex.getMessage(), ex);
			getStatistics().incrementDisclosureWriteFailures();
			throw new MethodException(ex);
		}
		
		PdfStatusReturnCode returnCode = PdfStatusReturnCode.getFromCode(retVal);
		if(returnCode == PdfStatusReturnCode.success)
		{
			getLogger().info("Completed generating PDF result, status is success. Rolling up results into zip file");
			ByteStreamPump pump = ByteStreamPump.getByteStreamPump();
			// output should be in the manifestDirectory
			File zipFile = new File(groupDirectory.getAbsolutePath() + File.separatorChar + "result.zip");
			try
			{
				ZipOutputStream zipStream = new ZipOutputStream(new FileOutputStream(zipFile));
				File [] files = manifestDirectory.listFiles();
				for(File file : files)
				{
					ZipEntry entry = new ZipEntry(file.getName());
					entry.setSize(file.length());
					zipStream.putNextEntry(entry);
					
					FileInputStream input = new FileInputStream(file);
					pump.xfer(input, zipStream);
					input.close();					
				}
				zipStream.close();
				getLogger().info("Zip file created, returning input stream to zip file");
				getStatistics().incrementDisclosureWriteSuccess();
				return new ByteBufferBackedImageInputStream(new FileInputStream(zipFile), 
						(int)zipFile.length());
				
			} 
			catch (FileNotFoundException fnfX)
			{
				getStatistics().incrementDisclosureWriteFailures();
				throw new MethodException("Exception creating zip output stream, " + fnfX.getMessage(), fnfX);				
			}
			catch(IOException ioX)
			{
				getStatistics().incrementDisclosureWriteFailures();
				throw new MethodException("Exception creating zip output stream, " + ioX.getMessage(), ioX);			}
		}
		else
		{
			getStatistics().incrementDisclosureWriteFailures();
			throw new MethodException("Error merging artifacts into PDF, returned status '" + (returnCode == null ? retVal : returnCode.getDescription()) + "'.");
		}
	}
	
	private Process createExecutingProcess(String groupIdentifier, 
			File groupFile, File groupDirectory, File manifestDirectory, Patient patient)
	throws IOException
	{
		 
		ROIMakePDFDebugMode debugMode = getDebugMode();
			
		DateFormat dateFormat = new SimpleDateFormat("MMMM_dd_yyyy");
		DateFormat timeFormat = new SimpleDateFormat("kk_mm_ss");
		DateFormat dobFormat = new SimpleDateFormat("MM/dd/yyyy");
			
		Calendar now = Calendar.getInstance();	
		String roiOfficeName = getImageGearConfiguration().getRoiOfficeName();
		String ssn = patient.getFilteredSsn();
		
		StringBuilder sb = new StringBuilder();
		String pdfExecutable = getImageGearConfiguration().getPdfGeneratorExePath();
		sb.append(pdfExecutable);
		sb.append(" \"" + groupIdentifier + "\"");
		sb.append(" \"" + debugMode.getValue() + "\"");
		sb.append(" \"" + roiOfficeName + "\"");		
		//sb.append(" \"" + groupFile.getAbsolutePath()  + "\"");
		sb.append(" \"" + groupDirectory.getAbsolutePath() + "\""); // this is where it looks for the input file, it appends the groupIdentifier and .txt
		sb.append(" \"" + groupDirectory.getAbsolutePath() + "\"");
		sb.append(" \"" + manifestDirectory.getName() + "\"");
		sb.append(" \"" + patient.getPatientName() + "\"");
		if(patient.isPatientIcnIncluded())
			sb.append(" \"" + patient.getPatientIcn() + "\"");
		else
			sb.append(" \"" + patient.getDfn() + "\"");
		sb.append(" \"" + ssn + "\"");
		sb.append(" \"" + dobFormat.format(patient.getDob()) + "\"");
		sb.append(" \"" + dateFormat.format(now.getTime()) + "\"");
		sb.append(" \"" + timeFormat.format(now.getTime()) + "\"");
		getLogger().debug("Executing PDF merge process [" + sb.toString() + "]");
		// not outputting to the transaction log because it contains patient identification information we don't want in the transaction log
		//TransactionContextFactory.get().addDebugInformation("PDF merge process [" + sb.toString() + "]");
		Process process = Runtime.getRuntime().exec(sb.toString());
		return process;
	}

	@Override
	public boolean isVersionCompatible() 
	throws SecurityException
	{
		return true;
	}
	
	protected String getDataSourceVersion()
	{
		return "1";
	}
}
