/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Mar 23, 2012
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:      DNS
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone  DNS
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.imagegear.datasource;

import gov.va.med.imaging.GUID;
import gov.va.med.imaging.artifactsource.ResolvedArtifactSource;
import gov.va.med.imaging.channels.ByteStreamPump;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.core.interfaces.exceptions.SecurityException;
import gov.va.med.imaging.disclosure.pdf.PdfStatusReturnCode;
import gov.va.med.imaging.disclosure.pdf.ROIMakePDFDebugMode;
import gov.va.med.imaging.exchange.business.annotations.ImageAnnotationDetails;
import gov.va.med.imaging.exchange.enums.ImageFormat;
import gov.va.med.imaging.exchange.storage.ByteBufferBackedInputStream;
import gov.va.med.imaging.exchange.storage.DataSourceInputStream;
import gov.va.med.imaging.roi.datasource.ImageAnnotationWriterDataSourceSpi;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

/**
 * @author     DNS
 *
 */
public class ImageGearImageAnnotationWriterDataSourceService
extends AbstractImageGearDataSourceService
implements ImageAnnotationWriterDataSourceSpi
{
	public final static String SUPPORTED_PROTOCOL = "imagegear";
	public final static float protocolVersion = 1.0F;
	
	public ImageGearImageAnnotationWriterDataSourceService(ResolvedArtifactSource resolvedArtifactSource, String protocol)
	{
		super(resolvedArtifactSource, protocol);
	}
	
	// to support local data source
	public ImageGearImageAnnotationWriterDataSourceService(ResolvedArtifactSource resolvedArtifactSource)
	{
		super(resolvedArtifactSource, SUPPORTED_PROTOCOL);
	}
	
	private final static ByteStreamPump pump = ByteStreamPump.getByteStreamPump();

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.roi.datasource.ImageAnnotationWriterDataSourceSpi#burnImageAnnotations(java.io.InputStream, gov.va.med.imaging.roi.datasource.ImageAnnotationDetails)
	 */
	@Override
	public DataSourceInputStream burnImageAnnotations(InputStream inputStream, ImageFormat imageFormat,
			ImageAnnotationDetails imageAnnotationDetails)
	throws MethodException, ConnectionException
	{
		setDataSourceMethodVersionAndProtocol("burnImageAnnotations", getDataSourceVersion());
		getStatistics().incrementBurnAnnotationRequests();
		
		GUID guid = new GUID();
		getLogger().info("Writing annotations to directory '" + guid.toLongString() + "'.");
		String groupOutputDirectory = getImageGearConfiguration().getGroupOutputDirectory();
		groupOutputDirectory = groupOutputDirectory + File.separatorChar + guid.toLongString();
		File outputDirectory = new File(groupOutputDirectory);
		outputDirectory.mkdirs();
		
		File imageFile = new File(outputDirectory.getAbsolutePath() + File.separatorChar + "image" + "." + imageFormat.getDefaultFileExtension());
		try
		{
			FileOutputStream imageOutputStream = new FileOutputStream(imageFile, false);
			
			pump.xfer(inputStream, imageOutputStream);
			inputStream.close();
			imageOutputStream.close();
		}
		catch(FileNotFoundException fnfX)
		{
			getStatistics().incrementBurnAnnotationFailures();
			getLogger().error("FileNotFoundException creating output file '" + imageFile.getAbsolutePath() + "', " + fnfX.getMessage());
			throw new MethodException(fnfX);
		}
		catch(IOException ioX)
		{
			getStatistics().incrementBurnAnnotationFailures();
			getLogger().error("IOException creating output file '" + imageFile.getAbsolutePath() + "', " + ioX.getMessage());
			throw new MethodException(ioX);
		}
		
		File annotationFile = new File(outputDirectory.getAbsolutePath() + File.separatorChar + "annotations.xml");
		try
		{
			BufferedWriter annotationWriter = new BufferedWriter(new FileWriter(annotationFile, false));
			annotationWriter.write(imageAnnotationDetails.getAnnotationXml());
			annotationWriter.flush();
			annotationWriter.close();
		}
		catch(IOException ioX)
		{
			getStatistics().incrementBurnAnnotationFailures();
			getLogger().error("IOException creating annotation file '" + annotationFile.getAbsolutePath() + "', " + ioX.getMessage());
			throw new MethodException(ioX);
		}
		
		//File annotatedFile = new File(outputDirectory.getAbsolutePath() + File.separatorChar + "annotated_image");
		
		String debugOutputFilename = outputDirectory.getAbsolutePath() + File.separatorChar + "burn_annotation.log";
		
		// image is in temp directory and annotations are in temp directory
		int retVal = -1;
		try
		{
			getLogger().info("About to merge annotations with image.");
			Process process =
				createExecutingProcess(guid.toLongString(), imageFile, annotationFile, 
						outputDirectory.getAbsoluteFile());
			
			InputStream is = process.getInputStream();
			InputStreamReader isr = new InputStreamReader(is);
		    BufferedReader br = new BufferedReader(isr);
		    BufferedWriter debugWriter = new BufferedWriter(new FileWriter(debugOutputFilename, false));
		    String line = null;
		    
		    while ((line = br.readLine()) != null) 
		    {
		         debugWriter.write(line + "\n");
		    }
			
			//retVal = process.waitFor();
		    CountDownLatch countdownLatch = new CountDownLatch(1);
		    long timeout = getImageGearConfiguration().getThreadTimeoutMs();
		    ImageGearWorkerThread thread = new ImageGearWorkerThread(process, countdownLatch, "Burn Annotation [" + guid.toLongString() + "]");
		    thread.start();
		    try
			{
		    	getLogger().debug("Waiting '" + timeout + "' ms for Annotation Burn thread to complete.");
				countdownLatch.await(timeout, 
						TimeUnit.MILLISECONDS);
			}
			catch (InterruptedException x)
			{
				String msg = "InterruptedException waiting for thread to burn annotations, " + x.getMessage();
				getLogger().error(msg, x);
				throw new MethodException(msg, x);
			}
		    
			if(countdownLatch.getCount() > 0)
				throw new MethodException("Unable to burn annotations in allowed time '" + timeout + " ms.");
			retVal = thread.getExitCode();
		    getLogger().debug("Annotation Burn completed with exit code '" + retVal + "'.");
		    
			debugWriter.flush();
			debugWriter.close();
			
			PdfStatusReturnCode returnCode = PdfStatusReturnCode.getFromCode(retVal);
			if(returnCode == PdfStatusReturnCode.success)
			{
				File annotatedFile = new File(
						outputDirectory.getAbsolutePath() + File.separatorChar + "image_Annotated." + imageFormat.getDefaultFileExtension());
				getLogger().info("Successfully burned in image annotations, status is success. Returning input stream to annotated image");
				if(annotatedFile.exists())
				{
					try
					{
						ByteBufferBackedInputStream result = new ByteBufferBackedInputStream(
								new FileInputStream(annotatedFile), (int)annotatedFile.length());
						getStatistics().incrementBurnAnnotationSuccess();
						return result;
					}
					catch(FileNotFoundException fnfX)
					{
						getStatistics().incrementBurnAnnotationFailures();
						throw new MethodException("FileNotFoundException opening annotated image, " + fnfX.getMessage(), fnfX);
					}
				}
				else
				{
					getStatistics().incrementBurnAnnotationFailures();
					throw new MethodException("Annotated File does not exist despite successful response from created annotated image");
				}
			}
			else
			{
				getStatistics().incrementBurnAnnotationFailures();
				throw new MethodException("Error burning annotations into image, returned status '" + retVal + "'.");
			}
		}
		catch(Exception ex)
		{
			getLogger().error("Exception thrown burning image annotations, " + ex.getMessage());
			getStatistics().incrementBurnAnnotationFailures();
			throw new MethodException(ex);
		}
	}
	
	private Process createExecutingProcess(String jobId, 
			File imageFile, File annotationsXml, File outputFile)
	throws IOException
	{
		ROIMakePDFDebugMode debugMode = getDebugMode();
		
		StringBuilder sb = new StringBuilder();
		String annotationExecutable = getImageGearConfiguration().getBurnAnnotationsExePath();
		sb.append(annotationExecutable);
		sb.append(" \"" + jobId + "\"");
		sb.append(" \"" + debugMode.getValue() + "\"");
		sb.append(" \"" + imageFile.getAbsolutePath() + "\"");		
		sb.append(" \"" + annotationsXml.getAbsolutePath() + "\"");
		sb.append(" \"" + outputFile.getAbsolutePath() + "\"");
		getLogger().debug("Executing Annotation burn process [" + sb.toString() + "]");
		// not outputting to the transaction log because it contains patient identification information we don't want in the transaction log
		//TransactionContextFactory.get().addDebugInformation("Annotation burn process [" + sb.toString() + "]");
		Process process = Runtime.getRuntime().exec(sb.toString());
		return process;
	}

	@Override
	public boolean isVersionCompatible() 
	throws SecurityException
	{
		return true;
	}
	
	protected String getDataSourceVersion()
	{
		return "1";
	}
}
