/**
 * 
 */
package gov.va.med.imaging.terminology;

import gov.va.med.imaging.terminology.SchemeTranslationProvider.SchemeTranslationServiceMapping;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.ServiceLoader;
import org.apache.log4j.Logger;

/**
 * @author  DNS
 *
 */
public class SchemeTranslationServiceFactory
{
	private static SchemeTranslationServiceFactory singleton;
	private static Logger logger = Logger.getLogger(SchemeTranslationServiceFactory.class);
	
	public static synchronized SchemeTranslationServiceFactory getFactory()
	{
		if(singleton == null)
			singleton = new SchemeTranslationServiceFactory();
		
		return singleton;
	}
	
	// ========================================================================
	private ServiceLoader<SchemeTranslationProvider> serviceLoader;

	private SchemeTranslationServiceFactory()
	{
		serviceLoader = ServiceLoader.load(SchemeTranslationProvider.class);
	}

	private ServiceLoader<SchemeTranslationProvider> getServiceLoader()
	{
		return this.serviceLoader;
	}
	
	private List<SchemeTranslationProvider.SchemeTranslationServiceMapping> availableServices; 
	
	private synchronized List<SchemeTranslationProvider.SchemeTranslationServiceMapping> getAvailableServices()
	{
		if(availableServices == null)
		{
			availableServices = new ArrayList<SchemeTranslationProvider.SchemeTranslationServiceMapping>();
			
			for(SchemeTranslationProvider provider : getServiceLoader() )
				availableServices.addAll( provider.getSchemeTranslationServiceMappings() );
		}
		
		return availableServices;
	}
	
	// ===============================================================================================
	// Various Factory Methods
	// ===============================================================================================
	
	/**
	 * 
	 * @param sourceScheme
	 * @param destinationScheme
	 * @return
	 */
	public SchemeTranslationSPI getSchemeTranslator(String sourceScheme, String destinationScheme)
	throws URISyntaxException
	{
		return getSchemeTranslator(null, new URI(sourceScheme), new URI(destinationScheme));
	}

	/**
	 * 
	 * @param manufacturer
	 * @param sourceScheme
	 * @param destinationScheme
	 * @return
	 * @throws URISyntaxException
	 */
	public SchemeTranslationSPI getSchemeTranslator(String manufacturer, String sourceScheme, String destinationScheme) 
	throws URISyntaxException
	{
		return getSchemeTranslator( manufacturer, new URI(sourceScheme), new URI(destinationScheme) );
	}
	
	/**
	 * 
	 * @param sourceScheme
	 * @param destinationScheme
	 * @return
	 */
	public SchemeTranslationSPI getSchemeTranslator(URI sourceScheme, URI destinationScheme) 
	{
		return getSchemeTranslator(null, sourceScheme, destinationScheme);
	}
	
	public SchemeTranslationSPI getSchemeTranslator(String manufacturer, URI sourceScheme, URI destinationScheme) 
	{
		if(sourceScheme == null)
		{
			logger.warn("The URI identifying the source coding scheme is null, no translation available.");
			return null;
		}
		if(destinationScheme == null)
		{
			logger.warn("The URI identifying the destination coding scheme is null, no translation available.");
			return null;
		}
		
		CodingScheme sourceCodingScheme = CodingScheme.valueOf(sourceScheme);
		CodingScheme destinationCodingScheme = CodingScheme.valueOf(destinationScheme);
		
		if(sourceCodingScheme == null)
		{
			logger.warn("The URI identifying the source coding scheme '" + sourceScheme + "' does not map to a known coding scheme.");
			return null;
		}
		if(destinationCodingScheme == null)
		{
			logger.warn("The URI identifying the destination coding scheme '" + destinationCodingScheme + "' does not map to a known coding scheme.");
			return null;
		}
		
		return getSchemeTranslator(manufacturer, sourceCodingScheme, destinationCodingScheme);
	}
	
	public SchemeTranslationSPI getSchemeTranslator(CodingScheme sourceScheme, CodingScheme destinationScheme) 
	{
		return getSchemeTranslator( null, sourceScheme, destinationScheme );
	}

	/**
	 * 
	 * @param manufacturer
	 * @param sourceScheme
	 * @param destinationScheme
	 * @return
	 * @throws URISyntaxException 
	 */
	public SchemeTranslationSPI getSchemeTranslator(String manufacturer, CodingScheme sourceScheme, CodingScheme destinationScheme)
	{
		logger.debug("Getting scheme translator(" + manufacturer + "," + sourceScheme.toString() + "," + destinationScheme.toString() + ").");
		if(sourceScheme == null || destinationScheme == null)
		{
			logger.info("scheme translator(" + manufacturer + "," + sourceScheme.toString() + "," + destinationScheme.toString() + 
				") is NULL because either source or destination scheme is null");
			return null;
		}
		
		// IdentitySchemeTranslation just passes back what it is given
		if(sourceScheme.equals(destinationScheme))
		{
			logger.info("scheme translator(" + manufacturer + "," + sourceScheme.toString() + "," + destinationScheme.toString() + ") is IDENTITY");
			return IdentitySchemeTranslation.create(sourceScheme);
		}
		
		for(SchemeTranslationServiceMapping serviceMapping : getAvailableServices() )
		{
			if( (manufacturer == null || manufacturer.equals(serviceMapping.getManufacturer())) &&
				sourceScheme.equals(serviceMapping.getSourceScheme()) && 
				destinationScheme.equals(serviceMapping.getDestinationScheme()) )
			{
				Class<? extends SchemeTranslationSPI> translationClass = serviceMapping.getTranslationClass();
				
				SchemeTranslationSPI translator = getOrCreateServiceInstance(translationClass);
				logger.debug("scheme translator(" + manufacturer + "," + sourceScheme.toString() + "," + destinationScheme.toString() + 
						") is '" + (translator == null ? "NULL (failed to create or get)" : translator.getClass().getName()) + 
						"'.");
				
				return translator;
			}
		}
		
		logger.info("scheme translator(" + manufacturer + "," + sourceScheme.toString() + "," + destinationScheme.toString() + ") is NULL");
		return null;
	}
	
	private List<SchemeTranslationSPI> serviceCache = new ArrayList<SchemeTranslationSPI>();
	
	/**
	 * Get from the cache or create a new service instance (and cache it).
	 * 
	 * @param translationClass
	 * @return
	 */
	private SchemeTranslationSPI getOrCreateServiceInstance(Class<? extends SchemeTranslationSPI> translationClass)
	{
		if(translationClass == null)
			return null;
		
		synchronized (serviceCache)
		{
			for(SchemeTranslationSPI serviceInstance : serviceCache)
				if( translationClass.getCanonicalName().equals(serviceInstance.getClass().getCanonicalName()) )
						return serviceInstance;
			
			try
			{
				SchemeTranslationSPI newServiceInstance = translationClass.newInstance();
				serviceCache.add(newServiceInstance);
				return newServiceInstance;
			}
			catch (InstantiationException x)
			{
				logger.error(x);
				return null;
			}
			catch (IllegalAccessException x)
			{
				logger.error(x);
				return null;
			}
		}
	}
}
