/**
 * 
 * Property of ISI Group, LLC
 * Date Created: Jul 8, 2014
 * Developer: Julian Werfel
 */
package gov.va.med.imaging.hydra.web;

import gov.va.med.SERIALIZATION_FORMAT;
import gov.va.med.URNFactory;
import gov.va.med.imaging.ImageURN;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.core.interfaces.exceptions.SecurityCredentialsExpiredException;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.enums.ImageFormat;
import gov.va.med.imaging.exchange.enums.ImageQuality;
import gov.va.med.imaging.transactioncontext.TransactionContext;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;
import gov.va.med.imaging.wado.AbstractBaseImageServlet;

import java.io.IOException;
import java.io.OutputStream;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author Julian
 *
 */
public abstract class AbstractHydraImageServlet
extends AbstractBaseImageServlet
{
	private static final long serialVersionUID = 4698967362249071140L;


	/* (non-Javadoc)
	 * @see gov.va.med.imaging.wado.AbstractBaseImageServlet#getUserSiteNumber()
	 */
	@Override
	public String getUserSiteNumber()
	{
		return null;
	}
	
	protected abstract List<ImageFormat> getAcceptableResponseContent(HttpServletRequest request)
	throws MethodException;
	
	protected abstract ImageQuality getImageQuality(HttpServletRequest request);
	
	protected abstract String getOperationName();

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.wado.AbstractBaseImageServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response)
	throws ServletException, IOException
	{
		TransactionContext transactionContext = TransactionContextFactory.get();
		transactionContext.setRequestType("Hydra Web App V1 " + getOperationName());
		String imageUrnString = request.getParameter("imageUrn");
		try
		{
			ImageURN imageUrn = URNFactory.create(imageUrnString, SERIALIZATION_FORMAT.RAW, ImageURN.class); 
			transactionContext.setPatientID(imageUrn.getThePatientIdentifier().toString());
			transactionContext.setUrn(imageUrnString);
			
			OutputStream outStream = response.getOutputStream();
			getLogger().info("Requesting image '" + imageUrnString + "'.");
			long bytesTransferred = streamImageInstanceByUrn(imageUrn, getImageQuality(request), 
				getAcceptableResponseContent(request), outStream, 
					new HydraImageMetadataNotification(response), false);
			
			transactionContext.setEntriesReturned( bytesTransferred==0 ? 0 : 1 );
			transactionContext.setFacadeBytesSent(bytesTransferred);
			transactionContext.setResponseCode(HttpServletResponse.SC_OK + "");
		}
		catch(URNFormatException urnfX)
		{
			String msg = "URNFormatException, " + urnfX.getMessage();
			getLogger().error(msg);
			transactionContext.setResponseCode(HttpServletResponse.SC_PRECONDITION_FAILED + "");
			transactionContext.setErrorMessage(msg);
			response.sendError(HttpServletResponse.SC_PRECONDITION_FAILED, msg);
		}
		catch(ImageServletException isX)
		{
			String msg = isX.getMessage();
			getLogger().error(msg);
			transactionContext.setResponseCode(isX.getResponseCode() + "");
			transactionContext.setErrorMessage(msg);
			response.sendError(isX.getResponseCode(), isX.getMessage());
		}
		catch(SecurityCredentialsExpiredException sceX)
		{
			String msg = "SecurityCredentials expired: " + sceX.getMessage();
			// logging of error already done
			// just need to set appropriate error code
			transactionContext.setResponseCode(HttpServletResponse.SC_PRECONDITION_FAILED + "");
			transactionContext.setErrorMessage(msg);
			response.sendError(HttpServletResponse.SC_PRECONDITION_FAILED, msg);
		}
		catch(MethodException mX)
		{
			String msg = "MethodException, " + mX.getMessage();
			getLogger().error(msg);
			transactionContext.setResponseCode(HttpServletResponse.SC_INTERNAL_SERVER_ERROR + "");
			transactionContext.setErrorMessage(msg);
			response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, msg);
		}
		
	}
}
