package gov.va.med.imaging.dicom.io;

import gov.va.med.imaging.dicom.dataset.TransferSyntaxUid;
import gov.va.med.imaging.dicom.dataset.elements.DataElementTag;
import gov.va.med.imaging.dicom.parser.io.DataElementLimitedInputStream;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;

import junit.framework.TestCase;

public class TestDataElementLimitedInputStream 
extends TestCase
{
	private DataElementLimitedInputStream openInputStream(String resourceName)
	{
		InputStream inStream = this.getClass().getClassLoader().getResourceAsStream(resourceName);
		assertNotNull(inStream);
		return new DataElementLimitedInputStream(
				new DataInputStream(inStream), 
				TransferSyntaxUid.RAW_EXPLICIT_VR_LITTLEENDIAN);
	}
	
	private DataElementLimitedInputStream openInputStream(DataElementLimitedInputStream parent, long maxLength)
	{
		return new DataElementLimitedInputStream(parent, maxLength);
	}
	
	private DataElementLimitedInputStream openInputStream(DataElementLimitedInputStream parent, DataElementTag... terminatorTags)
	{
		return new DataElementLimitedInputStream(parent, terminatorTags);
	}
	
	/**
	 * 
	 * @throws IOException
	 */
	public final void testReadNextDataElementTag() 
	throws IOException
	{
		DataElementLimitedInputStream in = openInputStream("elements/SequenceTagDelimited_ItemsDelimited.del");
		assertNotNull(in);
		
		DataElementTag tag = null;
		
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1140), tag);

	    byte[] buffy = new byte[16];
	    in.readFully(buffy);
	    assertEquals(0x00, buffy[2]);
	    assertEquals(0x00, buffy[3]);
	    assertEquals( 0x00ff, (0x00ff & buffy[4]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[5]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[6]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[7]) );
	    assertEquals( 0x00fe, (0x00ff & buffy[8]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[9]) );
	    
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1150), tag);
	    
	    in.skip(30);
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1155), tag);
	    
	    in.close();
	}

	/**
	 * 
	 * @throws IOException
	 */
	public final void testReadToSequenceDelimiterTag() 
	throws IOException
	{
		DataElementLimitedInputStream root = openInputStream("elements/SequenceTagDelimited_ItemsDelimited.del");
		DataElementLimitedInputStream in = openInputStream(root, DataElementTag.SEQUENCE_DELIMITATION_TAG);
		assertNotNull(in);
		
		DataElementTag tag = null;
		
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1140), tag);
	    

	    byte[] buffy = new byte[16];
	    in.readFully(buffy);
	    assertEquals(0x00, buffy[2]);
	    assertEquals(0x00, buffy[3]);
	    assertEquals( 0x00ff, (0x00ff & buffy[4]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[5]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[6]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[7]) );
	    assertEquals( 0x00fe, (0x00ff & buffy[8]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[9]) );
	    
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1150), tag);
	    
	    in.skip(30);
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1155), tag);
	    
	    while(in.read() != 0x0000);
	    
	    in.close();
	}

	/**
	 * 
	 * @throws IOException
	 */
	public final void testReadCounts() 
	throws IOException
	{
		DataElementLimitedInputStream root = openInputStream("elements/SequenceTagDelimited_ItemsDelimited.del");
		DataElementLimitedInputStream in = openInputStream(root, DataElementTag.SEQUENCE_DELIMITATION_TAG);
		assertNotNull(in);
		
		DataElementTag tag = null;
		
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1140), tag);
	    assertEquals( 4, in.getBytesRead() );

	    byte[] buffy = new byte[16];
	    in.readFully(buffy);
	    assertEquals( 20, in.getBytesRead() );
	    assertEquals(0x00, buffy[2]);
	    assertEquals(0x00, buffy[3]);
	    assertEquals( 0x00ff, (0x00ff & buffy[4]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[5]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[6]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[7]) );
	    assertEquals( 0x00fe, (0x00ff & buffy[8]) );
	    assertEquals( 0x00ff, (0x00ff & buffy[9]) );
	    
	    tag = in.readNextDataElementTag();
	    assertEquals( 24, in.getBytesRead() );
	    assertEquals(new DataElementTag(0x0008, 0x1150), tag);
	    
	    in.skip(30);
	    assertEquals( 54, in.getBytesRead() );
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1155), tag);
	    
	    while(in.read() != 0x0000);
	    
	    in.close();
	}

	/**
	 * Test that the length limitation really does limit the stream length
	 * @throws IOException
	 */
	public final void testLengthLimitedRead() 
	throws IOException
	{
		DataElementLimitedInputStream root = openInputStream("elements/SequenceTagDelimited_ItemsDelimited.del");
		DataElementLimitedInputStream in = openInputStream(root, 4);
		assertNotNull(in);
		
		DataElementTag tag = null;
		
	    tag = in.readNextDataElementTag();
	    assertEquals(new DataElementTag(0x0008, 0x1140), tag);
	    assertEquals( 4, in.getBytesRead() );
	    
	    assertEquals( -1, in.read() );
	}
	
}
