package gov.va.med.imaging.dicom;

import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.DicomFormatPixelDataException;
import gov.va.med.imaging.dicom.parser.impl.RawDataToPixelTransformer;
import junit.framework.TestCase;

/**
 * 
 * @author       DNS
 *
 */
public class RawDataToPixelTransformerTest 
extends TestCase
{
	protected void setUp() throws Exception
	{
		super.setUp();
	}

	protected void tearDown() throws Exception
	{
		super.tearDown();
	}

	public void testCreateByteLeftShiftedBitMask() 
	throws DicomFormatPixelDataException
	{
		testCreateByteLeftShiftedBitMaskTestPoint(1, 0);
		testCreateByteLeftShiftedBitMaskTestPoint(1, 1);
		testCreateByteLeftShiftedBitMaskTestPoint(1, 2);
		testCreateByteLeftShiftedBitMaskTestPoint(1, 3);
		testCreateByteLeftShiftedBitMaskTestPoint(1, 4);
		testCreateByteLeftShiftedBitMaskTestPoint(1, 5);
		testCreateByteLeftShiftedBitMaskTestPoint(1, 6);
		testCreateByteLeftShiftedBitMaskTestPoint(1, 7);

		testCreateByteLeftShiftedBitMaskTestPoint(2, 0);
		testCreateByteLeftShiftedBitMaskTestPoint(2, 1);
		testCreateByteLeftShiftedBitMaskTestPoint(2, 2);
		testCreateByteLeftShiftedBitMaskTestPoint(2, 3);
		testCreateByteLeftShiftedBitMaskTestPoint(2, 4);
		testCreateByteLeftShiftedBitMaskTestPoint(2, 5);
		testCreateByteLeftShiftedBitMaskTestPoint(2, 6);

		testCreateByteLeftShiftedBitMaskTestPoint(7, 0);
		testCreateByteLeftShiftedBitMaskTestPoint(7, 1);
	}
	
	public void testCreateIntegerSizedLeftShiftedBitMask() 
	throws DicomFormatPixelDataException
	{
		RawDataToPixelTransformer transformer = null;
		int mask = 0;
		
		transformer = new RawDataToPixelTransformer(false);
		
		mask = transformer.createIntegerSizedLeftShiftedBitMask(1, 0);
		assertEquals(0x80000000, mask);
		mask = transformer.createIntegerSizedLeftShiftedBitMask(1, 1);
		assertEquals(0x40000000, mask);
		mask = transformer.createIntegerSizedLeftShiftedBitMask(1, 2);
		assertEquals(0x20000000, mask);
		mask = transformer.createIntegerSizedLeftShiftedBitMask(2, 0);
		assertEquals(0xC0000000, mask);
		mask = transformer.createIntegerSizedLeftShiftedBitMask(2, 1);
		assertEquals(0x60000000, mask);
		mask = transformer.createIntegerSizedLeftShiftedBitMask(2, 2);
		assertEquals(0x30000000, mask);
		
	}
	
	public void testCreateIntegerSizedRightShiftedBitMask() 
	throws DicomFormatPixelDataException
	{
		RawDataToPixelTransformer transformer = null;
		int mask = 0;
		
		transformer = new RawDataToPixelTransformer(false);
		
		mask = transformer.createIntegerSizedRightShiftedBitMask(1, 0);
		assertEquals(0x00000001, mask);
		mask = transformer.createIntegerSizedRightShiftedBitMask(1, 1);
		assertEquals(0x00000002, mask);
		mask = transformer.createIntegerSizedRightShiftedBitMask(2, 0);
		assertEquals(0x00000003, mask);
		mask = transformer.createIntegerSizedRightShiftedBitMask(2, 1);
		assertEquals(0x00000006, mask);
	}
	
	/**
	 * @throws DicomFormatException 
	 * 
	 */
	public void testTransformByteData() 
	throws DicomFormatException
	{
		RawDataToPixelTransformer transformer = null;
		int[] result = null;
		
		transformer = new RawDataToPixelTransformer(false);
		result = transformer.transformRawByteDataToIntPixelArray(
		    	new byte[] {0x01},
		    	1, 1,
		    	8, 8, 7);		
		assertEquals(1, result.length);
		assertEquals(1, result[0]);
	}

	/**
	 * 
	 * @throws DicomFormatException
	 */
	public void testTransformWordDataSinglePixel() 
	throws DicomFormatException
	{
		RawDataToPixelTransformer transformer = null;
		int[] result = null;
		
		transformer = new RawDataToPixelTransformer(true);
		result = transformer.transformRawByteDataToIntPixelArray(
		    	new byte[] {0x01, 0x00},
		    	1, 1,
		    	8, 8, 7);		
		assertEquals(1, result.length);
		assertEquals(1, result[0]);
	}
	
	public void testTransformWordDataTwoPixel() 
	throws DicomFormatException
	{
		RawDataToPixelTransformer transformer = null;
		int[] result = null;
		
		transformer = new RawDataToPixelTransformer(true);
		result = transformer.transformRawByteDataToIntPixelArray(
		    	new byte[] {0x01, 0x02},
		    	1, 2,
		    	8, 8, 7);		
		assertEquals(2, result.length);
		assertEquals(1, result[0]);
		assertEquals(2, result[1]);
	}
	
	public void testTransformWordDataTwoPixelUnevenBoundaries() 
	throws DicomFormatException
	{
		RawDataToPixelTransformer transformer = null;
		int[] result = null;
		
		transformer = new RawDataToPixelTransformer(true);
		result = transformer.transformRawByteDataToIntPixelArray(
		    	new byte[] {0x00, 0x10, 0x02, 0x00},
		    	1, 2,
		    	12, 8, 7);		
		assertEquals(2, result.length);
		assertEquals(1, result[0]);
		assertEquals(2, result[1]);
	}
	
	public void testTransformWordDataFourPixelEvenWordBoundaries() 
	throws DicomFormatException
	{
		RawDataToPixelTransformer transformer = null;
		int[] result = null;
		
		transformer = new RawDataToPixelTransformer(true);
		result = transformer.transformRawByteDataToIntPixelArray(
		    	new byte[] {0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x40, 0x00},
		    	2, 2,
		    	16, 16, 15);		
		assertEquals(4, result.length);
		assertEquals(0x01, result[0]);
		assertEquals(0x02, result[1]);
		assertEquals(0x03, result[2]);
		assertEquals(0x4000, result[3]);
	}
	
	// ===================================================================================
	// Helper Methods
	// ===================================================================================
	
	private void testCreateByteLeftShiftedBitMaskTestPoint(int width, int leftOffset) 
	throws DicomFormatPixelDataException
	{
		RawDataToPixelTransformer transformer = new RawDataToPixelTransformer(false);

		byte mask = transformer.createByteSizedLeftShiftedBitMask(width, leftOffset);
		assertEquals(bitMasks[width][leftOffset], mask);
	}
	
    // a lookup table of pre-built masks ordered by width and left-offset
    // NOTE: left-offset > (8 - width) are not considered valid, contain 0xFF
    // Because we will be doing a lot of bit manipulation, using a lookup table should be faster
    // than calculating the bit mask each time.
    private static byte[][] bitMasks = new byte[][]
    {
    	// width of 0, left-offset of 0 to 7
    	{(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00},
    	
    	// width of 1, left-offset of 0 to 7
    	{(byte)0x80, (byte)0x40, (byte)0x20, (byte)0x10, (byte)0x08, (byte)0x04, (byte)0x02, (byte)0x01},
    	
    	// width of 2, left-offset of 0 to 7
    	{(byte)0xC0, (byte)0x60, (byte)0x30, (byte)0x18, (byte)0x0C, (byte)0x06, (byte)0x03, (byte)0xFF},
    	
    	// width of 3, left-offset of 0 to 7
    	{(byte)0xE0, (byte)0x70, (byte)0x38, (byte)0x1C, (byte)0x0E, (byte)0x07, (byte)0xFF, (byte)0xFF},
    	
    	// width of 4, left-offset of 0 to 7
    	{(byte)0xF0, (byte)0x78, (byte)0x3C, (byte)0x1E, (byte)0x0F, (byte)0xFF, (byte)0xFF, (byte)0xFF},
    	
    	// width of 5, left-offset of 0 to 7
    	{(byte)0xF8, (byte)0x7C, (byte)0x3E, (byte)0x1F, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF},
    	
    	// width of 6, left-offset of 0 to 7
    	{(byte)0xFC, (byte)0x7E, (byte)0x3F, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF},
    	
    	// width of 7, left-offset of 0 to 7
    	{(byte)0xFE, (byte)0x7F, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF},
    	
    	// width of 8, left-offset of 0 to 7
    	{(byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF, (byte)0xFF}
    };
    	
}
