package gov.va.med.imaging.hi5.server;

import gov.va.med.imaging.dicom.DicomServiceCreationException;
import gov.va.med.imaging.dicom.DicomServiceProvider;
import gov.va.med.imaging.dicom.dataset.DataSet;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;
import gov.va.med.imaging.dicom.exceptions.RawPixelInterpretationValuesNotSetException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInterpretationException;
import gov.va.med.imaging.dicom.spi.Part10DataSetParserSPI;
import gov.va.med.imaging.hi5.shared.ArrayConversion;
import gov.va.med.imaging.hi5.shared.Base64Coder;
import gov.va.med.imaging.hi5.shared.HTTPHeaders;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.logging.Logger;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

public class DicomImagePixelsServlet 
extends HttpServlet
{
    private static final long serialVersionUID = 1L;
    private Logger logger = Logger.getLogger("Hi5");
    private final DicomServiceProvider dicomServiceProvider = DicomServiceProvider.getSingleton();

    /**
     * Constructor of the object.
     */
    public DicomImagePixelsServlet()
    {
        super();
    }

    /**
     * Returns information about the servlet, such as author, version, and
     * copyright.
     * 
     * @return String information about this servlet
     */
    public String getServletInfo()
    {
        return "A servlet that will stream the pixels from a DICOM image as a byte stream, optionally BASE64 encoded.";
    }

    /**
     * Initialization of the servlet. <br>
     * 
     * @throws ServletException
     *             if an error occurs
     */
    public void init() 
    throws ServletException
    {
        logger.info("DicomImagePixelsServlet.init()");
    }

    /**
     * Destruction of the servlet. <br>
     */
    public void destroy()
    {
        super.destroy();
        logger.info("DicomImagePixelsServlet.destroy()");
    }

    /**
     * The doGet method of the servlet. <br>
     * 
     * This method is called when a form has its tag value method equals to get.
     * 
     * @param request
     *            the request send by the client to the server
     * @param response
     *            the response send by the server to the client
     * @throws ServletException
     *             if an error occurred
     * @throws IOException
     *             if an error occurred
     */
    public void doGet(HttpServletRequest request, HttpServletResponse response)
    throws ServletException, IOException
    {
        logger.info("DicomImagePixelsServlet.doGet()");
        String acceptType = request.getHeader("Accept-type");
        acceptType = acceptType == null ? null : acceptType.toLowerCase();
        String acceptEncoding = request.getHeader("Accept-encoding");
        acceptEncoding = acceptEncoding == null ? null : acceptEncoding.toLowerCase();

        // boolean gzipEncode = acceptEncoding != null &&
        // acceptEncoding.indexOf("x-gzip") >= 0; // somewhat standard
        boolean base64Encode = acceptEncoding != null
                && acceptEncoding.indexOf(HTTPHeaders.ACCEPT_ENCODING_BASE64) >= 0; // not
        // standard
        String imageID = request.getPathInfo().substring(1);

        logger.info("DicomImagePixelsServlet.doGet(imageID = " + imageID
                + "), " + (base64Encode ? "BASE64" : "binary"));
        try
        {
            if (acceptType != null && acceptType.indexOf(HTTPHeaders.ACCEPT_TYPE_DICOM) >= 0)
                streamImage(response, imageID, true);
            else
                throw new ServletException(
                        "DICOM been berry, berry good to me.");
        } catch (Exception e)
        {
            e.printStackTrace();
            throw new ServletException(e);
        }
    }

    /**
     * 
     * @param out
     * @param imageID
     * @param base64Encode
     * @throws IOException
     * @throws IllegalArgumentException
     * @throws ValueRepresentationInterpretationException
     * @throws ServletException 
     */
    private void streamImage(HttpServletResponse response, String imageID, boolean base64Encode) 
    throws IllegalArgumentException, IOException, ValueRepresentationInterpretationException, ServletException
    {
        response.setContentType("application/octet-stream");

        InputStream inStream = this.getClass().getClassLoader().getResourceAsStream("images/" + imageID);
        Part10DataSetParserSPI parser = null;
        try
        {
            parser = this.dicomServiceProvider.create(Part10DataSetParserSPI.class);
        } 
        catch (DicomServiceCreationException e1)
        {
            throw new ServletException(e1);
        }
        DataSet dicomDataSet = null;

        try
        {
            dicomDataSet = parser.parseInputStreamWithPixels(inStream);
            //Part10DataSetLoader loader = Part10DataSetLoader.load(inStream);
            //dicomDataSet = loader.getDataSet();
        } 
        catch (Exception e)
        {
            e.printStackTrace();
            throw new IllegalArgumentException(e.getMessage());
        } 
        finally
        {
            inStream.close();
        }

        //ByteStreamPump bytePump = ByteStreamPump.getByteStreamPump(TRANSFER_TYPE.FileToNetwork);

        response.setHeader(HTTPHeaders.IMAGE_WIDTH_HEADER, Integer.toString(dicomDataSet.getColumns()));
        response.setHeader(HTTPHeaders.IMAGE_HEIGHT_HEADER, Integer.toString(dicomDataSet.getRows()));
        response.setHeader(HTTPHeaders.IMAGE_BITDEPTH_HEADER, Integer.toString(dicomDataSet.getBitsStored()));

        PrintWriter writer = response.getWriter();

        try
        {
            String encodedPixels = new String( Base64Coder.encode(getDICOMPixels(dicomDataSet)) );
            response.setHeader("Content-encoding", "x-base64");
            writer.print(encodedPixels);
        } catch (DicomFormatException e)
        {
            e.printStackTrace();
        } catch (InvalidVRModeException e)
        {
            e.printStackTrace();
        } catch (InvalidVRException e)
        {
            e.printStackTrace();
        } catch (RawPixelInterpretationValuesNotSetException e)
        {
            e.printStackTrace();
        } finally
        {
            try
            {
                writer.flush();
                writer.close();
            } catch (Exception x)
            {
            }
        }
    }

    /**
     * 
     * @param dicomDataSet
     * @return
     * @throws DicomFormatException
     * @throws IOException
     * @throws InvalidVRModeException
     * @throws InvalidVRException
     * @throws RawPixelInterpretationValuesNotSetException
     */
    private byte[] getDICOMPixels(DataSet dicomDataSet) 
    throws DicomFormatException, IOException,
            InvalidVRModeException, InvalidVRException,
            RawPixelInterpretationValuesNotSetException
    {
        int[] imageFrame = DataSetUtility.getRawPixels(dicomDataSet);
        return ArrayConversion.integerArrayToByteArray(imageFrame);
    }

}
