/**
 * 
 */
package gov.va.med.imaging.hi5.client;

import gov.va.med.imaging.dicom.dataset.DataSet;
import gov.va.med.imaging.hi5.client.DataSourceEvent.DataEventType;
import gov.va.med.imaging.hi5.shared.Base64Coder;
import gov.va.med.imaging.hi5.shared.HTTPHeaders;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import com.google.gwt.core.client.GWT;
import com.google.gwt.http.client.Request;
import com.google.gwt.http.client.RequestBuilder;
import com.google.gwt.http.client.RequestCallback;
import com.google.gwt.http.client.RequestException;
import com.google.gwt.http.client.Response;
import com.google.gwt.http.client.URL;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;

/**
 * @author       DNS
 *
 */
public class DicomDataSetDataSource 
{
    private static final String DICOM_IMAGE_PIXELS_PATH = "/Hi5/dicomImagePixels/";
    private Logger logger = Logger.getLogger("DicomDataSetDataSource");
    
    private final DicomServiceAsync service = GWT.create(DicomService.class);
    
    private String imageURN = null;
    private DataSet currentDataSet = null;
    private byte[] currentPixelData = null;
    private int currentPixelDataWidth = 0;
    private int currentPixelDataHeight = 0;
    private int currentPixelDataDepth = 0;
    private List<DataSourceEventListener> dataSourceEventListeners = new ArrayList<DataSourceEventListener>();
    
    /**
     * @param jsObj
     */
    public DicomDataSetDataSource()
    {
    }

    // ============================================================================================
    // Visible State Management
    // ============================================================================================
    private void setCurrentImageUrn(String imageUrn)
    {
        this.imageURN = imageUrn;
    }
    
    public String getCurrentImageUrn()
    {
        return this.imageURN;
    }

    private void setCurrentDataSet(DataSet dataSet)
    {
        this.currentDataSet = dataSet;
    }
    public DataSet getCurrentDataSet()
    {
        return this.currentDataSet;
    }

    public byte[] getCurrentPixelData()
    {
        return currentPixelData;
    }

    private void setCurrentPixelData(byte[] currentPixelData)
    {
        this.currentPixelData = currentPixelData;
    }

    public int getCurrentPixelDataWidth()
    {
        return currentPixelDataWidth;
    }

    private void setCurrentPixelDataWidth(int currentPixelDataWidth)
    {
        this.currentPixelDataWidth = currentPixelDataWidth;
    }

    public int getCurrentPixelDataHeight()
    {
        return currentPixelDataHeight;
    }

    private void setCurrentPixelDataHeight(int currentPixelDataHeight)
    {
        this.currentPixelDataHeight = currentPixelDataHeight;
    }

    public int getCurrentPixelDataDepth()
    {
        return currentPixelDataDepth;
    }

    public void setCurrentPixelDataDepth(int currentPixelDataDepth)
    {
        this.currentPixelDataDepth = currentPixelDataDepth;
    }

    // ============================================================================================
    // Listener Management
    // ============================================================================================
    public void addDataSourceEventListener(DataSourceEventListener listener)
    {
        if(listener != null)
            dataSourceEventListeners.add(listener);
    }
    public boolean removedataSourceEventListener(DataSourceEventListener listener)
    {
        return dataSourceEventListeners.remove(listener);
    }
    private void notifyDataSourceEventListeners(DataSourceEvent event)
    {
        for(DataSourceEventListener listener : dataSourceEventListeners)
            listener.dataEvent(event);
    }
    
    /**
     * 
     * @param requestId
     * @param request
     * @param response
     */
    public void fetch(String imageUrn)
    {
        logger.info("calling service.getDicomDataSet(" + imageUrn + ")");
        if(imageUrn != null)
        {
            logger.info("before calling getDataSet(" + imageUrn + ")");
            service.getDataSet(imageUrn, new FetchDicomDataSetCallback(imageUrn));
            logger.info("after calling getDataSet(" + imageUrn + ")");
            
            String imagePixelsUrl = "http://" + getDicomPixelsSourceHostAndContext() + imageUrn;
            //String imageDataSetUrl = "http://" + getDicomDataSetSourceHostAndContext() + imageId;
            loadDicomPixels(imagePixelsUrl);
        }
    }

    /**
     * 
     * @param url
     */
    private void loadDicomPixels(String url)
    {
        RequestBuilder builder = new RequestBuilder(RequestBuilder.GET, URL.encode(url));
        builder.setHeader("Accept-type", HTTPHeaders.ACCEPT_TYPE_DICOM);
        builder.setHeader("Accept-encoding", HTTPHeaders.ACCEPT_ENCODING_BASE64);

        try
        {
            builder.setTimeoutMillis(5000);
            builder.sendRequest(null, new GetDicomPixelsCallback());
        } 
        catch (RequestException e)
        {
            logger.severe("Unable to connect to server, " + e.getMessage());
        }
    }

    private String getDicomPixelsSourceHostAndContext()
    {
        return Window.Location.getHost() + DICOM_IMAGE_PIXELS_PATH;
    }

    // =========================================================================================================
    // Metadata Callback Classes
    // =========================================================================================================
    
    /**
     * @author       BECKEC
     *
     */
    abstract class DicomDataSetCallback
    {
        private final String imageUrn; 
        public DicomDataSetCallback(String imageUrn)
        {
            super();
            this.imageUrn = imageUrn;
        }
        public String getImageUrn()
        {
            return this.imageUrn;
        }
    }
    
    class FetchDicomDataSetCallback 
    extends DicomDataSetCallback
    implements AsyncCallback<DataSet>
    {
        public FetchDicomDataSetCallback(String imageUrn)
        {
            super(imageUrn);
        }

        public void onSuccess(DataSet dataSet)
        {
            setCurrentImageUrn(getImageUrn());
            setCurrentDataSet(dataSet);
            notifyDataSourceEventListeners(new DataSourceEvent(DataEventType.DATA_NEW_EVENT));
        }

        public void onFailure(Throwable caught)
        {
            caught.printStackTrace();
            notifyDataSourceEventListeners(new DataSourceEvent(DataEventType.DATA_ERROR_EVENT));
        }
    }
    
    // =========================================================================================================
    // Pixel Callback Classes
    // =========================================================================================================
    
    /**
     * 
     */
    class GetDicomPixelsCallback 
    implements RequestCallback
    {
        public void onResponseReceived(Request request, Response response)
        {
            if (200 == response.getStatusCode())
            {
                String widthParam = response.getHeader(HTTPHeaders.IMAGE_WIDTH_HEADER);
                String heightParam = response.getHeader(HTTPHeaders.IMAGE_HEIGHT_HEADER);
                String bitsPerPixelParam = response.getHeader(HTTPHeaders.IMAGE_BITDEPTH_HEADER);
                int height = Integer.parseInt(heightParam);
                int width = Integer.parseInt(widthParam);
                int bitsPerPixel = Integer.parseInt(bitsPerPixelParam);

                // byte array should be interpreted as an integer array
                byte[] byteData = Base64Coder.decode(response.getText());
                setCurrentPixelData(byteData);
                setCurrentPixelDataDepth(bitsPerPixel);
                setCurrentPixelDataHeight(height);
                setCurrentPixelDataWidth(width);
                
                notifyDataSourceEventListeners(new DataSourceEvent(DataEventType.PIXELS_NEW_EVENT));
            } 
            else
            {
                notifyDataSourceEventListeners(new DataSourceEvent(DataEventType.PIXELS_ERROR_EVENT));
            }
        }

        public void onError(Request request, Throwable exception)
        {
            notifyDataSourceEventListeners(new DataSourceEvent(DataEventType.PIXELS_ERROR_EVENT));
        }
    }
}
