/**
 * 
 */
package gov.va.med.imaging.hi5.client;

import java.util.logging.Logger;

import com.google.gwt.canvas.dom.client.CanvasPixelArray;
import com.google.gwt.touch.client.Point;

/**
 * @author      DNS
 * 
 *         see http://en.wikipedia.org/wiki/Affine_transformation -or-
 *         http://www.w3.org/TR/2dcontext/#transformations
 * 
 *         The formula is : x'=m11 * x + m12 * y + dx y'=m21 * x + m22 * y + dy
 * 
 */
public class AffineTransform
{
    public final static double SIN_RIGHT_ANGLE = 1.0;
    public final static double COS_RIGHT_ANGLE = 0.0;
    
    private Logger logger = Logger.getLogger("AffineTransformation");

    private final ImageStatistics imageStats;
    private final int firstRow;
    private final int lastRow;
    private final int firstColumn;
    private final int lastColumn;

    // the translation and scaling transformation matrix, organized as follows:
    // m11, m12
    // m21, m22
    // dx,  dy
    private AffineTransformationMatrix translationMatrix = new AffineTransformationMatrix();
    private AffineTransformationMatrix rotationMatrix = new AffineTransformationMatrix();

    // ====================================================================================
    // Constructor (note that instances of this class are image specific
    // ====================================================================================
    public AffineTransform(ImageStatistics imageStats)
    {
        this.imageStats = imageStats;
        this.firstRow = (int) -(imageStats.getHeight() / 2.0);
        this.lastRow = (int) (imageStats.getHeight() / 2.0 + (imageStats.getHeight() / 2.0 == 0.0 ? 0.0 : 1.0));
        this.firstColumn = (int) -(imageStats.getWidth() / 2.0);
        this.lastColumn = (int) (imageStats.getWidth() / 2.0 + (imageStats.getWidth() / 2.0 > 0.1 ? 1.0 : 0.0));
    }

    public ImageStatistics getImageStats()
    {
        return imageStats;
    }

    public int getFirstRow()
    {
        return firstRow;
    }

    public int getLastRow()
    {
        return lastRow;
    }

    public int getFirstColumn()
    {
        return firstColumn;
    }

    public int getLastColumn()
    {
        return lastColumn;
    }

    // ====================================================================================
    // Generic Transformation methods
    // ====================================================================================
    
    /**
     * 
     */
    public void revert()
    {
        logger.info("revert() start " + toString());
        this.translationMatrix.revert();
        this.rotationMatrix.revert();
        
        logger.info("revert() complete " + toString());
    }
    
    // ====================================================================================
    // Translation methods
    // ====================================================================================
    /**
     * 
     * @param dx
     * @param dy
     */
    public final void setTranslate(double dx, double dy)
    {
        this.translationMatrix.setDX(dx);
        this.translationMatrix.setDY(dy);
    }
    
    public final void translate(double dx, double dy)
    {
        this.translationMatrix.addDX(dx);
        this.translationMatrix.addDY(dy);
    }
    
    public final void setTranslateToOrigin()
    {
        this.translationMatrix.setDX(0.0);
        this.translationMatrix.setDY(0.0);
    }

    public final void flip()
    {
        this.translationMatrix.multiplyM22(-1.0);
    }
    
    public final void mirror()
    {
        this.translationMatrix.multiplyM11(-1.0);
    }
    
    // ====================================================================================
    // Scale method
    // ====================================================================================
    public final void setScale(double s)
    {
        setScale(s, s);
    }
    public final void scale(double s)
    {
        scale(s, s);
    }
    
    /**
     * 
     * @param dx
     * @param dy
     */
    public final void setScale(double sx, double sy)
    {
        this.translationMatrix.setM11(sx);
        this.translationMatrix.setM22(sy);
    }
    public final void scale(double sx, double sy)
    {
        this.translationMatrix.multiplyM11(sx);
        this.translationMatrix.multiplyM22(sy);
    }
    
    // ====================================================================================
    // Rotation method
    // ====================================================================================
    private double theta = 0.0;
    
    public double getRotationAngle()
    {
        return theta;
    }

    public final void rotate(double theta)
    {
        this.theta = this.theta + theta;
        this.rotate();
    }

    public final void setRotate(double theta)
    {
        this.theta = theta;
        this.rotate();
    }
    
    private final void rotate()
    {
        AffineTransformationMatrix atm = new AffineTransformationMatrix(
                Math.cos(getRotationAngle()), -Math.sin(getRotationAngle()),
                Math.sin(getRotationAngle()), Math.cos(getRotationAngle()),
                0.0, 0.0);
        this.rotationMatrix.set(atm);
    }
    
    /**
     * 
     * @return
     */
    public boolean isIdentityTransform()
    {
        return this.translationMatrix.equals(IDENTITY_TRANSFORM) && this.rotationMatrix.equals(IDENTITY_TRANSFORM);
    }

    /**
     * Transform an entire image given the current transformation matrix. Does
     * nothing if the current transformation matrix is the identity matrix.
     * 
     * @param cpa
     * @param height
     * @param width
     * @param bitDepth
     * 
     *            The formula is : 
     *            x' = m11 * x + m12 * y + dx 
     *            y' = m21 * x + m22 * y + dy
     */
    public void execute(CanvasPixelArray cpa, double height, double width,
            int bitDepth)
    {
        String msg = "";

        msg = "AffineTransform is: " + toString();

        if (!isIdentityTransform())
        {
            int[] buffer = new int[cpa.getLength()];

            // the matrix math used to express the affine transform assumes that
            // the
            // pixel array (a matrix) has its origin in the center of the array,
            // whereas the
            // "real" origin is at the top left. That is why the following loops
            // start at
            // a negative number and are transformed into the "real" array
            // offsets.

            // iterate in image matrix terms
            for (int y = getFirstRow(); y < getLastRow(); ++y)
                for (int x = getFirstColumn(); x < getLastColumn(); ++x)
                {
                    Point p = new Point(x, y);
                    Point p1 = transformPoint(p);

                    int originPixelOffset = (int) (((y + (-getFirstRow())) * getImageStats()
                            .getWidth()) + (x + (-getFirstColumn())));
                    int destinationPixelOffset = (int) (((p1.getY() + (-getFirstRow())) * imageStats
                            .getWidth()) + (p1.getX() + (-getFirstColumn())));

                    // msg = msg + "    "
                    // + x + "," + y + "(" + originPixelOffset + ")"
                    // + x1 + "," + y1 + "(" + destinationPixelOffset + ")";

                    buffer[4 * destinationPixelOffset + 0] = cpa.get(4 * originPixelOffset + 0);
                    buffer[4 * destinationPixelOffset + 1] = cpa.get(4 * originPixelOffset + 1);
                    buffer[4 * destinationPixelOffset + 2] = cpa.get(4 * originPixelOffset + 2);
                    buffer[4 * destinationPixelOffset + 3] = cpa.get(4 * originPixelOffset + 3);
                }

            for (int index = 0; index < cpa.getLength(); ++index)
                cpa.set(index, buffer[index]);
        } else
            msg = "AffineTransform is identity transform, skipping.";

        logger.info(msg);
    }

    /**
     * Given an offset in the origin pixel array, return a point in a coordinate
     * system with the origin at the center of the image.
     * 
     * @param originOffset
     * @return
     */
    public Point originPoint(int originOffset)
    {
        int y = originOffset / getImageStats().getWidth();
        int x = originOffset % getImageStats().getWidth();
        
        y = y + getFirstRow();          // note: getFirstRow() returns a negative number
        x = x + getFirstColumn();       // note: getFirstColumn() returns a negative number
        return new Point(x, y);        
    }
    
    /**
     * Given a point in a coordinate system with the origin at the center of the
     * image, return the transformed point in the same coordinate system.
     * 
     * @param p
     * @return
     */
    public Point transformPoint(Point p)
    {
        Point translated = translatePoint(p);
        Point rotated = rotatePoint(translated);
        
        return rotated;
    }

    public Point translatePoint(Point p)
    {
        double x1 = ((this.translationMatrix.getM11() * p.getX()) + (this.translationMatrix.getM12() * p.getY()) + this.translationMatrix.getDX());
        double y1 = ((this.translationMatrix.getM21() * p.getX()) + (this.translationMatrix.getM22() * p.getY()) + this.translationMatrix.getDY());
        return new Point(x1, y1);
    }

    public Point rotatePoint(Point p)
    {
        double x1 = (this.rotationMatrix.getM11() * p.getX()) + (this.rotationMatrix.getM12() * p.getY());
        double y1 = (this.rotationMatrix.getM21() * p.getX()) + (this.rotationMatrix.getM22() * p.getY());
        return new Point(x1, y1);
    }
    
    /**
     * 
     * @param p
     * @return
     */
    public int originCPAPixelOffset(Point p)
    {
        return RGBPixel.PIXEL_STORAGE_SIZE * (int) (
            ((p.getY() + (-getFirstRow())) * getImageStats().getWidth()) 
            + (p.getX() + (-getFirstColumn()))
        );
    }

    /**
     * 
     * @param p
     * @return
     */
    public int destinationCPAPixelOffset(Point p)
    {
        return RGBPixel.PIXEL_STORAGE_SIZE * (int) (
            ((p.getY() + (-getFirstRow())) * getImageStats().getWidth()) 
            + (p.getX() + (-getFirstColumn()))
        );
    }

    @Override
    public String toString()
    {
        return this.translationMatrix.toString() + this.rotationMatrix.toString();
    }
    
    /**
     * 
     * @author        DNS
     *
     */
    class AffineTransformationMatrix
    {
        private double[][] matrix = new double[3][2];
        
        public AffineTransformationMatrix()
        {
            revert();
        }
        
        AffineTransformationMatrix(double[][] elements)
        {
            matrix[0][0] = elements[0][0];
            matrix[0][1] = elements[0][1];
            matrix[1][0] = elements[1][0];
            matrix[1][1] = elements[1][1];
            matrix[2][0] = elements[2][0];
            matrix[2][1] = elements[2][1];
        }
        
        AffineTransformationMatrix(double[] elements)
        {
            matrix[0][0] = elements[0];
            matrix[0][1] = elements[1];
            matrix[1][0] = elements[2];
            matrix[1][1] = elements[3];
            matrix[2][0] = elements[4];
            matrix[2][1] = elements[5];
        }
        
        AffineTransformationMatrix(double m11, double m12, double m21, double m22, double dx, double dy )
        {
            matrix[0][0] = m11;
            matrix[0][1] = m12;
            matrix[1][0] = m21;
            matrix[1][1] = m22;
            matrix[2][0] = dx;
            matrix[2][1] = dy;
        }
        
        public void revert()
        {
            matrix[0][0] = 1.0;
            matrix[0][1] = 0.0;
            matrix[1][0] = 0.0;
            matrix[1][1] = 1.0;
            matrix[2][0] = 0.0;
            matrix[2][1] = 0.0;
        }

        double getM11(){return matrix[0][0];}
        double getM12(){return matrix[0][1];}
        double getM21(){return matrix[1][0];}
        double getM22(){return matrix[1][1];}
        double getDX(){return matrix[2][0];}
        double getDY(){return matrix[2][1];}
        
        void setM11(double v){matrix[0][0] = v;}
        void setM12(double v){matrix[0][1] = v;}
        void setM21(double v){matrix[1][0] = v;}
        void setM22(double v){matrix[1][1] = v;}
        void setDX(double v){matrix[2][0] = v;}
        void setDY(double v){matrix[2][1] = v;}
        
        void addM11(double v){matrix[0][0] = matrix[0][0] + v;}
        void addM12(double v){matrix[0][1] = matrix[0][1] + v;}
        void addM21(double v){matrix[1][0] = matrix[1][0] + v;}
        void addM22(double v){matrix[1][1] = matrix[1][1] + v;}
        void addDX(double v){matrix[2][0] = matrix[2][0] + v;}
        void addDY(double v){matrix[2][1] = matrix[2][1] + v;}
        
        void multiplyM11(double v){matrix[0][0] = matrix[0][0] * v;}
        void multiplyM12(double v){matrix[0][1] = matrix[0][1] * v;}
        void multiplyM21(double v){matrix[1][0] = matrix[1][0] * v;}
        void multiplyM22(double v){matrix[1][1] = matrix[1][1] * v;}
        void multiplyDX(double v){matrix[2][0] = matrix[2][0] * v;}
        void multiplyDY(double v){matrix[2][1] = matrix[2][1] * v;}
        
        void add(AffineTransformationMatrix m2)
        {
            matrix[0][0] = matrix[0][0] + m2.matrix[0][0];
            matrix[0][1] = matrix[0][1] + m2.matrix[0][1];
            matrix[1][0] = matrix[1][0] + m2.matrix[1][0];
            matrix[1][1] = matrix[1][1] + m2.matrix[1][1];
            matrix[2][0] = matrix[2][0] + m2.matrix[2][0];
            matrix[2][1] = matrix[2][1] + m2.matrix[2][1];
        }
        
        void multiply(AffineTransformationMatrix m2)
        {
            matrix[0][0] = matrix[0][0] * m2.matrix[0][0];
            matrix[0][1] = matrix[0][1] * m2.matrix[0][1];
            matrix[1][0] = matrix[1][0] * m2.matrix[1][0];
            matrix[1][1] = matrix[1][1] * m2.matrix[1][1];
            matrix[2][0] = matrix[2][0] * m2.matrix[2][0];
            matrix[2][1] = matrix[2][1] * m2.matrix[2][1];
        }
        
        void set(AffineTransformationMatrix m2)
        {
            matrix[0][0] = m2.matrix[0][0];
            matrix[0][1] = m2.matrix[0][1];
            matrix[1][0] = m2.matrix[1][0];
            matrix[1][1] = m2.matrix[1][1];
            matrix[2][0] = m2.matrix[2][0];
            matrix[2][1] = m2.matrix[2][1];
        }
        
        @Override
        public String toString()
        {
            return 
            "m11 = " + this.matrix[0][0] 
            + ", m12 = " + this.matrix[0][1]
            + ", m21 = " + this.matrix[1][0] 
            + ", m22 = " + this.matrix[1][1] 
            + "dx = " + this.matrix[2][0]
            + "dy = " + this.matrix[2][1];
        }
        
        @Override
        public boolean equals(Object obj)
        {
            if(obj instanceof AffineTransformationMatrix)
            {
               AffineTransformationMatrix that = (AffineTransformationMatrix)obj;
               return 
                   this.getM11() == that.getM11()
                   && this.getM12() == that.getM12()
                   && this.getM21() == that.getM21()
                   && this.getM22() == that.getM22()
                   && this.getDX() == that.getDX()
                   && this.getDY() == that.getDY();
            }
            else if(obj instanceof double[][])
            {
                double[][] that = (double[][])obj;
                return 
                    this.getM11() == that[0][0]
                    && this.getM12() == that[0][1]
                    && this.getM21() == that[1][0]
                    && this.getM22() == that[1][1]
                    && this.getDX() == that[2][0]
                    && this.getDY() == that[2][1];
            }
            else if(obj instanceof double[])
            {
                double[] that = (double[])obj;
                return 
                    this.getM11() == that[0]
                    && this.getM12() == that[1]
                    && this.getM21() == that[2]
                    && this.getM22() == that[3]
                    && this.getDX() == that[4]
                    && this.getDY() == that[5];
            }
            return false;
        }
    }
    
    public static double[][] IDENTITY_TRANSFORM = new double[][]
    {
       { 1.0, 0.0 },
       { 0.0, 1.0 },
       { 0.0, 0.0 } 
    };
}
