package gov.va.med.imaging.dicom.parser.io;

import gov.va.med.imaging.dicom.dataset.ValueRepresentation;
import gov.va.med.imaging.dicom.dataset.elements.DataElement;
import gov.va.med.imaging.dicom.dataset.elements.DataElementTag;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.DicomFormatPixelDataException;
import gov.va.med.imaging.dicom.exceptions.IncompatibleValueLengthField;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;
import gov.va.med.imaging.dicom.parser.impl.DataElementFactory;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This class reads from the given stream, returning a DataElement for 
 * each call to getNextDataElement()
 * 
 * @author       DNS
 *
 */
public class DataElementReader
{
	private final DataElementLimitedInputStream inputStream;
	private final DataElementFactory dataElementFactory;
	private Logger logger = Logger.getLogger(this.getClass().getName());
	
	
	/**
	 * A package level constructor to be used to creater readers of child data sets.
	 * 
	 * @param inputStream
	 * @param dataElementFactory
	 */
	public DataElementReader(DataElementLimitedInputStream inputStream, DataElementFactory dataElementFactory)
	{
		if(inputStream == null || dataElementFactory == null)
			throw new IllegalArgumentException(
					"Input Stream is " + (inputStream == null ? "NULL" : "not null") + 
					", Data Element Factory is " + (dataElementFactory == null ? "NULL," : "not null.") );
		this.inputStream = inputStream;
		this.dataElementFactory = dataElementFactory;
	}
	
	/**
     * @return the inputStream
     */
    public DataElementLimitedInputStream getInputStream()
    {
    	return inputStream;
    }

	/**
     * @return the elementFactory
     */
    public DataElementFactory getDataElementFactory()
    {
    	return dataElementFactory;
    }

    /**
     * 
     * @return
     * @throws IOException
     * @throws InvalidVRModeException
     * @throws InvalidVRException
     * @throws IncompatibleValueLengthField 
     */
	public DataElement<?> readNextDataElement() 
	throws 
		IOException, InvalidVRModeException, InvalidVRException, IncompatibleValueLengthField, 
		DicomFormatException
	{
		ValueRepresentation vr = null;
		long valueLength = 0;		// a 16 or 32 bit unsigned, stored as long
		byte[] value = null;
		DataElementTag dataElementTag = inputStream.readNextDataElementTag();
		
		boolean readingPixelData = DataElementTag.PIXEL_DATA_TAG.equals(dataElementTag);
		
		logger.log(Level.FINER, 
			"Reading element tag '" + (dataElementTag == null ? "NULL, (EOF)" : dataElementTag.toString()) + "'." );
		if(dataElementTag == null)		// NULL indicates the end of file
			return null;
		
		// If we are in explicit VR mode then the next piece of data MUST be a VR.
		// EXCEPT (from the specification Section 5 Page 40):
		// There are three special SQ related Data Elements that are not ruled by the VR 
		// encoding rules conveyed by the Transfer Syntax. They shall be encoded as Implicit VR. 
		// These special Data Elements are Item (FFFE,E000), Item Delimitation Item (FFFE,E00D), 
		// and Sequence Delimitation Item (FFFE,E0DD). However, the Data Set within the Value Field 
		// of the Data Element Item (FFFE,E000) shall be encoded according to the rules conveyed by 
		// the Transfer Syntax.
		// The special sequence elements always have no VR and always have a 4 byte length immediately
		// following.
		if( dataElementTag.equals(DataElementTag.SEQUENCE_DELIMITATION_TAG) ||
			dataElementTag.equals(DataElementTag.ITEM_DELIMITATION_TAG) ||
			dataElementTag.equals(DataElementTag.SEQUENCE_ITEM_TAG) )
		{
			logger.log(Level.FINER, "Sequence delimitation element " + dataElementTag.toString());
			valueLength = inputStream.readFourBytesIntoLong();		// should be 0x00000000 for SEQUENCE_DELIMITATION_TAG and ITEM_DELIMITATION_TAG 
		}
		
		else if( getDataElementFactory().isExplicitVR() )
		{
			vr = inputStream.readValueRepresentation();
			logger.log(Level.FINER, "Explicit VR - " + vr.toString());
			
			// For VRs of OB, OW, OF, SQ, UN and UT the 16 bits following the two character VR Field
			// the Value Length Field is a 32-bit unsigned integer.
			// If the Value Field has an Explicit Length, then the Value Length Field shall contain a value 
			// equal to the length (in bytes) of the Value Field. Otherwise, the Value Field has an 
			// Undefined Length and a Sequence Delimitation Item marks the end of the Value Field.
			if( vr == ValueRepresentation.OB || vr == ValueRepresentation.OW || vr == ValueRepresentation.OF || 
				vr == ValueRepresentation.SQ || vr == ValueRepresentation.UN || vr == ValueRepresentation.UT )
			{
				logger.log(Level.FINER, "Explicit VR, discarding two bytes between VR and length.");
    			inputStream.skip(2);
				logger.log(Level.FINER, "Explicit VR, reading 4 byte length.");
    			valueLength = inputStream.readFourBytesIntoLong();
    			
    			// Note: VRs of UT may not have an Undefined Length, ie. a Value Length of FFFFFFFFH.
    			if(vr == ValueRepresentation.UT && valueLength == DataElement.INDETERMINATE_LENGTH)
    				throw new IncompatibleValueLengthField(vr, valueLength);
			}
			else
			{
				int valueLengthLength = vr.getExpectedDataLengthFieldLength();
				logger.log(Level.FINER, "Explicit VR, reading value length field, " + valueLengthLength + " bytes.");
				valueLength = valueLengthLength == 2 ?
					inputStream.readTwoBytesIntoLong() : 
					inputStream.readFourBytesIntoLong();
			}
		}
		
		// When using the Implicit VR structure.
		// If the Value Field has an Explicit Length then the Value Length Field shall contain a value 
		// equal to the length (in bytes) of the Value Field. Otherwise, the Value Field has an Undefined 
		// Length and a Sequence Delimitation Item marks the end of the Value Field.
		else
		{
			logger.log(Level.FINER, "Implicit VR, reading 4 byte length.");
			valueLength = inputStream.readFourBytesIntoLong();
		}
		
		// at this point the data element value representation is known, either explicitly stated
		// or implicitly determined
		ValueRepresentation dataElementVR = 
			vr == null ?  
			getDataElementFactory().getImplicitVR(dataElementTag) :
			vr;
		
		logger.log(Level.FINE, 
			"element tag " + dataElementTag.toString() + ", " + 
			(vr == null ? "implicit-" : "explicit-") + "-" + dataElementVR + " " + 
			valueLength + " bytes.");
		
		// if the data element is a sequence (either explicitly stated or implicitly determined)
		if( dataElementVR == ValueRepresentation.SQ ) 
		{
			if( valueLength == DataElement.INDETERMINATE_LENGTH)
			{
				value = inputStream.readUntilElementTag(DataElementTag.ITEM_DELIMITATION_TAG, DataElementTag.SEQUENCE_DELIMITATION_TAG);
			}
			else if( valueLength == 0)
			{
				value = null;
			}
			else
			{
				// always read an even number of bytes
				value = new byte[(int)valueLength + ((int)valueLength % 2)];		// safely downcast because its an unsigned number
				inputStream.readFully(value);			// read the entire tag element
			}
			
			return getDataElementFactory().isExplicitVR() ? 
					getDataElementFactory().createDataElement(dataElementTag, vr, valueLength, value) :
					getDataElementFactory().createDataElement(dataElementTag, null, valueLength, value);
		}
		// See PS 3.5-2007 Page 67 for how OB, OW, and OF are interpreted
		// In addition to SQ, the following VRs may also have an indeterminate length.
		// ValueRepresentation.OB, ValueRepresentation.OW, ValueRepresentation.OF,
		// ValueRepresentation.UN, ValueRepresentation.UT
		// A valueLength of 0xFFFFFFFF means an indeterminate length, read until the delimiter element
		if( readingPixelData )
		{
			if( dataElementVR == ValueRepresentation.OB || 
				dataElementVR == ValueRepresentation.OW ||
				dataElementVR == ValueRepresentation.OF )
			{
				if( valueLength == DataElement.INDETERMINATE_LENGTH)
				{
					value = inputStream.readUntilElementTag(DataElementTag.ITEM_DELIMITATION_TAG, DataElementTag.SEQUENCE_DELIMITATION_TAG);
				}
				else if( valueLength == 0)
				{
					value = null;
				}
				else
				{
					// always read an even number of bytes
					value = new byte[(int)valueLength + ((int)valueLength % 2)];		// safely downcast because its an unsigned number
					inputStream.readFully(value);			// read the entire tag element
				}
				
				return getDataElementFactory().createDataElement(
						dataElementTag, 
						(getDataElementFactory().isExplicitVR() ? vr : null), 
						valueLength, 
						value);
			}
			else
			{
				throw new DicomFormatPixelDataException("The pixel data was found in a '" + dataElementVR + " value representation.");
			}
		}
		else
		{
			// should only be ValueRepresentation.UN and ValueRepresentation.UT
			if( valueLength == DataElement.INDETERMINATE_LENGTH)
			{
				value = inputStream.readUntilElementTag(DataElementTag.ITEM_DELIMITATION_TAG);
			}
			else if( valueLength == 0)
			{
				value = null;
			}
			else
			{
				value = new byte[(int)valueLength];		// safely downcast because its an unsigned number
				inputStream.readFully(value);			// read the entire tag element
			}
			
			return getDataElementFactory().isExplicitVR() ? 
				getDataElementFactory().createDataElement(dataElementTag, vr, valueLength, value) :
				getDataElementFactory().createDataElement(dataElementTag, null, valueLength, value);
		}
	}
}
