/**
 * Package: MAG - VistA Imaging
 * WARNING: Per VHA Directive 2004-038, this routine should not be modified.
 * Date Created: Aug 11, 2008
 * Site Name:  Washington OI Field Office, Silver Spring, MD
 * @author DNS
 * @version 1.0
 *
 * ----------------------------------------------------------------
 * Property of the US Government.
 * No permission to copy or redistribute this software is given.
 * Use of unreleased versions of this software requires the user
 * to execute a written test agreement with the VistA Imaging
 * Development Office of the Department of Veterans Affairs,
 * telephone (DNS
 * 
 * The Food and Drug Administration classifies this software as
 * a Class II medical device.  As such, it may not be changed
 * in any way.  Modifications to this software may result in an
 * adulterated medical device under 21CFR820, the use of which
 * is considered to be a violation of US Federal Statutes.
 * ----------------------------------------------------------------
 */
package gov.va.med.imaging.dicom.parser.io;

import gov.va.med.imaging.dicom.dataset.TransferSyntaxUid;
import gov.va.med.imaging.dicom.dataset.ValueRepresentation;
import gov.va.med.imaging.dicom.dataset.elements.DataElementTag;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.OddLengthWithoutZeroFillException;
import gov.va.med.imaging.dicom.exceptions.UnexpectedEOFException;
import gov.va.med.imaging.dicom.parser.impl.TransferSyntaxUidUtility;

import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * An aggregation of DataInputStream that reads up to the maximum bytes
 * given or until the given byte pattern (as a DataElementTag) is encountered.
 * The terminating data element tags are NOT returned.
 * DataElementTags must be read using the readNextDataElementTag() method for this
 * class to detect the terminating data element tags.
 * 
 * Notes:
 * If the InputStream passed to the constructor is an instance of DataInputStream then this
 * instance will not create a new wrapped instance of DataInputStream, it will simply use the
 * passed DataInputStream.  Otherwise this instance will create an instance of DataInputStream
 * around the supplied InputStream.
 * The close() method WILL CLOSE the wrapped InputStream instance.
 * This class is specifically intended to allow multiple streams in a stack arrangement
 * over one "real" DataInputStream.  Only the uppermost instance may be "active" at a time, that is reading
 * from the stream.  The stacked instances are intended to be used for child data sets.
 * 
 * @author DNS
 */
public class DataElementLimitedInputStream
extends InputStream
{
	private final long maxLength;
	private final TransferSyntaxUid transferSyntax;
	private final DataElementTag[] terminatingTags;
	private final DataInputStream rootStream;		// not null when this instance is directly wrapping the real input stream
	private final DataElementLimitedInputStream parentStream;	// not null when this instance is a progeny of the root input stream
	
	private long bytesRead = 0L;
	private long bytesReadAtMark = 0L;
	
	private Logger logger = Logger.getLogger(this.getClass().getName());
	
	/**
	 * Read until the "real" EOF.  This constructor just provides semantic equivalence
	 * for the root data set and the child data sets.
	 * 
	 * @param in
	 */
	public DataElementLimitedInputStream(DataInputStream root, TransferSyntaxUid transferSyntax)
	{
		if(root == null)
			throw new IllegalArgumentException("Root DataInputStream cannot be null.");
		this.rootStream = root;
		this.parentStream = null;
		this.maxLength = Long.MAX_VALUE;
		this.transferSyntax = transferSyntax;
		this.terminatingTags = null;
	}
	
	/**
	 * Read from an input stream until the maximum length is reached.
	 * 
	 * @param in
	 */
	public DataElementLimitedInputStream(DataElementLimitedInputStream parent, long maxLength)
	{
		if(parent == null)
			throw new IllegalArgumentException("Parent DataElementLimitedInputStream cannot be null.");
		this.parentStream = parent;
		this.rootStream = null;
		this.maxLength = maxLength;
		this.terminatingTags = null;
		this.transferSyntax = null;
	}
	
	/**
	 * Read from an input stream until one of the data element tags in the given list is encountered.
	 * 
	 * @param in
	 * @param transferSyntax
	 * @param terminatingTags
	 */
	public DataElementLimitedInputStream(DataElementLimitedInputStream parent, DataElementTag... terminatingTags)
	{
		if(parent == null)
			throw new IllegalArgumentException("Parent DataElementLimitedInputStream cannot be null.");
		this.parentStream = parent;
		this.rootStream = null;
		this.transferSyntax = null;
		this.terminatingTags = terminatingTags;
		this.maxLength = Long.MAX_VALUE;
	}

	public boolean isRootDataSetStream()
	{
		return this.rootStream != null;
	}
	
	public long getMaxLength()
    {
    	return maxLength;
    }

	public TransferSyntaxUid getTransferSyntax()
    {
    	return parentStream == null ? transferSyntax : parentStream.getTransferSyntax();
    }

	public DataElementTag[] getTerminatingTags()
    {
    	return terminatingTags;
    }

	public long getBytesRead()
    {
    	return bytesRead;
    }

	public long getBytesReadAtMark()
    {
    	return bytesReadAtMark;
    }
	
	/**
	 * If this instance was created with a length limit then this
	 * will return the number of bytes remaining until that limit
	 * is reached.
	 * If this instance was created with a delimiter tag then this
	 * will return Long.MAX_VALUE always.
	 * 
	 * @return
	 */
	public long getBytesRemaining()
	{
		if(this.terminatingTags != null)
			return Long.MAX_VALUE;
		
		return this.maxLength - this.bytesRead;
	}

	// ============================================================================
	// Mark and reset support
	// ============================================================================
	@Override
	public boolean markSupported()
	{
		return isRootDataSetStream() ? rootStream.markSupported() : parentStream.markSupported();
	}
	
	@Override
    public synchronized void mark(int readlimit)
    {
		if(isRootDataSetStream())
			rootStream.mark(readlimit);
		else
			parentStream.mark(readlimit);
		
		bytesReadAtMark = bytesRead;
    }

	// ============================================================================
	// InputStream Overrides
	// ============================================================================
	@Override
    public int read() 
	throws IOException
    {
		if(bytesRead >= maxLength)
			return -1;
		
	    int charRead = isRootDataSetStream() ? rootStream.read() : parentStream.read();
		++bytesRead;
		return charRead;
    }
	
	@Override
	public final int read(byte[] dest)
    throws IOException
	{
		return this.read(dest, 0, dest.length);
	}
	
	@Override
	public final int read(byte[] dest, int off, int len)
    throws IOException
    {
		int readBytes = isRootDataSetStream() ? rootStream.read(dest, off, len) : parentStream.read(dest, off, len);
		this.bytesRead += readBytes;
		
		return readBytes;
    }

	@Override
    public synchronized void reset() 
	throws IOException
    {
		if(isRootDataSetStream())
			rootStream.reset();
		else
			parentStream.reset();
	    bytesRead = bytesReadAtMark;
    }

	@Override
    public long skip(long n) throws IOException
    {
		long bytesSkipped = isRootDataSetStream() ? rootStream.skip(n) : parentStream.skip(n);
		this.bytesRead += bytesSkipped;
	    return bytesSkipped;
    }
	
	@Override
    public int available() 
	throws IOException
    {
		long avail = getBytesRemaining();
		avail = avail > Integer.MAX_VALUE ? Integer.MAX_VALUE : avail;
		return Math.min( (int)avail, isRootDataSetStream() ? rootStream.available() : parentStream.available() );
    }
    
	@Override
    public void close()
    throws IOException
    {
		if(isRootDataSetStream())
			rootStream.close();
    }
	
	// ============================================================================
	// DataInputStream-like methods
	// Note that a number of needed DataInputStream methods are declared final
	// so this class definition cannot derive from that class definition.
	// ============================================================================
	public final void readFully(byte[] dest)
    throws IOException
	{
		this.readFully(dest, 0, dest.length);
	}
	
	public final void readFully(byte[] dest, int off, int len)
    throws IOException
    {
		if(isRootDataSetStream())
			rootStream.readFully(dest, off, len);
		else
			parentStream.readFully(dest, off, len);
		this.bytesRead += len;
    }
	
    // ====================================================================
    // DICOM Data Element specific methods
    // ====================================================================

	/**
	 * Reads the next four bytes and makes a DataElementTag from them.
	 * If the DataElementTag read matches any of the DataElementTag in the
	 * terminating tags list or if the maximum number of bytes has been read
	 * then this method returns null.
	 * 
	 * @return null when no more data remains to be read
	 * @throws IOException
	 */
	public DataElementTag readNextDataElementTag()
	throws IOException
	{
		int groupNumber = -1;
		
		if(available() <= 0)
			return null;
		
		// read the group number, if the EOF is reached when trying to read the group number
		// then return null
    	try{groupNumber = readTwoBytesIntoInt();}
    	catch(EOFException eofX){return null;}		// break when EOF is reached
    	
    	// read the element number, if the EOF is reached then an exception occurs 
    	// because the group/element is not complete. 
    	int elementNumber = readTwoBytesIntoInt();

    	
    	DataElementTag dataElementTag = new DataElementTag(groupNumber, elementNumber);
    	
    	// if the data element read is on the list of terminator tags
    	// then return null to indicate no more elements available
    	if(isTerminatorTag(dataElementTag))
    		return null;
    	
    	return dataElementTag;
	}
	
	/**
	 * Is the given data element tag in the array of terminator tags.
	 * 
	 * @param dataElementTag
	 * @return
	 */
	private boolean isTerminatorTag(DataElementTag dataElementTag)
    {
		if(terminatingTags == null)
			return false;
		
		for(DataElementTag terminatingTag : terminatingTags)
			if(terminatingTag.equals(dataElementTag))
				return true;
		
	    return false;
    }

	/**
	 * @return
	 * @throws IOException
	 */
	public int readTwoBytesIntoInt() 
	throws IOException
	{
		byte[] buffy = new byte[2];
		this.readFully(buffy);
		return TransferSyntaxUidUtility.makeUnsignedIntFrom2Bytes(getTransferSyntax().isLittleEndian(), buffy);
	}
	
	/**
	 * Read bytes into a buffer until the any element tag in the list is found.
	 * The element tag that stops the read is NOT read, he stream is positioned at
	 * that element when the method returns.
	 * 
     * @return
     * @throws IOException
     */
	public byte[] readUntilElementTag(DataElementTag... delimiterTags) 
    throws IOException, DicomFormatException
    {
		if(logger.isLoggable(Level.FINE))
		{
			StringBuffer sb = new StringBuffer();
			for(DataElementTag delimiterTag : delimiterTags)
				sb.append(delimiterTag.toString() + ", ");
			logger.log(Level.FINE, "Reading delimited value until any of tags [" + sb.toString() + "]");
		}
		
	    byte[] value;
	    List<Byte> temp = new ArrayList<Byte>(1024);
	    byte[] buffy = new byte[2];		// safely read two bytes at a time
	    
	    for(DataElementTag currentTag = peekNextDataElementTag();
	    	currentTag != null && !isAnyDataElementTag(currentTag, delimiterTags);
	    	currentTag = peekNextDataElementTag())
	    {
	    	logger.log(Level.FINER, "Current [" + currentTag.toString() + "] is not an end delimiter tag.");
	    	if(isRootDataSetStream())
	    	{
		    	if( rootStream.read(buffy) != buffy.length)
		    		throw new UnexpectedEOFException("End of file encountered reading delimited value.");
	    	}
		    else
		    {
		    	if( parentStream.read(buffy) != buffy.length)
		    		throw new UnexpectedEOFException("End of file encountered reading delimited value.");
		    }

	    	temp.add(buffy[0]);
	    	temp.add(buffy[1]);
	    }
	    
	    value = new byte[temp.size()];
	    int index = 0;
	    for(Byte src : temp)
	    	value[index++] = src.byteValue();
	    
	    return value;
    }
	
	/**
	 * Return true if the subjectTag is equal to any of the delimiterTags
	 * @param subjectTag - the subject tag to 
	 * @param delimiterTags - an array of DataElementTag
	 * @return
	 */
    private boolean isAnyDataElementTag(DataElementTag subjectTag, DataElementTag... delimiterTags)
    {
		if(subjectTag == null)
			return (delimiterTags == null || delimiterTags.length == 0);
		
		for(DataElementTag delimiterTag : delimiterTags)
			if( subjectTag.equals(delimiterTag) )
			{
				logger.log(Level.FINER, "Matching data element tag [" + subjectTag + "] found.");
				return true;
			}
		return false;
    }

	
	/**
	 * A simple wrapper method to make the code clearer.
	 * @return
	 * @throws IOException
	 * @throws DicomFormatException
	 * @throws InvalidVRException 
	 */
	public ValueRepresentation readValueRepresentation()
	throws IOException, DicomFormatException, InvalidVRException
	{
		logger.log(Level.FINE, "Reading value representation into string.");
		String vrAsString = readEvenNumberOfBytesIntoString(2);
		ValueRepresentation vr = ValueRepresentation.valueOfIgnoreCase(vrAsString);
		if(vr == null)
			throw new InvalidVRException(vrAsString);
		return vr;
	}
	
	/**
	 * Read the specified number of bytes and return the bytes read as a String.
	 * If the bytesToread specifies an odd number then this method will read one
	 * additional byte and assure that it is a 0x00.
	 * 
	 * @param dicomInputStream
	 * @param bytesToRead
	 * @return
	 * @throws IOException
	 */
	public String readEvenNumberOfBytesIntoString(int bytesToRead) 
	throws IOException, DicomFormatException
	{
		logger.log(Level.FINE, "Reading " + bytesToRead + " bytes into string.");

		byte[] buffy = new byte[bytesToRead];
		StringBuffer sbuff = new StringBuffer(bytesToRead);
		readFully(buffy);
		for (int i = 0; i < bytesToRead; i++)
			sbuff.append((char) buffy[i]);
		if(logger.isLoggable(Level.FINEST))
		{
			StringBuffer sb = new StringBuffer();
			for(byte b : buffy)
			{
				if(sb.length() > 0)
					sb.append(',');
				sb.append("0x" + Integer.toHexString((int)b));
			}
			logger.log(Level.FINEST, "Read [" + sb.toString() + "]");
		}
		
		if(bytesToRead % 2 == 1)
			if( read() != 0 )
				throw new OddLengthWithoutZeroFillException();
		return sbuff.toString();
	}
	
	/**
	 * 
	 * @param bytesToRead
	 * @return
	 * @throws IOException
	 */
	public long readFourBytesIntoLong() 
	throws IOException
	{
		byte[] buffy = new byte[4];
		readFully(buffy);
		return TransferSyntaxUidUtility.makeUnsignedLongFrom4Bytes(this.transferSyntax.isLittleEndian(), buffy);
	}
	
	/**
	 * 
	 * @param bytesToRead
	 * @return
	 * @throws IOException
	 */
	public long readTwoBytesIntoLong() 
	throws IOException
	{
		byte[] buffy = new byte[2];
		readFully(buffy);
		return TransferSyntaxUidUtility.makeUnsignedLongFrom2Bytes(this.transferSyntax.isLittleEndian(), buffy);
	}
	
    /**
     * Peek at the next DataElementTag (group and element number), leaving the
     * stream at the same position when the call was made.
     *  
     * @return The next DataElementTag in the stream or null for EOF.
     * @throws UnsupportedOperationException if the underlying stream does not support mark() and reset()
     * @throws IOException 
     */
    public DataElementTag peekNextDataElementTag()
    throws UnsupportedOperationException, IOException
    {
    	try
    	{
	    	if( isRootDataSetStream() ? rootStream.markSupported() : parentStream.markSupported() )
	    	{
	    		mark(4);		// allow 4 bytes as that is all we need
	    		DataElementTag dataElementTag = readNextDataElementTag();
	    		reset();
	    		logger.log(Level.FINER, "Peeking at next data element tag, found " + dataElementTag);
	    		return dataElementTag;
	    	}
	    	else
	    		throw new UnsupportedOperationException("Underlying stream does not support mark() and reset(), unable to peek forward.");
    	}
    	catch(EOFException ioX)
    	{
    		return null;
    	}
    }
}
