/**
 * 
 */
package gov.va.med.imaging.dicom.parser.impl.rawvalueparsers;

import gov.va.med.imaging.dicom.dataset.TransferSyntaxUid;
import gov.va.med.imaging.dicom.dataset.elements.DataElement;
import gov.va.med.imaging.dicom.dataset.elements.DataElement_OB;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;
import gov.va.med.imaging.dicom.exceptions.RawPixelInterpretationValuesNotSetException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInvalidDataLengthException;
import gov.va.med.imaging.dicom.parser.impl.DataElementFactory;
import gov.va.med.imaging.dicom.parser.impl.RawDataToPixelTransformer;
import gov.va.med.imaging.dicom.parser.impl.TransferSyntaxUidUtility;
import gov.va.med.imaging.dicom.parser.io.DataElementLimitedInputStream;
import gov.va.med.imaging.dicom.parser.io.DataElementReader;

import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferInt;
import java.awt.image.WritableRaster;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.imageio.ImageIO;

/**
 * @author       DNS
 *
 */
public class RawValueParser_OB 
extends RawValueParser_NativePixelDataElement<DataElement_OB>
{
    private List<BufferedImage> bufferedImages = null;
    

    /**
     * @param factory
     * @param dataElement
     */
    public RawValueParser_OB(DataElementFactory factory, DataElement_OB dataElement)
    {
        super(factory, dataElement);
    }

    /* (non-Javadoc)
     * @see gov.va.med.imaging.hi5.server.dicom.RawValueParser#parseRawValue()
     */
    @Override
    public void parseRawValue() 
    throws DicomFormatException
    {
        // do nothing, 
    }

    /**
     * 
     * @author      DNS
     *
     */
    class ByteArrayListInputStream
    extends InputStream
    {
        private final List<ByteArrayInputStream> fragmentStreams;
        private int streamIndex = 0;
        
        ByteArrayListInputStream(List<byte[]> fragments)
        {
            this.fragmentStreams = new ArrayList<ByteArrayInputStream>();
            for(byte[] fragmentBuffer : fragments)
                fragmentStreams.add(new ByteArrayInputStream(fragmentBuffer));
        }

        @Override
        public int read() throws IOException
        {
            int value = -1;
            while(value < 0 && streamIndex < fragmentStreams.size())
            {
                value = fragmentStreams.get(streamIndex).read();
                if(value < 0) ++streamIndex;
            }
            
            return value;
        }
    }
    
    /**
     * @see gov.va.med.imaging.dicom.dataelement.PixelDataElement#getImageFrames()
     */
    public List<BufferedImage> getImageFrames() 
    throws DicomFormatException, IOException, InvalidVRModeException, InvalidVRException, RawPixelInterpretationValuesNotSetException
    {
        if(this.getFactory().getTransferSyntaxUid().isEncapsulated())
            return getEncapsulatedPixelData();
        else
            return getBufferedImages();
    }
    
    //====================================================================================================
    // Encapsulated Pixel Data (JPG, JP2) Handling
    //====================================================================================================
    private List<BufferedImage> frames = null;
    
    public synchronized List<BufferedImage> getEncapsulatedPixelData() 
    throws DicomFormatException, IOException, InvalidVRModeException, InvalidVRException
    {
        // if the pixel data is not encapsulated (i.e. is a raw bitmap) return null from this method
        // use getRawPixelData()
        if(! getFactory().getTransferSyntaxUid().isEncapsulated())
            return null;
        
        // parse this on demand as it is an expensive operation
        if(frames == null)
            frames = parseEncapsulatedPixelData();
        
        return frames;
    }

    /**
     * Return a List of byte arrays.
     * Each list element is a frame.
     * Each frame may contain many fragments but this method concatenates them together into one buffer.
     * The byte arrays are the content of the fragments.
     * 
     * @return
     * @throws DicomFormatException
     * @throws IOException
     * @throws InvalidVRModeException
     * @throws InvalidVRException
     */
    private List<BufferedImage> parseEncapsulatedPixelData() 
    throws DicomFormatException, IOException, InvalidVRModeException, InvalidVRException
    {
        List<BufferedImage> frameList = null;
        List<Long> offsetTable = null;
        TransferSyntaxUid transferSyntax = getFactory().getTransferSyntaxUid();
        //String imageMimeType = transferSyntax.getMimeType();
        
        if(transferSyntax.isEncapsulated())
        {
            // Spec 3.5-2007 Pg 64
            if(getDataElement().getValueLength() != DataElement.INDETERMINATE_LENGTH)
                throw new ValueRepresentationInvalidDataLengthException("An encapsulated transfer syntax pixel data MUST have a length of 0xffffffff.");
            
            DataElementLimitedInputStream parsingInputStream = 
                new DataElementLimitedInputStream(
                        new DataInputStream(
                                new ByteArrayInputStream(
                                        getDataElement().getRawValue())), getFactory().getTransferSyntaxUid());
            
            DataElementReader pixelDataReader = new DataElementReader(parsingInputStream, getFactory());
            DataElement<?> offsetTableDataElement = pixelDataReader.readNextDataElement();
            offsetTable = parseOffsetTable(offsetTableDataElement);
            
            /*
             * If the offset table is non-existent or is zero-length then there is one frame
             * in the data set.  The frame MAY BE spread over multiple fragments, each divided by
             * ITEM_DELIMITATION_TAG elements.
             * See PS 3.5-2007 Page 66 for an example.
             */
            if(offsetTable == null || offsetTable.size() == 0)
            {
                getLogger().finer("DataElement_OB, found no offset table, parsing pixel data as 1 fragment in 1 frame.");
                frameList = new ArrayList<BufferedImage>(1);
                
                List<byte[]> fragments = new ArrayList<byte[]>();
                for( DataElement<?> pixelDataFragment = pixelDataReader.readNextDataElement();
                    pixelDataFragment != null;
                    pixelDataFragment = pixelDataReader.readNextDataElement() )
                        fragments.add( pixelDataFragment.getRawValue() );
                
                InputStream imageInputStream = new ByteArrayListInputStream(fragments);
                BufferedImage frame = ImageIO.read(imageInputStream);
                
                frameList.add(frame);
            }
            else
            {
                // iterate through each data item in the element
                // each data item is an image fragment, the offset table 
                // specifies the offset at the start of each frame.
                frameList = new ArrayList<BufferedImage>(offsetTable.size());
                
                Iterator<Long> offsetTableIterator = offsetTable.iterator();
                Long nextOffset = offsetTableIterator.next();
                long totalOffset = 0L;
                
                List<byte[]> fragments = new ArrayList<byte[]>();
                for( DataElement<?> pixelDataFragment = pixelDataReader.readNextDataElement();
                    pixelDataFragment != null;
                    pixelDataFragment = pixelDataReader.readNextDataElement() )
                {
                    if(totalOffset >= nextOffset.longValue())
                    {
                        if(fragments.size() > 0)
                        {
                            InputStream imageInputStream = new ByteArrayListInputStream(fragments);
                            BufferedImage frame = ImageIO.read(imageInputStream);
                            frameList.add(frame);
                        }
                        fragments.clear();
                    }
                    fragments.add( pixelDataFragment.getRawValue() );
                    totalOffset += pixelDataFragment.getValueLength();
                }
                // add the last frame
                if(fragments.size() > 0)
                {
                    InputStream imageInputStream = new ByteArrayListInputStream(fragments);
                    BufferedImage frame = ImageIO.read(imageInputStream);
                    frameList.add(frame);
                }
            }
        }
        
        return frameList;
    }
    
    private List<Long> parseOffsetTable(DataElement<?> offsetTableDataElement)
    {
        if(offsetTableDataElement.getValueLength() == 0)
            return null;
        List<Long> fragmentOffsets = new ArrayList<Long>();
        
        TransferSyntaxUid transferSyntax = getFactory().getTransferSyntaxUid();
        byte[] rawOffsetTable = offsetTableDataElement.getRawValue();
        for( int fragmentIndex = 0; fragmentIndex < rawOffsetTable.length/4; ++fragmentIndex )
            fragmentOffsets.add( 
                new Long(
                    TransferSyntaxUidUtility.makeUnsignedLongFrom4Bytes(transferSyntax.isLittleEndian(), rawOffsetTable, fragmentIndex * 4)) );
        
        return fragmentOffsets;
    }
    
    // ====================================================================================================================
    /**
     * Returns an array of int, one for each pixel, each is a maximum of 8 significant bits
     * per pixel (unsigned values, hence the wasted bits)
     * 
     * If native, it shall have a defined Value Length, and be encoded as follows:
     * - where Bits Allocated (0028,0100) has a value greater than 8 shall have 
     *   Value Representation OW and shall be encoded in Little Endian;
     * - where Bits Allocated (0028,0100) has a value less than or equal to 8 shall have the 
     *   Value Representation OB or OW and shall be encoded in Little Endian.
     *   
     * The raw data is organized in chunks specified by the bitsAllocated parameter.
     * The only guarantee is that the bitsAllocated must be less than 8 bits (else the
     * image must be stored in a OW element.
     * 
     * @see gov.va.med.imaging.hi5.shared.dicom.dataelement.NativePixelDataElement#getRawPixelData(int, int, int)
     */
    public synchronized List<BufferedImage> getBufferedImages()
    throws DicomFormatException, RawPixelInterpretationValuesNotSetException
    {
        if(getFactory().getTransferSyntaxUid().isEncapsulated())
            return null;
        
        if( bufferedImages == null)
        {
            if(getDataElement().getBitsStored() == 0)
                throw new RawPixelInterpretationValuesNotSetException();
            
            bufferedImages = new ArrayList<BufferedImage>();
            
            int[] destination = getRawPixels(); 
            
            DataBuffer dataBuffer = new DataBufferInt(destination, destination.length);
            WritableRaster imageRaster = WritableRaster.createPackedRaster(
                    dataBuffer, getDataElement().getWidth(), getDataElement().getHeight(), getDataElement().getBitsStored(), new Point(0,0));
            BufferedImage image = new BufferedImage(getDataElement().getWidth(), getDataElement().getHeight(), BufferedImage.TYPE_BYTE_GRAY);
            
            image.setData(imageRaster);
            
            bufferedImages.add( image );
        }
        
        return bufferedImages;
    }    
    
    /**
     * 
     * @return
     * @throws DicomFormatException
     */
    public int[] getRawPixels() 
    throws DicomFormatException
    {
        RawDataToPixelTransformer transformer = new RawDataToPixelTransformer();
        
        return transformer.transformRawByteDataToIntPixelArray(
            getDataElement().getRawValue(), 
            getDataElement().getHeight(), getDataElement().getWidth(), 
            getDataElement().getBitsAllocated(), getDataElement().getBitsStored(), getDataElement().getHighBit());
    }
}
