/**
 * 
 */
package gov.va.med.imaging.dicom.parser.impl;

import gov.va.med.imaging.dicom.dataset.TransferSyntaxUid;

import java.util.regex.Pattern;

/**
 * @author      DNS
 *
 */
public class TransferSyntaxUidUtility
{
    public static TransferSyntaxUid getByUid(String uid)
    {
        for(TransferSyntaxUid transferSyntax : TransferSyntaxUid.values())
            if(matches(transferSyntax, uid))
                return transferSyntax;
        
        return null;
    }
    

    /**
     * 
     * @param transferSyntax
     * @param pattern
     * @return
     */
    public static boolean matches(TransferSyntaxUid transferSyntax, String uid)
    {
        if(transferSyntax == null || uid == null)
            return false;
        
        String transferSyntaxUidPattern = transferSyntax.getPattern();
        if(transferSyntaxUidPattern.indexOf('.') >= 0)
            transferSyntaxUidPattern.replaceAll(".", "\\x2e");
        Pattern pattern = Pattern.compile(transferSyntaxUidPattern);
        return pattern.matcher(uid).matches();
    }
    
    /**
     * Given a two byte array, form an unsigned long, taking into account whether
     * the format is big or little endian.
     * 
     * @param argtmp
     * @return
     */
    public static long makeUnsignedLongFrom2Bytes(boolean littleEndian, byte[] argtmp)
    {
        return (long)makeUnsignedIntFrom2Bytes(littleEndian, argtmp);   
    }

    /**
     * @param value
     * @return
     */
    public static byte[] make2BytesFromLong(boolean littleEndian, long value)
    {
        byte[] result = new byte[2];
        
        if( littleEndian )
        {
            result[1] = (byte)((0x000000000000ff00 & value)>>8);
            result[0] = (byte)(0x00000000000000ff & value);
        }
        else
        {
            result[0] = (byte)((0x000000000000ff00 & value)>>8);
            result[1] = (byte)(0x00000000000000ff & value);
        }
        
        return result;
    }
    
    /**
     * Given a two byte array, form an unsigned integer, taking into account whether
     * the format is big or little endian.
     * 
     * Little Endian
     * The least significant byte (LSB) value is at the lowest address. 
     * The other bytes follow in increasing order of significance.
     * 
     * Big Endian
     * The most significant byte (LSB) value is at the lowest address. 
     * The other bytes follow in decreasing order of significance.
     * 
     * @param argtmp
     * @return
     */
    public static int makeUnsignedIntFrom2Bytes(boolean littleEndian, byte[] argtmp)
    {
        return makeUnsignedIntFrom2Bytes(littleEndian, argtmp, 0);
    }
    
    public static int makeUnsignedIntFrom2Bytes(boolean littleEndian, byte[] argtmp, int offset)
    {
        return littleEndian ?
                ( (0x000000ff & argtmp[1]) << 8 ) | (0x000000ff & argtmp[offset]) :
                ( (0x000000ff & argtmp[0]) << 8 ) | (0x000000ff & argtmp[offset + 1]);
    }
    
    public static byte[] make2BytesFromInt(boolean littleEndian, int value)
    {
        byte[] result = new byte[2];
        
        if( littleEndian )
        {
            result[1] = (byte)((0x0000ff00 & value)>>8);
            result[0] = (byte)(0x000000ff & value);
        }
        else
        {
            result[0] = (byte)((0x0000ff00 & value)>>8);
            result[1] = (byte)(0x000000ff & value);
        }
        
        return result;
    }
    
    public static int makeUnsignedIntFrom4Bytes(boolean littleEndian, byte[] rawValue)
    {
        return makeUnsignedIntFrom4Bytes(littleEndian, rawValue, 0);
    }    
    public static int makeUnsignedIntFrom4Bytes(boolean littleEndian, byte[] rawValue, int offset)
    {
        return (int)makeLongFromBytes(littleEndian, rawValue, offset, 4);
    }    
    
    public static byte[] make4BytesFromInt(boolean littleEndian, int value)
    {
        byte[] result = new byte[4];
        
        if( littleEndian )
        {
            result[3] = (byte)((0xff000000 & value)>>24);
            result[2] = (byte)((0x00ff0000 & value)>>16);
            result[1] = (byte)((0x0000ff00 & value)>>8);
            result[0] = (byte) (0x000000ff & value);
        }
        else
        {
            result[0] = (byte)((0xff0000 & value)>>24);
            result[1] = (byte)((0x00ff0000 & value)>>16);
            result[2] = (byte)((0x0000ff00 & value)>>8);
            result[3] = (byte)(0x000000ff & value);
        }
        
        return result;
    }
    
    /**
     * Given a four byte array, form an unsigned integer, taking into account whether
     * the format is big or little endian.
     * 
     * @param argtmp
     * @return
     */
    public static long makeUnsignedLongFrom4Bytes(boolean littleEndian, byte[] argtmp)
    {
        return (long)makeUnsignedIntFrom4Bytes(littleEndian, argtmp, 0);
    }    
    public static long makeUnsignedLongFrom4Bytes(boolean littleEndian, byte[] rawValue, int offset)
    {
        return makeLongFromBytes(littleEndian, rawValue, offset, 4);
    }    
    
    public static byte[] make4BytesFromLong(boolean littleEndian, long value)
    {
        return make4BytesFromInt(littleEndian, (int)value);
    }

    public static long makeUnsignedLongFrom8Bytes(boolean littleEndian, byte[] rawValue)
    {
        return makeUnsignedLongFrom8Bytes(littleEndian, rawValue, 0);
    }
    
    public static long makeUnsignedLongFrom8Bytes(boolean littleEndian, byte[] rawValue, int offset)
    {
        return makeLongFromBytes(littleEndian, rawValue, offset, 8);
    }
    
    /**
     * 
     * @param rawValue
     * @param offset
     * @param count
     * @return
     */
    public static long makeLongFromBytes(boolean littleEndian, byte[] rawValue, int offset, int count)
    {
        long result = 0l;
        
        if(littleEndian)
            for(int index=count-1; index >= 0; index--)
                result |= (0x000000ff & rawValue[offset + index]) << (8*index);
        else
            for(int index=0; index < count; ++index)
                result |= (0x000000ff & rawValue[offset + ((count-1)-index)]) << (8*index);
        
        return result;
    }

    /**
     * 
     * @param rawValue
     * @return
     */
    public static Float makeFloatFrom4Bytes(boolean littleEndian, byte[] rawValue)
    {
        return makeFloatFrom4Bytes(littleEndian, rawValue, 0);
    }
    
    /**
     * 
     * @param rawValue
     * @param offset
     * @return
     */
    public static Float makeFloatFrom4Bytes(boolean littleEndian, byte[] rawValue, int offset)
    {
        int bits = makeUnsignedIntFrom4Bytes(littleEndian, rawValue, offset);
        return new Float( Float.intBitsToFloat(bits) );
    }
    
    /**
     * Make a word array in transfer syntax defined order from a byte array.
     * Each two bytes in the byte array makes up a word, the resulting array
     * will be in the defined byte order (i.e. for big-endian does nothing,
     * for little-endian, swaps every other byte).
     * 
     * To save memory, the resulting array is written to the array referenced 
     * by the rawValue parameter (i.e. in place). 
     * 
     * The offset and length are both in terms of bytes.  The length MUST be an
     * even number
     * 
     * @param rawValue
     * @param offset
     * @return
     */
    public static void intepretByteArrayAsWordArray(boolean littleEndian, byte[] rawValue)
    {
        intepretByteArrayAsWordArray(littleEndian, rawValue, 0, rawValue.length);
    }
    
    public static void intepretByteArrayAsWordArray(boolean littleEndian, byte[] rawValue, int offset, int length)
    {
        byte[] result = new byte[length];
        
        if(littleEndian)
        {
            for(int index = 0; index < length; index += 2)
            {
                byte tmp = result[index];
                result[index] = rawValue[index+1];
                result[index+1] = tmp;
            }
        }
    }
}
