/**
 * Package: MAG - VistA Imaging
 * WARNING: Per VHA Directive 2004-038, this routine should not be modified.
 * Date Created: Jul 18, 2008
 * Site Name:  Washington OI Field Office, Silver Spring, MD
 * @author       DNS
 * @version 1.0
 *
 * ----------------------------------------------------------------
 * Property of the US Government.
 * No permission to copy or redistribute this software is given.
 * Use of unreleased versions of this software requires the user
 * to execute a written test agreement with the VistA Imaging
 * Development Office of the Department of Veterans Affairs,
 * telephone (DNS.
 * 
 * The Food and Drug Administration classifies this software as
 * a Class II medical device.  As such, it may not be changed
 * in any way.  Modifications to this software may result in an
 * adulterated medical device under 21CFR820, the use of which
 * is considered to be a violation of US Federal Statutes.
 * ----------------------------------------------------------------
 */
package gov.va.med.imaging.dicom.parser.impl;

import gov.va.med.imaging.dicom.dataset.TransferSyntaxUid;
import gov.va.med.imaging.dicom.dataset.ValueRepresentation;
import gov.va.med.imaging.dicom.dataset.elements.DataElement;
import gov.va.med.imaging.dicom.dataset.elements.DataElementTag;
import gov.va.med.imaging.dicom.dictionary.DicomDictionary;
import gov.va.med.imaging.dicom.dictionary.DicomDictionaryEntry;
import gov.va.med.imaging.dicom.dictionary.impl.DicomDictionaryFactory;
import gov.va.med.imaging.dicom.dictionary.impl.DicomDictionaryUtility;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;
import gov.va.med.imaging.dicom.exceptions.UnspecifiedVRWithExplicitVRException;
import gov.va.med.imaging.dicom.parser.impl.rawvalueparsers.RawValueParserFactory;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author       DNS
 *
 */
public class DataElementFactoryImpl 
implements DataElementFactory
{
	private final DicomDictionary dictionary;
	private final TransferSyntaxUid transferSyntaxUid;
	private final RawValueParserFactory rawValueParserFactory;
	
	private final boolean createUnknownDataElementTag;
	private Logger logger = Logger.getLogger(this.getClass().getName());
	
	private static DataElementFactory defaultHeaderElementFactory;
	
	/**
	 * A special element factory that reads the DICOM header elements.
	 * ===> Explicit VR Little Endian <===
	 * 
	 * See DICOM spec, section 7.3 for details.
	 * 
	 * ..., the File Meta Information shall be encoded 
	 * using the Explicit VR Little Endian Transfer Syntax (UID=1.2.840.10008.1.2.1) as defined in DICOM PS 3.5. 
	 * Values of each File Meta Element shall be padded when necessary to achieve an even length as specified in 
	 * PS 3.5 by their corresponding Value Representation. 
	 * For compatibility with future versions of this Standard, any Tag (0002,xxxx) not defined in Table 7.1-1 
	 * shall be ignored.
	 * 
	 * @return
	 */
	public static synchronized DataElementFactory getDefaultHeaderElementFactory()
	{
		if(defaultHeaderElementFactory == null)
			defaultHeaderElementFactory = new DataElementFactoryImpl(DicomDictionaryFactory.getDefaultHeaderDictionary(), TransferSyntaxUid.RAW_EXPLICIT_VR_LITTLEENDIAN, false);
		
		return defaultHeaderElementFactory;
	}
	
	/**
	 * 
	 * @param transferSyntax
	 * @return
	 */
	public static DataElementFactory getDataElementFactory(TransferSyntaxUid transferSyntax)
	{
		return new DataElementFactoryImpl( DicomDictionaryFactory.getDefault(), transferSyntax, true );
	}
	
	/**
	 * 
	 * @param dictionary
	 */
	public DataElementFactoryImpl(
			DicomDictionary dictionary, 
			TransferSyntaxUid transferSyntaxUid, 
			boolean createUnknownDataElementTag)
	{
		this.dictionary = dictionary;
		this.transferSyntaxUid = transferSyntaxUid;
		this.createUnknownDataElementTag = createUnknownDataElementTag;
		this.rawValueParserFactory = new RawValueParserFactory(this);
	}
	
	/* (non-Javadoc)
     * @see gov.va.med.imaging.hi5.server.dicom.DataElementFactory#getDictionary()
     */
    @Override
    public DicomDictionary getDictionary()
    {
    	return dictionary;
    }

    /* (non-Javadoc)
     * @see gov.va.med.imaging.hi5.server.dicom.DataElementFactory#getTransferSyntaxUid()
     */
    @Override
    public TransferSyntaxUid getTransferSyntaxUid()
    {
    	return this.transferSyntaxUid;
    }
    
	/* (non-Javadoc)
     * @see gov.va.med.imaging.hi5.server.dicom.DataElementFactory#isExplicitVR()
     */
    @Override
    public boolean isExplicitVR()
    {
    	return this.transferSyntaxUid.isExplicitVR();
    }

	/* (non-Javadoc)
     * @see gov.va.med.imaging.hi5.server.dicom.DataElementFactory#isLittleEndian()
     */
    @Override
    public boolean isLittleEndian()
    {
    	return this.transferSyntaxUid.isLittleEndian();
    }

	/* (non-Javadoc)
     * @see gov.va.med.imaging.hi5.server.dicom.DataElementFactory#isCreateUnknownDataElementTag()
     */
    @Override
    public boolean isCreateUnknownDataElementTag()
    {
    	return createUnknownDataElementTag;
    }

	/* (non-Javadoc)
     * @see gov.va.med.imaging.hi5.server.dicom.DataElementFactory#createDataElement(gov.va.med.imaging.hi5.server.dicom.DataElementTag, gov.va.med.imaging.hi5.server.dicom.ValueRepresentation, long, byte[])
     */
	@Override
    public DataElement<?> createDataElement(
			DataElementTag dataElementTag, 
			ValueRepresentation vrField, 	// must be null if implicit VR, must be non-null if explicit VR
			long valueLength, 
			byte[] value)
	throws InvalidVRModeException
	{
		DataElement<?> dataElement = null;
		
		try
        {
	        if( ! isExplicitVR() ||
	        	DataElementTag.SEQUENCE_DELIMITATION_TAG.equals(dataElementTag) ||
	        	DataElementTag.ITEM_DELIMITATION_TAG.equals(dataElementTag) ||
	        	DataElementTag.SEQUENCE_ITEM_TAG.equals(dataElementTag) )
	        {
	        	DicomDictionaryEntry dictionaryEntry = dictionary.get(dataElementTag);
	        	
	        	// create a tag if it is not in the dictionary
	        	if(dictionaryEntry == null && isCreateUnknownDataElementTag())
	        	{
	        		String userEntryName = dictionary.createUserEntry(dataElementTag);
	        		dictionaryEntry = dictionary.get(userEntryName);
	        	}
	        	
	        	// create an instance of the DataElement
	        	Class<? extends DataElement<?>> dataElementClass = dictionaryEntry.getVr()[0].getDataElementClass();
	        	Constructor<? extends DataElement<?>> constructor = 
	        		dataElementClass.getConstructor(DataElementFactoryImpl.class, DataElementTag.class, DicomDictionaryEntry.class, long.class, byte[].class );
	        	
	        	dataElement = (DataElement<?>)constructor.newInstance(this, dataElementTag, dictionaryEntry, valueLength, value);
	        }
	        else
	        {
	        	if(vrField == null)
	        		throw new UnspecifiedVRWithExplicitVRException();
	        	
	        	Class<? extends DataElement<?>> dataElementClass = vrField.getDataElementClass();
	        	Constructor<? extends DataElement<?>> constructor = 
	        		dataElementClass.getConstructor(DataElementTag.class, ValueRepresentation.class, long.class, byte[].class );
	        	
	        	dataElement = (DataElement<?>)constructor.newInstance(dataElementTag, vrField, valueLength, value);
	        }
	        
            RawValueParser<?,?> rawValueParser = this.rawValueParserFactory.createRawValueParser(dataElement);
            rawValueParser.parseRawValue();
        } 
		catch (InvocationTargetException itX)
        {
			logger.log(Level.SEVERE, "Error creating DataElement", itX.getCause());
        }
		catch (Exception e)
        {
			logger.log(Level.SEVERE, "Error creating DataElement", e);
        } 
		
		return dataElement;
	}

	/* (non-Javadoc)
     * @see gov.va.med.imaging.hi5.server.dicom.DataElementFactory#getImplicitVR(gov.va.med.imaging.hi5.server.dicom.DataElementTag)
     */
	@Override
    public ValueRepresentation getImplicitVR(DataElementTag dataElementTag)
	{
		DicomDictionaryEntry entry = DicomDictionaryUtility.getByElementKey(getDictionary(), dataElementTag);
		return (entry == null ? null : entry.getVr()[0]);
	}
}
