package gov.va.med.imaging.dicom.dictionary.impl;

import gov.va.med.imaging.dicom.dataset.ValueRepresentation;
import gov.va.med.imaging.dicom.dictionary.DicomDictionary;
import gov.va.med.imaging.dicom.dictionary.DicomDictionaryEntry;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Reader;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 
 * @author       DNS
 *
 */
public class DicomDictionaryReaderImpl 
{
	private static final long serialVersionUID = 1L;

	/**
	 * 
	 * @param dictionarySource
	 * @throws IOException
	 */
	public static DicomDictionary create(Reader dictionarySource) 
	throws IOException
	{
	    DicomDictionary dictionary = new DicomDictionary();
		LineNumberReader dictionaryReader = new LineNumberReader(dictionarySource);
			
		for(String line = dictionaryReader.readLine(); line != null; line = dictionaryReader.readLine())
			parseAndPutLine(dictionary, line);
		
		return dictionary;
	}

	private static String ElementNameRegex = "[\\w][\\w\\-]*";
	private static String ElementRangeRegex = "0[Xx][0-9a-fA-F]{4},[\\s]*0[Xx][0-9a-fA-F]{4},[\\s]*0[Xx][0-9a-fA-F]{4},[\\s]*0[Xx][0-9a-fA-F]{4}";
	private static String ElementDescriptionRegex = "\"[\\w][\\w\\-\\s]*[\\w\\-]\"";
	private static String MultiplicityRangeRegex = "[\\d]{1,10},[\\s]*(Integer\\x2eMAX_VALUE|[\\d]{1,10})";
	private static String BooleanRegex = "[Tt][Rr][Uu][Ee]|[Ff][Aa][Ll][Ss][Ee]";
	private static String ValueRepresentationRegex = "(ValueRepresentation\\x2e[A-Z]{2})";
	private static String WhitespaceCommaRegex = ",[\\s]*";
	private static String LineRegEx = 
		"[\\s]*" + 
		ElementNameRegex + 
		"\\x28" +
		ElementRangeRegex + WhitespaceCommaRegex + 
		ElementDescriptionRegex + WhitespaceCommaRegex +
		MultiplicityRangeRegex + WhitespaceCommaRegex +
		BooleanRegex + WhitespaceCommaRegex +
		ValueRepresentationRegex +
		"\\x29" +
		"[\\x2c]?[\\s]*";
	private static String LineRegex = 
		"[\\s]*" + 			// leading whitespace
		"([\\w][\\w\\-]*)" + 	// the name of the element
		"\\x28" + 			// left parenthesis
		"(0[Xx])([0-9a-fA-F]{4})" + ",[\\s]*" + 
		"(0[Xx])([0-9a-fA-F]{4})" + ",[\\s]*" +
		"(0[Xx])([0-9a-fA-F]{4})" + ",[\\s]*" + 
		"(0[Xx])([0-9a-fA-F]{4})" + ",[\\s]*" +
		"\"([\\w][\\w\\-\\s]*[\\w\\-])\"" + ",[\\s]*" + 
		"([\\d]{1,10})" + ",[\\s]*" + 
		"(Integer\\x2eMAX_VALUE|[\\d]{1,10})" + ",[\\s]*" + 
		"([Tt][Rr][Uu][Ee]|[Ff][Aa][Ll][Ss][Ee])" + ",[\\s]*" +
		"ValueRepresentation\\x2e([A-Z][A-Z])" + 
		"\\x29" + "[\\s]*" + 		// the right parenthesis
		"[\\x2c]?" + 	// an optional comma
		"[\\s]*";		// trailing whitespace
	private static Pattern LinePattern = Pattern.compile(LineRegex);
	
	/**
	 * CommandGroupLength(0x0000, 0x0000, 0x0000, 0x0000, "CommandGroupLength", 1, 1, false, ValueRepresentation.UL)
	 * @param dictionary 
	 * 
	 * @param line
	 * @return
	 */
	private static void parseAndPutLine(DicomDictionary dictionary, String line)
	{
		Matcher matcher = LinePattern.matcher(line);
		if( matcher.matches() )
		{
			// Pap31ImageSequence(0x0041, 0x0041, 0x1050, 0x1050, "Pap31ImageSequence", 1, 1, false, ValueRepresentation.SQ),
			// Group1'Pap31ImageSequence
			// Group2'0x'
			// Group3'0041'
			// Group4'0x'
			// Group5'0041'
			// Group6'0x'
			// Group7'1050'
			// Group8'0x'
			// Group9'1050'
			// Group10'Pap31ImageSequence'
			// Group11'1'
			// Group12'1'
			// Group13'false'
			// Group14'SQ'
			
			String elementName = matcher.group(1);
			int groupStart = Integer.parseInt(matcher.group(3), 16);
			int groupEnd = Integer.parseInt(matcher.group(5), 16);
			int elementStart = Integer.parseInt(matcher.group(7), 16);
			int elementEnd = Integer.parseInt(matcher.group(9), 16);
			String elementDescription = matcher.group(10);
			int minMultiplicity = Integer.parseInt(matcher.group(11));
			int maxMultiplicity = matcher.group(12).equals("Integer.MAX_VALUE") ? Integer.MAX_VALUE : Integer.parseInt(matcher.group(12));
			boolean retired = Boolean.parseBoolean(matcher.group(13));
			ValueRepresentation vr = ValueRepresentation.valueOf(matcher.group(14));

			dictionary.put(elementName, 
				new DicomDictionaryEntry(groupStart, groupEnd, elementStart, elementEnd, elementDescription,
				minMultiplicity, maxMultiplicity, retired, vr) );
		}
	}

	/**
	 * 
	 * @param argc
	 */
	public static void main(String[] argc)
	{
		InputStreamReader reader = null;
		DicomDictionary dictionary = null;
		
		System.out.println(DicomDictionaryReaderImpl.LineRegEx);
		try
		{
			reader = new InputStreamReader(
				DicomDictionary.class.getClassLoader().getResourceAsStream("DicomDictionary.txt") );
			dictionary = DicomDictionaryReaderImpl.create(reader);
			
			for(String elementName : dictionary.keySet())
			{
				System.out.println( elementName + "->" + dictionary.get(elementName).toString() );
			}
		}
		catch(Exception x)
		{
			x.printStackTrace();
		}
		finally
		{
			try{reader.close();}catch(Exception x){}
		}
	}
}
