package gov.va.med.imaging.dicom;

import gov.va.med.imaging.dicom.spi.DicomSPI;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.logging.Logger;

/**
 * 
 * @author     DNS
 *
 */
public class DicomServiceProvider
{
    private Logger logger = Logger.getLogger("DicomServiceProvider");
    
    // ==================================================================================================================
    // Singleton Implementation
    // ==================================================================================================================
    private static DicomServiceProvider singleton = null;
    public static synchronized DicomServiceProvider getSingleton()
    {
        if(singleton == null)
            singleton = new DicomServiceProvider();
        
        return singleton;
    }
    
    // ==================================================================================================================
    // Service Loader
    // ==================================================================================================================
    private final Object serviceLoaderSynch = new Object();     // used for its monitor only 
    private ServiceLoader<? extends DicomServiceProvider> serviceLoader;
    private SortedSet<DicomServiceDescription<?>> availableServices;

    /**
     * 
     * @return
     */
    private final ServiceLoader<? extends DicomServiceProvider> getServiceLoader()
    {
        synchronized (serviceLoaderSynch)
        {
            if(serviceLoader == null)
                serviceLoader = ServiceLoader.load(DicomServiceProvider.class);
        }
        return serviceLoader;
    }
    
    /**
     * 
     * @return
     */
    private final SortedSet<? extends DicomServiceDescription<?>> getAllServices()
    {
        synchronized (serviceLoaderSynch)
        {
            if(this.availableServices == null)
            {
                this.availableServices = new TreeSet<DicomServiceDescription<?>>();
                for(Iterator<? extends DicomServiceProvider> providerIter = getServiceLoader().iterator();
                    providerIter.hasNext(); )
                {
                    DicomServiceProvider provider = providerIter.next();
                    logger.info("Initializing provider " + provider.getName() + " - " + provider.getVersion());
                    
                    // if the provider initializes correctly then add its services to the available set
                    if( provider.initialize() )
                    {
                        this.availableServices.addAll( provider.getServices() );
                        logger.warning("Initialized provider " + provider.getName() + " - " + provider.getVersion());
                    }
                    else
                        logger.warning("Failed to initialize provider " + provider.getName() + " - " + provider.getVersion());
                }
            }
        }        
        return this.availableServices;
    }
    
    /**
     * 
     * @param <S>
     * @param spiType
     * @return
     */
    @SuppressWarnings("unchecked")
    private final <S extends DicomSPI> SortedSet<DicomServiceDescription<S>> getServicesOfType(Class<S> spiType)
    {
        SortedSet<DicomServiceDescription<S>> services = new TreeSet<DicomServiceDescription<S>>();
        
        for(DicomServiceDescription<?> serviceDescription : getAllServices())
        {
            if(spiType == serviceDescription.getImplementedSpi())
                services.add((DicomServiceDescription<S>) serviceDescription);
        }
        
        return services;
    }
    
    // ==================================================================================================================
    // Service Availability Methods
    // ==================================================================================================================
    private Map<Class<? extends DicomSPI>, DicomServiceFactory<? extends DicomSPI>> factoryMap = null;
    
    /**
     * 
     * @param <S>
     * @param spiClass
     * @return
     */
    private <S extends DicomSPI> DicomServiceFactory<S> getOrCreateFactory(Class<S> spiClass)
    {
        synchronized (serviceLoaderSynch)
        {
            if(factoryMap == null)
                factoryMap = new HashMap<Class<? extends DicomSPI>, DicomServiceFactory<? extends DicomSPI>>();
            
            @SuppressWarnings("unchecked")
            DicomServiceFactory<S> factory = (DicomServiceFactory<S>) factoryMap.get(spiClass);
            if(factory == null)
            {
                factory = new DicomServiceFactory<S>(getServicesOfType(spiClass));
                factoryMap.put(spiClass, factory);
            }
            return factory;
        }
    }
    
    /**
     * 
     * @param <S>
     * @param spiClass
     * @return
     * @throws DicomServiceCreationException
     */
    public <S extends DicomSPI> S create(Class<S> spiClass) 
    throws DicomServiceCreationException
    {
        DicomServiceFactory<S> factory = getOrCreateFactory(spiClass);
        return factory.create();
    }
    
    /**
     * 
     */
    public final void reloadServices()
    {
        synchronized (serviceLoaderSynch)
        {
            this.serviceLoader = null;
            this.availableServices = null;
            this.factoryMap = null;
        }
    }
    
    // ==================================================================================================================
    // Members that should be overridden by derived classes
    // ==================================================================================================================
    private static final String PROVIDER_NAME = "RootDicomProvider";
    private static final float PROVIDER_VERSION = 1.0f;
    private static final String PROVIDER_INFO = "No SPI implementation provided";

    private final String name; 
    private final float version;
    private final String info;
    
    /**
     * The public "nullary" constructor that is used by the ServiceLoader class
     * to create instances.
     */
    private DicomServiceProvider()
    {
        this(PROVIDER_NAME, PROVIDER_VERSION, PROVIDER_INFO);
    }

    /**
     * 
     * @param name
     * @param version
     * @param info
     */
    protected DicomServiceProvider(String name, float version, String info)
    {
        this.name = name;
        this.version = version;
        this.info = info;
    }

    public String getName()
    {
        return name;
    }

    public float getVersion()
    {
        return version;
    }

    public String getInfo()
    {
        return info;
    }
    
    /**
     * 
     * @return
     */
    protected SortedSet<DicomServiceDescription<?>> getServices()
    {
        return null;
    }

    /**
     * 
     * @return
     */
    public boolean initialize()
    {
        return false;
    }
}
