package gov.va.med.imaging.dicom;

import gov.va.med.imaging.dicom.spi.DicomSPI;

/**
 * 
 * @author       DNS
 *
 * @param <S>
 */
public class DicomServiceDescription<S extends DicomSPI>
implements Comparable<DicomServiceDescription<?>>
{
    private final DicomServiceProvider provider;
    private final Class<S> implementedSpi;
    private final float version;
    private final Class<?> implementingClass; 
    private final boolean reentrant;
    
    public DicomServiceDescription(
        DicomServiceProvider provider, 
        Class<S> implementedSpi,
        float version, 
        Class<?> implementingClass, 
        boolean reentrant)
    {
        if(provider == null)
            throw new IllegalArgumentException("DicomProvider cannot be null.");
        if(implementedSpi == null)
            throw new IllegalArgumentException("ImplementedSPI cannot be null.");
        if(implementingClass == null)
            throw new IllegalArgumentException("ImplementingClass cannot be null.");
        
        this.provider = provider;
        this.implementedSpi = implementedSpi;
        this.version = version;
        this.implementingClass = implementingClass;
        this.reentrant = reentrant;
    }

    public DicomServiceProvider getProvider()
    {
        return provider;
    }

    public Class<S> getImplementedSpi()
    {
        return implementedSpi;
    }

    public float getVersion()
    {
        return version;
    }

    public Class<?> getImplementingClass()
    {
        return implementingClass;
    }

    public boolean isReentrant()
    {
        return reentrant;
    }

    /**
     * Sort order is:
     * ascending by implementedSPI
     * descending by version
     * ascending by provider
     * 
     * Sort order is consistent with hashCode() and equals()
     * @return a negative integer, zero, or a   positive integer as this object is 
     *           less than,        equal to, or greater than the specified object.
     */
    @Override
    public int compareTo(DicomServiceDescription<?> other)
    {
        int implementedSpiCompare = implementedSpi.getName().compareTo(other.implementedSpi.getName());
        if(implementedSpiCompare != 0)
            return implementedSpiCompare;
        
        if( this.getVersion() < other.getVersion())
            return -1;
        else if( this.getVersion() > other.getVersion())
            return 1;
        
        int providerComparator = getProvider().getClass().getName().compareTo(other.getProvider().getClass().getName());

        return providerComparator;
    }

    @Override
    public int hashCode()
    {
        final int prime = 31;
        int result = 1;
        result = prime * result
                + ((implementedSpi == null) ? 0 : implementedSpi.hashCode());
        result = prime * result
                + ((provider == null) ? 0 : provider.hashCode());
        result = prime * result + Float.floatToIntBits(version);
        return result;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        
        DicomServiceDescription<?> other = (DicomServiceDescription<?>) obj;
        if (implementedSpi == null)
        {
            if (other.implementedSpi != null)
                return false;
        } else if (!implementedSpi.equals(other.implementedSpi))
            return false;
        
        if (Float.floatToIntBits(version) != Float.floatToIntBits(other.version))
            return false;
        
        if (provider == null)
        {
            if (other.provider != null)
                return false;
        } else if (!provider.equals(other.provider))
            return false;
        
        return true;
    }
    
    
}
