package gov.va.med.imaging.dicom.io;

import gov.va.med.imaging.dicom.DataElement;
import gov.va.med.imaging.dicom.DataElementFactory;
import gov.va.med.imaging.dicom.DataElementTag;
import gov.va.med.imaging.dicom.DataSet;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.IncompatibleValueLengthField;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This class is given an InputStream, positioned at the start of a DataSet,
 * and reads until the stream is exhausted.
 * 
 * @author       DNS
 *
 */
public class DataSetReader
{
	private final DataElementReader reader;
	private Logger logger = Logger.getLogger(this.getClass().getName());
	
	/**
	 * 
	 * @param inputStream
	 * @param dataElementFactory
	 */
	public DataSetReader(DataElementLimitedInputStream inputStream, DataElementFactory elementFactory)
	{
		this.reader = new DataElementReader(inputStream, elementFactory);
	}

	/**
	 * Read a DataSet, starting from the current position and stopping at the end of
	 * the file.
	 * 
	 * @return
	 * @throws IOException 
	 * @throws UnsupportedOperationException 
	 * @throws InvalidVRException 
	 * @throws InvalidVRModeException 
	 * @throws DicomFormatException 
	 */
	public DataSet readDataSet() 
	throws UnsupportedOperationException, IOException, InvalidVRModeException, InvalidVRException, DicomFormatException
	{
		logger.log( Level.INFO, "Reading root data set" );
		DataSet dataSet = new DataSet(reader.getDataElementFactory().getTransferSyntaxUid());
		
		for( DataElement<?> dataElement = reader.readNextDataElement();
			dataElement != null;
			dataElement = reader.readNextDataElement() )
		{
			dataSet.add(dataElement);
		}
		
		return dataSet;
	}
	
	/**
	 * Read a DataSet, starting from the current position and stopping at the next
	 * SEQUENCE_ITEM_DELIMITER tag.  The delimiter tag is NOT included in either
	 * the returned DataSet or the parent data set.
	 * 
	 * @return
	 * @throws UnsupportedOperationException
	 * @throws IOException
	 * @throws InvalidVRModeException
	 * @throws InvalidVRException
	 * @throws DicomFormatException 
	 */
	public DataSet readSequenceElementDataSet(long valueLength) 
	throws UnsupportedOperationException, IOException, InvalidVRModeException, InvalidVRException, DicomFormatException
	{
		logger.log( Level.INFO, "Reading sequence element data set, length is " + 
				(valueLength == DataElement.INDETERMINATE_LENGTH ? "indeterminate" : Long.toHexString(valueLength)) );
		
		return valueLength == DataElement.INDETERMINATE_LENGTH ?
				readIndeterminateLengthSequenceElementDataSet() :
				readDeterminateLengthSequenceElementDataSet(valueLength);
	}
	
	private DataSet readIndeterminateLengthSequenceElementDataSet()
	throws IncompatibleValueLengthField, IOException, InvalidVRModeException, InvalidVRException, DicomFormatException
	{
		logger.log( Level.INFO, "Reading indeterminate length sequence element data set." );
		
		// create an input stream that will stop at the SEQUENCE_DELIMITATION_TAG
		DataElementLimitedInputStream inStream = 
			new DataElementLimitedInputStream(
					reader.getInputStream(), 
					DataElementTag.SEQUENCE_DELIMITATION_TAG);
		
		// create a DataSetReader using the same factory (and same transfer syntax) as this data set
		DataSetReader childDataSetReader = new DataSetReader(inStream, reader.getDataElementFactory());
		
		// read the data set
		DataSet dataSet = childDataSetReader.readDataSet();
		
		return dataSet;
	}
	
	private DataSet readDeterminateLengthSequenceElementDataSet(long length) 
	throws IncompatibleValueLengthField, IOException, InvalidVRModeException, InvalidVRException, DicomFormatException
	{
		logger.log( Level.INFO, "Reading determinate length sequence element data set, length is '" + length + "'." );
		if(length == 0)
			return null;
		
		// create an input stream that will stop after the specified number of bytes
		DataElementLimitedInputStream inStream = 
			new DataElementLimitedInputStream(
					reader.getInputStream(), 
					length);
		DataSetReader childDataSetReader = new DataSetReader(inStream, reader.getDataElementFactory());
		DataSet dataSet = childDataSetReader.readDataSet();
		
		return dataSet;
		
	}
}
