package gov.va.med.imaging.dicom.dictionary;

import gov.va.med.imaging.dicom.DataElementTag;
import gov.va.med.imaging.dicom.ValueRepresentation;

import java.io.IOException;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 
 * @author       DNS
 *
 */
public class DicomDictionary
extends HashMap<String, DicomDictionaryEntry>
{
	private static final long serialVersionUID = 1L;
	private static DicomDictionary defaultInstance = null;
	private static DicomDictionary defaultHeaderInstance = null;
	
	public static synchronized DicomDictionary getDefault()
	{
		if(defaultInstance == null)
		{
			InputStreamReader reader = null;
			try
			{
				reader = new InputStreamReader(
					DicomDictionary.class.getClassLoader().getResourceAsStream("DicomDictionary.txt") );
				defaultInstance = DicomDictionaryReaderImpl.create(reader);
			} 
			catch (IOException e)
            {
	            e.printStackTrace();
            }
			finally
			{
				try{reader.close();}catch(Exception x){}
			}
		}
		
		return defaultInstance;
	}
	
	/**
	 * Return the default dictionary for DICOM header information.  This dictionary contains ONLY
	 * the elements that are permitted in the header.
	 * 
	 * @return
	 */
	public static synchronized DicomDictionary getDefaultHeaderDictionary()
	{
		if(defaultHeaderInstance == null)
		{
			InputStreamReader reader = null;
			try
			{
				reader = new InputStreamReader(
					DicomDictionary.class.getClassLoader().getResourceAsStream("DicomHeaderDictionary.txt") );
				defaultHeaderInstance = DicomDictionaryReaderImpl.create(reader);
			} 
			catch (IOException e)
            {
	            e.printStackTrace();
            }
			finally
			{
				try{reader.close();}catch(Exception x){}
			}
		}
		
		return defaultHeaderInstance;
	}
	
	private static String ElementRegex = "[0x28]?0[xX]([\\d]{4}),0[xX]([\\d]{4})[0x29]?";
	private Pattern elementPattern = Pattern.compile(ElementRegex);
	
	public DicomDictionaryEntry getByElement(String elementAsString)
	{
		elementAsString = elementAsString.replace(" ", "");
		Matcher matcher = elementPattern.matcher(elementAsString);
		
		if( matcher.find() )
		{
			int group = Integer.parseInt( matcher.group(1) );
			int element = Integer.parseInt( matcher.group(2) );
			
			return getByElementKey(group, element);
		}
		
		return null;
	}
	
	public DicomDictionaryEntry getByElementKey(DataElementTag elementKey)
	{
		return getByElementKey(elementKey.getGroup(), elementKey.getElement());
	}
	
	public DicomDictionaryEntry getByElementKey(int group, int element)
	{
		for(String dicomElementName : this.keySet())
		{
			DicomDictionaryEntry entry = this.get(dicomElementName);
			if( group >= entry.getGroupStart() && group <= entry.getGroupEnd() &&
				element >= entry.getElementStart() && element <= entry.getElementEnd())
					return entry;
		}
		
		return null;
	}
	
	/**
	 * 
	 * @param elementKey
	 * @return
	 */
	public String createUserEntry(DataElementTag elementKey)
	{
		return createUserEntry(elementKey.getGroup(), elementKey.getElement());
	}
	
	/**
	 * 
	 * @param group
	 * @param element
	 * @return
	 */
	public String createUserEntry(int group, int element)
	{
		String entryName = "UserDefinedElement" + "-" + group + ":" + element;
		DicomDictionaryEntry userEntry = 
			new DicomDictionaryEntry(
					group, group, 
					element, element, 
					entryName, 
					0, Integer.MAX_VALUE, 
					false, 
					ValueRepresentation.NA);
		this.put(entryName, userEntry);
		
		return entryName;
	}
}
