package gov.va.med.imaging.dicom.dataelement;

import java.awt.Point;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferInt;
import java.awt.image.WritableRaster;
import java.util.ArrayList;
import java.util.List;

import gov.va.med.imaging.dicom.DataElement;
import gov.va.med.imaging.dicom.DataElementFactory;
import gov.va.med.imaging.dicom.DataElementTag;
import gov.va.med.imaging.dicom.RawDataToPixelTransformer;
import gov.va.med.imaging.dicom.ValueRepresentation;
import gov.va.med.imaging.dicom.dictionary.DicomDictionaryEntry;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException_OW;
import gov.va.med.imaging.dicom.exceptions.InvalidRawPixelInterpretationException;
import gov.va.med.imaging.dicom.exceptions.RawPixelInterpretationValuesNotSetException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInterpretationException;

/**
 * A string of 16-bit words where the encoding of the contents is specified by the 
 * negotiated Transfer Syntax. OW is a VR that requires byte swapping within each word 
 * when changing between Little Endian and Big Endian byte ordering (see Section 7.3).
 * 
 * @author      DNS
 *
 */
public class DataElement_OW 
extends NativePixelDataElement
{
	public DataElement_OW(DataElementFactory instantiatingFactory, DataElementTag dataElementTag, ValueRepresentation explicitVRField,
	        long valueLength, byte[] value) throws DicomFormatException
	{
		super(instantiatingFactory, dataElementTag, explicitVRField, valueLength, value);
	}

	public DataElement_OW(DataElementFactory instantiatingFactory, DataElementTag dataElementTag, DicomDictionaryEntry dictionaryEntry,
	        long valueLength, byte[] value) throws DicomFormatException
	{
		super(instantiatingFactory, dataElementTag, dictionaryEntry, valueLength, value);
	}

	/**
     * Always returns the raw byte value array.
     * Use the PixelData methods to get interpreted data.
     * 
	 * @see gov.va.med.imaging.dicom.DataElement#getValue()
	 */
	@Override
	public byte[] getValue() 
	throws ValueRepresentationInterpretationException
	{
		return getRawValue();
	}

	/**
	 * Parsing the raw value just does the byte swapping, if required by the transfer syntax.
	 * Otherwise it does nothing.
	 * 
	 * @see gov.va.med.imaging.dicom.DataElement#parseRawValue()
	 */
	@Override
	protected void parseRawValue() 
	throws DicomFormatException
	{
		if(getRawValue() == null)
			setRawValue(new byte[0]);
		else if(getRawValue().length % 2 > 0)
			throw new DicomFormatException_OW("The length of the raw value array must be an even number of bytes and it is not.");
		else
		{
			getInstantiatingFactory().getTransferSyntaxUid().intepretByteArrayAsWordArray(getRawValue());
		}
	}

	// ====================================================================================================
	// Raw Pixel Data Handling, OW does not do encapsulated image handling
	// ====================================================================================================
	private int bitsAllocated = 0;
	private int bitsStored = 0;
	private int highBit = 0;
	private int width = 0;
	private int height = 0;
	
	private List<BufferedImage> rawPixelData = null;
	
	/**
	 * 
	 * @see gov.va.med.imaging.dicom.dataelement.NativePixelDataElement#setNativePixelInterpretationParameters(int, int, int)
	 */
	@Override
    public void setNativePixelInterpretationParameters(
    		int bitsAllocated, 
    		int bitsStored, 
    		int highBit,
    		int width,
    		int height) 
	throws DicomFormatException
    {
		if(bitsAllocated > 16)
			throw new InvalidRawPixelInterpretationException("Invalid bit allocation for value representation of 'OB'");
		
		this.bitsAllocated = bitsAllocated;
		this.bitsStored = bitsStored;
		this.highBit = highBit;
		this.width = width;
		this.height = height;
    }

	/**
	 * Returns an array of int, one for each pixel, each is a maximum of 8 significant bits
	 * per pixel (unsigned values, hence the wasted bits)
	 * @see gov.va.med.imaging.dicom.dataelement.NativePixelDataElement#getRawPixelData(int, int, int)
	 */
	public synchronized List<BufferedImage> getNativePixelData()
	throws DicomFormatException, RawPixelInterpretationValuesNotSetException
	{
		if(getInstantiatingFactory().getTransferSyntaxUid().isEncapsulated())
			return null;
		
		if( rawPixelData == null)
			rawPixelData = parseNativePixelData();
		
		return rawPixelData;
	}
	
	@Override
    protected int getWordSize()
    {
	    return 16;	// operates on single byte
    }

	/**
	 * If native, it shall have a defined Value Length, and be encoded as follows:
	 * - where Bits Allocated (0028,0100) has a value greater than 8 shall have 
	 *   Value Representation OW and shall be encoded in Little Endian;
	 * - where Bits Allocated (0028,0100) has a value less than or equal to 8 shall have the 
	 *   Value Representation OB or OW and shall be encoded in Little Endian.
 	 * @return
	 * @throws RawPixelInterpretationValuesNotSetException
	 * @throws DicomFormatException 
	 */
	private List<BufferedImage> parseNativePixelData() 
	throws RawPixelInterpretationValuesNotSetException, DicomFormatException
	{
		if(bitsAllocated == 0 || bitsStored == 0)
			throw new RawPixelInterpretationValuesNotSetException();
		
		List<BufferedImage> frames = new ArrayList<BufferedImage>();
		frames.add( parseImage(getRawValue(), this.height, this.width, this.bitsStored, this.bitsAllocated, this.highBit) );
		
		return frames;
	}

	/**
     * The raw data is organized in chunks specified by the bitsAllocated parameter.
     * The only guarantee is that the bitsAllocated must be less than 8 bits (else the
     * image must be stored in a OW element.
     * If bitsAllocated is less than 8 then the 
	 * @throws DicomFormatException 
     */
    private static BufferedImage parseImage(
    	byte[] rawData, 
    	int height, 
    	int width, 
    	int bitsStored, 
    	int bitsAllocated, 
    	int highBit) 
    throws DicomFormatException
    {
    	RawDataToPixelTransformer transformer = new RawDataToPixelTransformer(true);
    	
	    int[] destination = transformer.transformRawByteDataToIntPixelArray(
	    	rawData, 
	    	height, width, 
	    	bitsAllocated, bitsStored, highBit);

	    DataBuffer dataBuffer = new DataBufferInt(destination, destination.length);
		WritableRaster imageRaster = 
			WritableRaster.createPackedRaster(dataBuffer, width, height, bitsStored, new Point(0,0));
		
		//BufferedImage image = new BufferedImage(createColorModel(ColorSpace.CS_GRAY, 16), imageRaster, true, null);
		BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_USHORT_GRAY);
		
		image.setData(imageRaster);
		
		return image;
    }
}
