package gov.va.med.imaging.dicom.dataelement;

import gov.va.med.imaging.dicom.DataElement;
import gov.va.med.imaging.dicom.DataElementFactory;
import gov.va.med.imaging.dicom.DataElementTag;
import gov.va.med.imaging.dicom.ValueRepresentation;
import gov.va.med.imaging.dicom.dictionary.DicomDictionaryEntry;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException_OF;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInterpretationException;

/**
 * A string of 32-bit IEEE 754:1985 floating point words. 
 * OF is a VR which requires byte swapping within each 32-bit word when changing between 
 * Little Endian and Big Endian byte ordering (see Section 7.3).
 * 
 * @author       DNS
 *
 */
public class DataElement_OF extends DataElement<Float[]>
{
	private Float[] value;
	
	public DataElement_OF(DataElementFactory instantiatingFactory, DataElementTag dataElementTag, ValueRepresentation explicitVRField,
	        long valueLength, byte[] value) throws DicomFormatException
	{
		super(instantiatingFactory, dataElementTag, explicitVRField, valueLength, value);
	}

	public DataElement_OF(DataElementFactory instantiatingFactory, DataElementTag dataElementTag, DicomDictionaryEntry dictionaryEntry,
	        long valueLength, byte[] value) throws DicomFormatException
	{
		super(instantiatingFactory, dataElementTag, dictionaryEntry, valueLength, value);
	}

	@Override
	public Float[] getValue() 
	throws ValueRepresentationInterpretationException
	{
		return value;
	}

	@Override
	protected void parseRawValue() 
	throws DicomFormatException
	{
		if(getRawValue() == null)
			value = new Float[0];
		else if(getRawValue().length % 4 > 0)
			throw new DicomFormatException_OF();
		else
		{
			int count = getRawValue().length / 4;
			value = new Float[count];
			for(int index = 0; index < count; index++)
				value[index] = 
					getInstantiatingFactory().getTransferSyntaxUid().makeFloatFrom4Bytes(getRawValue(), index * 4);
		}
	}

}
