/**
 * Package: MAG - VistA Imaging
 * WARNING: Per VHA Directive 2004-038, this routine should not be modified.
 * Date Created: Aug 7, 2008
 * Site Name:  Washington OI Field Office, Silver Spring, MD
 * @author DNS
 * @version 1.0
 *
 * ----------------------------------------------------------------
 * Property of the US Government.
 * No permission to copy or redistribute this software is given.
 * Use of unreleased versions of this software requires the user
 * to execute a written test agreement with the VistA Imaging
 * Development Office of the Department of Veterans Affairs,
 * telephone DNS.
 * 
 * The Food and Drug Administration classifies this software as
 * a Class II medical device.  As such, it may not be changed
 * in any way.  Modifications to this software may result in an
 * adulterated medical device under 21CFR820, the use of which
 * is considered to be a violation of US Federal Statutes.
 * ----------------------------------------------------------------
 */
package gov.va.med.imaging.dicom.dataelement;

import gov.va.med.imaging.dicom.DataElement;
import gov.va.med.imaging.dicom.DataElementFactory;
import gov.va.med.imaging.dicom.DataElementTag;
import gov.va.med.imaging.dicom.RawDataToPixelTransformer;
import gov.va.med.imaging.dicom.TransferSyntaxUid;
import gov.va.med.imaging.dicom.ValueRepresentation;
import gov.va.med.imaging.dicom.dictionary.DicomDictionaryEntry;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.InvalidRawPixelInterpretationException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;
import gov.va.med.imaging.dicom.exceptions.RawPixelInterpretationValuesNotSetException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInvalidDataLengthException;
import gov.va.med.imaging.dicom.io.DataElementLimitedInputStream;
import gov.va.med.imaging.dicom.io.DataElementReader;

import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferInt;
import java.awt.image.WritableRaster;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.imageio.ImageIO;

/**
 * @author DNS
 *
 * A string of bytes where the encoding of the contents is specified by the negotiated Transfer Syntax. 
 * OB is a VR which is insensitive to Little/Big Endian byte ordering (see Section 7.3). 
 * The string of bytes shall be padded with a single trailing NULL byte value (00H) when necessary to achieve 
 * even length.
 * 
 * Data Element (7FE0,0010) Pixel Data may be encapsulated or native.
 * If native, it shall have a defined Value Length, and be encoded as follows:
 * - where Bits Allocated (0028,0100) has a value greater than 8 shall have 
 *   Value Representation OW and shall be encoded in Little Endian;
 * - where Bits Allocated (0028,0100) has a value less than or equal to 8 shall have the 
 *   Value Representation OB or OW and shall be encoded in Little Endian.
 *   
 * If encapsulated, it has the Value Representation OB and is a sequence of bytes resulting from one of the 
 * encoding processes. It contains the encoded pixel data stream fragmented into one or more Item(s). 
 * This Pixel Data Stream may represent a Single or Multi-frame Image. See Tables A.4-1 and A.4-2:
 * - The Length of the Data Element (7FE0,0010) shall be set to the Value for Undefined Length (FFFFFFFFH).
 * - Each Data Stream Fragment encoded according to the specific encoding process shall be encapsulated as a 
 *   DICOM Item with a specific Data Element Tag of Value (FFFE,E000). The Item Tag is followed by a 4 byte 
 *   Item Length field encoding the explicit number of bytes of the Item.
 * - All items containing an encoded fragment shall be made of an even number of bytes greater or equal to two. 
 *   The last fragment of a frame may be padded, if necessary, to meet the sequence item format requirements of the 
 *   DICOM Standard.
 */
public class DataElement_OB 
extends NativePixelDataElement
implements EncapsulatedPixelDataElement
{
	/**
	 * @param instantiatingFactory
	 * @param dataElementTag
	 * @param explicitVRField
	 * @param valueLength
	 * @param value
	 * @throws DicomFormatException 
	 */
	public DataElement_OB(
		DataElementFactory instantiatingFactory, 
		DataElementTag dataElementTag, 
		ValueRepresentation explicitVRField,
        long valueLength, 
        byte[] value) 
	throws DicomFormatException
	{
		super(instantiatingFactory, dataElementTag, explicitVRField, valueLength, value);
	}

	/**
	 * @param instantiatingFactory
	 * @param dataElementTag
	 * @param dictionaryEntry
	 * @param valueLength
	 * @param value
	 * @throws DicomFormatException 
	 */
	public DataElement_OB(
		DataElementFactory instantiatingFactory, 
		DataElementTag dataElementTag, 
		DicomDictionaryEntry dictionaryEntry,
	    long valueLength, 
	    byte[] value) 
	throws DicomFormatException
	{
		super(instantiatingFactory, dataElementTag, dictionaryEntry, valueLength, value);
	}
	
	/**
     * Always returns the raw byte value array.
     * Use the PixelData methods to get interpreted data.
     * 
     * @see gov.va.med.imaging.dicom.DataElement#getValue()
     */
    @Override
    public byte[] getValue()
    {
    	return getRawValue();
    }

	@Override
    protected void parseRawValue() 
	throws DicomFormatException
	{
		// do nothing, 
		// for pixel data use getPixelData() to get the image
		// otherwise this is an identity DataElement
	}

	/**
     * @see gov.va.med.imaging.dicom.dataelement.PixelDataElement#getImageFrames()
     */
	public List<BufferedImage> getImageFrames() 
	throws DicomFormatException, IOException, InvalidVRModeException, InvalidVRException, RawPixelInterpretationValuesNotSetException
	{
		if(this.getInstantiatingFactory().getTransferSyntaxUid().isEncapsulated())
			return getEncapsulatedPixelData();
		else
			return getNativePixelData();
	}
	
	//====================================================================================================
	// Encapsulated Pixel Data (JPG, JP2) Handling
	//====================================================================================================
	private List<BufferedImage> frames = null;
	
	public synchronized List<BufferedImage> getEncapsulatedPixelData() 
	throws DicomFormatException, IOException, InvalidVRModeException, InvalidVRException
	{
		// if the pixel data is not encapsulated (i.e. is a raw bitmap) return null from this method
		// use getRawPixelData()
		if(! getInstantiatingFactory().getTransferSyntaxUid().isEncapsulated())
			return null;
		
		// parse this on demand as it is an expensive operation
		if(frames == null)
			frames = parseEncapsulatedPixelData();
		
		return frames;
	}

	/**
	 * Return a List of byte arrays.
	 * Each list element is a frame.
	 * Each frame may contain many fragments but this method concatenates them together into one buffer.
	 * The byte arrays are the content of the fragments.
	 * 
	 * @return
	 * @throws DicomFormatException
	 * @throws IOException
	 * @throws InvalidVRModeException
	 * @throws InvalidVRException
	 */
    private List<BufferedImage> parseEncapsulatedPixelData() 
	throws DicomFormatException, IOException, InvalidVRModeException, InvalidVRException
    {
    	List<BufferedImage> frameList = null;
		List<Long> offsetTable = null;
        TransferSyntaxUid transferSyntax = getInstantiatingFactory().getTransferSyntaxUid();
		String imageMimeType = transferSyntax.getMimeType();
		
        if(transferSyntax.isEncapsulated())
        {
        	// Spec 3.5-2007 Pg 64
        	if(getValueLength() != DataElement.INDETERMINATE_LENGTH)
        		throw new ValueRepresentationInvalidDataLengthException("An encapsulated transfer syntax pixel data MUST have a length of 0xffffffff.");
        	
        	DataElementLimitedInputStream parsingInputStream = 
        		new DataElementLimitedInputStream(new DataInputStream(new ByteArrayInputStream(getRawValue())), getInstantiatingFactory().getTransferSyntaxUid());
        	
        	DataElementReader pixelDataReader = new DataElementReader(parsingInputStream, getInstantiatingFactory());
        	DataElement<?> offsetTableDataElement = pixelDataReader.readNextDataElement();
        	offsetTable = parseOffsetTable(offsetTableDataElement);
        	
        	/*
        	 * If the offset table is non-existent or is zero-length then there is one frame
        	 * in the data set.  The frame MAY BE spread over multiple fragments, each divided by
        	 * ITEM_DELIMITATION_TAG elements.
        	 * See PS 3.5-2007 Page 66 for an example.
        	 */
        	if(offsetTable == null || offsetTable.size() == 0)
        	{
        		getLogger().finer("DataElement_OB, found no offset table, parsing pixel data as 1 fragment in 1 frame.");
        		frameList = new ArrayList<BufferedImage>(1);
        		
        		List<byte[]> fragments = new ArrayList<byte[]>();
        		for( DataElement<?> pixelDataFragment = pixelDataReader.readNextDataElement();
        			pixelDataFragment != null;
        			pixelDataFragment = pixelDataReader.readNextDataElement() )
        				fragments.add( pixelDataFragment.getRawValue() );
        		
        		InputStream imageInputStream = new ByteArrayListInputStream(fragments);
        		BufferedImage frame = ImageIO.read(imageInputStream);
        		
        		frameList.add(frame);
        	}
        	else
        	{
        		// iterate through each data item in the element
        		// each data item is an image fragment, the offset table 
        		// specifies the offset at the start of each frame.
        		frameList = new ArrayList<BufferedImage>(offsetTable.size());
        		
        		Iterator<Long> offsetTableIterator = offsetTable.iterator();
        		Long nextOffset = offsetTableIterator.next();
        		long totalOffset = 0L;
        		
        		List<byte[]> fragments = new ArrayList<byte[]>();
	        	for( DataElement<?> pixelDataFragment = pixelDataReader.readNextDataElement();
	        		pixelDataFragment != null;
	        		pixelDataFragment = pixelDataReader.readNextDataElement() )
	        	{
	        		if(totalOffset >= nextOffset.longValue())
	        		{
	        			if(fragments.size() > 0)
	        			{
	                		InputStream imageInputStream = new ByteArrayListInputStream(fragments);
	                		BufferedImage frame = ImageIO.read(imageInputStream);
	        				frameList.add(frame);
	        			}
	        			fragments.clear();
	        		}
    				fragments.add( pixelDataFragment.getRawValue() );
	        		totalOffset += pixelDataFragment.getValueLength();
	        	}
	        	// add the last frame
    			if(fragments.size() > 0)
    			{
            		InputStream imageInputStream = new ByteArrayListInputStream(fragments);
            		BufferedImage frame = ImageIO.read(imageInputStream);
    				frameList.add(frame);
    			}
        	}
        }
		
		return frameList;
    }
	
	private List<Long> parseOffsetTable(DataElement<?> offsetTableDataElement)
	{
		if(offsetTableDataElement.getValueLength() == 0)
			return null;
		List<Long> fragmentOffsets = new ArrayList<Long>();
		
		TransferSyntaxUid transferSyntax = getInstantiatingFactory().getTransferSyntaxUid();
		byte[] rawOffsetTable = offsetTableDataElement.getRawValue();
		for( int fragmentIndex = 0; fragmentIndex < rawOffsetTable.length/4; ++fragmentIndex )
			fragmentOffsets.add( new Long(transferSyntax.makeUnsignedLongFrom4Bytes(rawOffsetTable, fragmentIndex * 4)) );
		
		return fragmentOffsets;
	}

	// ====================================================================================================
	// Raw Pixel Data Handling
	// ====================================================================================================
	private int bitsAllocated = 0;
	private int bitsStored = 0;
	private int highBit = 0;
	private int width = 0;
	private int height = 0;
	
	private List<BufferedImage> rawPixelData = null;
	
	@Override
    protected int getWordSize()
    {
	    return 8;	// operates on single byte
    }

	/**
	 * 
	 * @see gov.va.med.imaging.dicom.dataelement.NativePixelDataElement#setNativePixelInterpretationParameters(int, int, int)
	 */
	@Override
    public void setNativePixelInterpretationParameters(
    		int bitsAllocated, 
    		int bitsStored, 
    		int highBit,
    		int width,
    		int height) 
	throws DicomFormatException
    {
		if(bitsAllocated > 8)
			throw new InvalidRawPixelInterpretationException("Invalid bit allocation for value representation of 'OB'");
		
		this.bitsAllocated = bitsAllocated;
		this.bitsStored = bitsStored;
		this.highBit = highBit;
		this.width = width;
		this.height = height;
    }

	/**
	 * Returns an array of int, one for each pixel, each is a maximum of 8 significant bits
	 * per pixel (unsigned values, hence the wasted bits)
	 * @see gov.va.med.imaging.dicom.dataelement.NativePixelDataElement#getRawPixelData(int, int, int)
	 */
	public synchronized List<BufferedImage> getNativePixelData()
	throws DicomFormatException, RawPixelInterpretationValuesNotSetException
	{
		if(getInstantiatingFactory().getTransferSyntaxUid().isEncapsulated())
			return null;
		
		if( rawPixelData == null)
			rawPixelData = parseNativePixelData();
		
		return rawPixelData;
	}
	
	/**
	 * If native, it shall have a defined Value Length, and be encoded as follows:
	 * - where Bits Allocated (0028,0100) has a value greater than 8 shall have 
	 *   Value Representation OW and shall be encoded in Little Endian;
	 * - where Bits Allocated (0028,0100) has a value less than or equal to 8 shall have the 
	 *   Value Representation OB or OW and shall be encoded in Little Endian.
 	 * @return
	 * @throws RawPixelInterpretationValuesNotSetException
	 */
	private List<BufferedImage> parseNativePixelData() 
	throws RawPixelInterpretationValuesNotSetException, DicomFormatException
	{
		if(bitsAllocated == 0 || bitsStored == 0)
			throw new RawPixelInterpretationValuesNotSetException();
		
		List<BufferedImage> frames = new ArrayList<BufferedImage>();
		frames.add( parseImage(getRawValue(), this.height, this.width, this.bitsStored, this.bitsAllocated, this.highBit) );
		
		return frames;
	}

	/**
     * The raw data is organized in chunks specified by the bitsAllocated parameter.
     * The only guarantee is that the bitsAllocated must be less than 8 bits (else the
     * image must be stored in a OW element.
     * If bitsAllocated is less than 8 then the 
	 * @throws DicomFormatException 
     */
    private static BufferedImage parseImage(
    	byte[] rawData, 
    	int height, 
    	int width, 
    	int bitsStored, 
    	int bitsAllocated, 
    	int highBit) 
    throws DicomFormatException
    {
    	RawDataToPixelTransformer transformer = new RawDataToPixelTransformer();
    	
	    int[] destination = transformer.transformRawByteDataToIntPixelArray(
	    	rawData, 
	    	height, width, 
	    	bitsAllocated, bitsStored, highBit);
		
	    DataBuffer dataBuffer = new DataBufferInt(destination, destination.length);
		WritableRaster imageRaster = WritableRaster.createPackedRaster(dataBuffer, width, height, bitsStored, new Point(0,0));
		BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_BYTE_GRAY);
		
		image.setData(imageRaster);
		
		return image;
    }
    
    class ByteArrayListInputStream
    extends InputStream
    {
    	private final List<ByteArrayInputStream> fragmentStreams;
    	private int streamIndex = 0;
    	
    	ByteArrayListInputStream(List<byte[]> fragments)
    	{
    		this.fragmentStreams = new ArrayList<ByteArrayInputStream>();
    		for(byte[] fragmentBuffer : fragments)
    			fragmentStreams.add(new ByteArrayInputStream(fragmentBuffer));
    	}

		@Override
        public int read() throws IOException
        {
			int value = -1;
			while(value < 0 && streamIndex < fragmentStreams.size())
			{
				value = fragmentStreams.get(streamIndex).read();
				if(value < 0) ++streamIndex;
			}
			
	        return value;
        }
    }
}
