package gov.va.med.imaging.dicom;

/**
 * The permissable values for the VR in the DICOM Data Dictionary.
 * 
 * The Value Representation of a Data Element describes the data type and format of that 
 * Data Element's Value(s). PS 3.6 lists the VR of each Data Element by Data Element Tag.
 * 
 * Values with VRs constructed of character strings, except in the case of the VR UI, shall 
 * be padded with SPACE characters (20H, in the Default Character Repertoire) when necessary 
 * to achieve even length. Values with a VR of UI shall be padded with a single trailing NULL (00H) 
 * character when necessary to achieve even length. Values with a VR of OB shall be padded with a 
 * single trailing NULL byte value (00H) when necessary to achieve even length.
 * 
 * UNDEFINED LENGTH: The ability to specify an unknown length for a Data Element Value 
 * (of Value Representation SQ, OW, or OB) or Item. Data Elements and Items of Undefined Length are 
 * delimited with Sequence Delimitation Items and Item Delimiter Data Elements, respectively.
 * 
 * @author       DNS
 *
 * @see DICOM Part 5
 */
public enum ValueRepresentation
{
	AE( "Application Entity", 
		"A string of characters that identifies an Application Entity with leading and trailing spaces (20H) being non-significant." +
		"A value consisting solely of spaces shall not be used.",  
		"Default Character Repertoire excluding character code 5CH (the BACKSLASH \\ in ISO-IR 6), and control characters LF, FF, CR and ESC.",
		1, 16,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_AE.class),
	AS( "Age String",
		"A string of characters with one of the following formats -- nnnD, nnnW, nnnM, nnnY; where nnn shall contain the " +
		"number of days for D, weeks for W, months for M, or years for Y." + 
		"Example: 018M would represent an age of 18 months.", 
		"0-9, D, W, M, Y of Default Character Repertoire",
		4, 4,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_AS.class),
	AT( "Attribute Tag",
		"Ordered pair of 16-bit unsigned integers that is the value of a Data Element Tag." +
		"Example: A Data Element Tag of (0018,00FF) would be encoded as a series of 4 bytes in a Little-Endian Transfer Syntax as 18H,00H,FFH,00H and in a Big-Endian Transfer Syntax as 00H,18H,00H,FFH." +
		"Note: The encoding of an AT value is exactly the same as the encoding of a Data Element Tag as defined in Section 7.",
		"not applicable",
		4, 4,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_AT.class),	
	CS( "Code String",
		"A string of characters with leading or trailing spaces (20H) being non-significant.",
		"Uppercase characters, 0-9, the SPACE character, and underscore _, of the Default Character Repertoire",
		0, 16,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_CS.class),
	DA( "Date",
		"A string of characters of the format yyyymmdd; where yyyy shall contain year, mm shall contain the month, and dd shall contain the day. This conforms to the ANSI HISPP MSDS Date common data type." +
		"Example: 19930822 would represent August 22, 1993." +
		"Notes: 1. For reasons of backward compatibility with versions of this standard prior to V3.0, it is recommended that implementations also support a string of characters of the format yyyy.mm.dd for this VR." +
		"2. See also DT VR in this table.",
		"0-9 of Default Character Repertoire. Note: For reasons specified in the previous column, implementations may wish to support the . character as well.",
		8, 10,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_DA.class),
	DS( "Decimal String",
		"A string of characters representing either a fixed point number or a floating point number. A fixed point number shall contain only the characters 0-9 with an optional leading '+' or '-' and an optional '.' to mark the decimal point. A floating point number shall be conveyed as defined in ANSI X3.9, with an 'E' or 'e' to indicate the start of the exponent. Decimal Strings may be padded with leading or trailing spaces. Embedded spaces are not allowed." + 
		"Note: Data Elements with multiple values using this VR may not be properly encoded if Explicit-VR transfer syntax is used and the VL of this attribute exceeds 65534 bytes.",
		"'0'-'9', '+', '-', E, 'e', '.' of Default Character Repertoire",
		0, 16,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_DS.class),
	DT( "Date Time",
		"The Date Time common data type. Indicates a concatenated date-time ASCII string in the format: YYYYMMDDHHMMSS.FFFFFF&ZZZZThe components of this string, from left to right, are YYYY = Year, MM = Month, DD = Day, HH = Hour, MM = Minute, SS = Second, FFFFFF = Fractional Second, & = + or -, and ZZZZ = Hours and Minutes of offset. &ZZZZ is an optional suffix for plus/minus offset from Coordinated Universal Time. A component that is omitted from the string is termed a null component. Trailing null components of Date Time are ignored. Non-trailing null components are prohibited, given that the optional suffix is not considered as a component." +
		"Note: For reasons of backward compatibility with versions of this standard prior to V3.0, many existing DICOM Data Elements use the separate DA and TM VRs. Standard and Private Data Elements defined in the future should use DT, when appropriate, to be more compliant with ANSI HISPP MSDS.",
		"'0'-'9', '+', '-', '.' of Default Character Repertoire",
		0, 26,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_DT.class),
	FD( "Floating Point Double",
		"Double precision binary floating point number represented in IEEE 754:1985 64-bit Floating Point Number Format.",
		"not applicable",
		8, 8,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_FD.class),
	FL( "Floating Point Single",
		"Single precision binary floating point number represented in IEEE 754:1985 32-bit Floating Point Number Format.",
		"not applicable",
		4, 4,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_FL.class),
	IS( "Integer String",
		"A string of characters representing an Integer in base-10 (decimal), shall contain only the characters 0 - 9, with an optional leading '+' or '-'. It may be padded with leading and/or trailing spaces. Embedded spaces are not allowed." +
		"range: -231 <= n <= (231 - 1)", 
		"'0'-'9', '+', '-' of Default Character Repertoire",
		0, 12,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_IS.class),
	LO( "Long String",
		"A character string that may be padded with leading and/or trailing spaces. The character code 5CH (the BACKSLASH '\' in ISO-IR 6) shall not be present, as it is used as the delimiter between values in multiple valued data elements. The string shall not have Control Characters except for ESC.",
		"Default Character Repertoire and/or as defined by (0008,0005).",
		0, 64,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_LO.class),
	LT( "Long Text",
		"A character string that may contain one or more paragraphs. It may contain the Graphic Character set and the Control Characters, CR, LF, FF, and ESC. It may be padded with trailing spaces, which may be ignored, but leading spaces are considered to be significant. Data Elements with this VR shall not be multi-valued and therefore character code 5CH (the BACKSLASH '\' in ISO-IR 6) may be used.",
		"Default Character Repertoire and/or as defined by (0008,0005).",
		0, 10240,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_LT.class),
	NA( "Not Applicable",
		"Not a real value representation, used for vendor specific fields",
		"",
		0, Integer.MAX_VALUE,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_NA.class),
	OB( "Other Byte String",
		"A string of bytes where the encoding of the contents is specified by the negotiated Transfer Syntax. OB is a VR which is insensitive to Little/Big Endian byte ordering (see Section 7.3). The string of bytes shall be padded with a single trailing NULL byte value (00H) when necessary to achieve even length.",
		"not applicable",
		0, Long.MAX_VALUE,
		true,
		LengthField.LONG,
		gov.va.med.imaging.dicom.dataelement.DataElement_OB.class),
	OF( "Other Float String",
		"A string of 32-bit IEEE 754:1985 floating point words. OF is a VR which requires byte swapping within each 32-bit word when changing between Little Endian and Big Endian byte ordering (see Section 7.3).",
		"not applicable",
		0, (2^32)-4,
		false,
		LengthField.LONG,
		gov.va.med.imaging.dicom.dataelement.DataElement_OF.class),
	OW( "Other Word String",
		"A string of 16-bit words where the encoding of the contents is specified by the negotiated Transfer Syntax. OW is a VR that requires byte swapping within each word when changing between Little Endian and Big Endian byte ordering (see Section 7.3).",
		"not applicable",
		0, Long.MAX_VALUE,
		true,
		LengthField.LONG,
		gov.va.med.imaging.dicom.dataelement.DataElement_OW.class),
	PN( "Person Name", 
		"A character string encoded using a 5 component convention. The character code 5CH (the BACKSLASH '\' in ISO-IR 6) shall not be present, as it is used as the delimiter between values in multiple valued data elements. The string may be padded with trailing spaces. For human use, the five components in their order of occurrence are: family name complex, given name complex, middle name, name prefix, name suffix. Any of the five components may be an empty string. The component delimiter shall be the caret ^ character (5EH). Delimiters are required for interior null components. Trailing null components and their delimiters may be omitted. " +
		"Multiple entries are permitted in each component and are encoded as natural text strings, in the format preferred by the named person. This conforms to the ANSI HISPP MSDS Person Name common data type." +
		"For veterinary use, the first two of the five components in their order of occurrence are: responsible party family name or responsible organization name, patient name. The remaining components are not used and shall not be present." +
		"This group of five components is referred to as a Person Name component group." +
		"For the purpose of writing names in ideographic characters and in phonetic characters, up to 3 groups of components (see Annex H examples 1 and 2) may be used. The delimiter for component groups shall be the equals character = (3DH). The three component groups of components in their order of occurrence are: a single-byte character representation, an ideographic representation, and a phonetic representation." +
		"Any component group may be absent, including the first component group. In this case, the person name may start with one or more = delimiters. Delimiters are required for interior null component groups. Trailing null component groups and their delimiters may be omitted." +
		"Precise semantics are defined for each component group. See section 6.2.1.",
		"Default Character Repertoire and/or as defined by (0008,0005) excluding Control Characters LF, FF, and CR but allowing Control Character ESC.",
		0, 64,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_PN.class), 
	SH( "Short String", 
		"A character string that may be padded with leading and/or trailing spaces. The character code 05CH (the BACKSLASH '\' in ISO-IR 6) shall not be present, as it is used as the delimiter between values for multiple data elements. The string shall not have Control Characters except ESC.",
		"Default Character Repertoire and/or as defined by (0008,0005).",
		0, 16,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_SH.class),
	SL( "Signed Long", 
		"Signed binary integer 32 bits long in 2's complement form. Represents an integer, n, in the range: -231 <= n <= (231 - 1).",
		"not applicable",
		4, 4,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_SL.class),
	SQ( "Sequence of Items",
		"Value is a Sequence of zero or more Items, as defined in Section 7.5.",
		"not applicable (see Section 7.5)",
		0, Long.MAX_VALUE,
		true,
		LengthField.LONG,
		gov.va.med.imaging.dicom.dataelement.DataElement_SQ.class),
	SS( "Signed Short",
		"Signed binary integer 16 bits long in 2's complement form. Represents an integer n in the range: -2^15 <= n <= (2^15 - 1).",
		"not applicable",
		2, 2,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_SS.class), 
	ST( "Short Text",
		"A character string that may contain one or more paragraphs. It may contain the Graphic Character set and the Control Characters, CR, LF, FF, and ESC. It may be padded with trailing spaces, which may be ignored, but leading spaces are considered to be significant. Data Elements with this VR shall not be multi-valued and therefore character code 5CH (the BACKSLASH '\' in ISO-IR 6) may be used.",
		"Default Character Repertoire and/or as defined by (0008,0005).",
		0, 1024,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_ST.class),
	TM( "Time", 
		"A string of characters of the format hhmmss.frac; where hh contains hours (range '00' - '23'), mm contains minutes (range '00' - '59'), ss contains seconds (range '00' - '59'), and frac contains a fractional part of a second as small as 1 millionth of a second (range '000000' - '999999'). A 24 hour clock is assumed. Midnight can be represented by only '0000' since '2400' would violate the hour range. The string may be padded with trailing spaces. Leading and embedded spaces are not allowed. One or more of the components mm, ss, or frac may be unspecified as long as every component to the right of an unspecified component is also unspecified. If frac is unspecified the preceding '.' may not be included. Frac shall be held to six decimal places or less to ensure its format conforms to the ANSI HISPP MSDS Time common data type.",
		"'0'-'9', '.' of Default Character Repertoire",
		0, 16,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_TM.class),
	UI( "Unique Identifier (UID)",
		"A character string containing a UID that is used to uniquely identify a wide variety of items. The UID is a series of numeric components separated by the period '.' character. If a Value Field containing one or more UIDs is an odd number of bytes in length, the Value Field shall be padded with a single trailing NULL (00H) character to ensure that the Value Field is an even number of bytes in length. See Section 9 and Annex B " + 
		"for a complete specification and examples.",
		"'0'-'9', '.' of Default Character Repertoire",
		0, 64,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_UI.class),
	UL( "Unsigned Long",
		"Unsigned binary integer 32 bits long. Represents an integer n in the range: 0 <= n < 232.",
		"not applicable",
		4, Long.MAX_VALUE,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_UL.class),
	UN( "Unknown", 
		"A string of bytes where the encoding of the contents is unknown (see Section 6.2.2).",
		"not applicable",
		0, Long.MAX_VALUE,
		false,
		LengthField.LONG,
		gov.va.med.imaging.dicom.dataelement.DataElement_UN.class),
	US( "Unsigned Short",
		"Unsigned binary integer 16 bits long. Represents integer n in the range: 0 <= n < 2^16.",
		"not applicable",
		2, Long.MAX_VALUE,
		false,
		LengthField.SHORT,
		gov.va.med.imaging.dicom.dataelement.DataElement_US.class),
	UT( "Unlimited Text", 
		"A character string that may contain one or more paragraphs. It may contain the Graphic Character set and the Control Characters, CR, LF, FF, and ESC. It may be padded with trailing spaces, which may be ignored, but leading spaces are considered to be significant. Data Elements with this VR shall not be multi-valued and therefore character code 5CH (the BACKSLASH '\\' in ISO-IR 6) may be used.",
		"Default Character Repertoire and/or as defined by (0008,0005).",
		0, (2^32)-2,
		false,
		LengthField.LONG,
		gov.va.med.imaging.dicom.dataelement.DataElement_UT.class);
	
	/**
	 * A little enum that describes how many bits the length field is expected to be.
	 * 
	 */
	enum LengthField 
	{
		SHORT(16), LONG(32);
		
		private int bitCount;
		
		LengthField(int bitCount)
		{
			this.bitCount = bitCount;
		}
		
		public int getBitCount(){return bitCount;}
		public int getByteCount(){return getBitCount() / 8;}
	};
	
	//===========================================================================================
	// 
	//===========================================================================================
	private final String name;
	private final String definition;
	private final String characterRepertoire;
	private final long minLengthOfValue;
	private final long maxLengthOfValue;
	private final boolean allowUndefinedLength;
	private final LengthField lengthField;
	private final Class<? extends DataElement<?>> dataElementClass; 
	
	ValueRepresentation(
			String name, 
			String definition, 
			String characterRepertoire, 
			long minLengthOfValue, long maxLengthOfValue,
			boolean allowUndefinedLength,
			LengthField lengthField,
			Class<? extends DataElement<?>> dataElementClass)
	{
		this.name = name;
		this.definition = definition;
		this.characterRepertoire = characterRepertoire;
		this.minLengthOfValue = minLengthOfValue;
		this.maxLengthOfValue = maxLengthOfValue;
		this.allowUndefinedLength = allowUndefinedLength;
		this.lengthField = lengthField;
		this.dataElementClass = dataElementClass;
	}

	/**
     * @return the name
     */
    public String getName()
    {
    	return name;
    }

	/**
     * @return the definition
     */
    public String getDefinition()
    {
    	return definition;
    }

	/**
     * @return the characterRepertoire
     */
    public String getCharacterRepertoire()
    {
    	return characterRepertoire;
    }

	/**
     * @return the minLengthOfValue
     */
    public long getMinLengthOfValue()
    {
    	return minLengthOfValue;
    }

	/**
     * @return the maxLengthOfValue
     */
    public long getMaxLengthOfValue()
    {
    	return maxLengthOfValue;
    }

    /**
     * @return the allowUndefinedLength
     */
    public boolean isAllowUndefinedLength()
    {
    	return allowUndefinedLength;
    }

    /**
     * Return the expected data length field length in bytes (either 2 or 4).
     * @return
     */
    public int getExpectedDataLengthFieldLength()
    {
    	return (this.lengthField.bitCount / 8);
    }
    
	@Override
    public String toString()
    {
	    return getName();
    }
    
    /**
     * 
     * @return
     */
	protected Class<? extends DataElement<?>> getDataElementClass()
    {
    	return dataElementClass;
    }

	/**
     * 
     * @param value
     * @return
     */
    public static ValueRepresentation valueOfIgnoreCase(String value)
    {
    	try
    	{
    		return valueOf(value.toUpperCase());
    	}
    	catch(IllegalArgumentException iaX)
    	{
    		return null;
    	}
    }
}
