/**
 * Package: MAG - VistA Imaging
 * WARNING: Per VHA Directive 2004-038, this routine should not be modified.
 * Date Created: Jul 21, 2008
 * Site Name:  Washington OI Field Office, Silver Spring, MD
 * @author       DNS
 * @version 1.0
 *
 * ----------------------------------------------------------------
 * Property of the US Government.
 * No permission to copy or redistribute this software is given.
 * Use of unreleased versions of this software requires the user
 * to execute a written test agreement with the VistA Imaging
 * Development Office of the Department of Veterans Affairs,
 * telephone (DNS
 * 
 * The Food and Drug Administration classifies this software as
 * a Class II medical device.  As such, it may not be changed
 * in any way.  Modifications to this software may result in an
 * adulterated medical device under 21CFR820, the use of which
 * is considered to be a violation of US Federal Statutes.
 * ----------------------------------------------------------------
 */
package gov.va.med.imaging.dicom;

import java.util.regex.Pattern;

/**
 * @author       DNS
 *
 */
public enum TransferSyntaxUid
{
	// The ordering of the following enums is critical because finding a matching pattern
	// is done with a sequential search where the first match is taken as the match.
	// More specific patterns MUST precede more general patterns.
	RAW_EXPLICIT_VR_BIGENDIAN("1.2.840.10008.1.2.2", true, false, "image/bmp", false),
	RAW_EXPLICIT_VR_LITTLEENDIAN("1.2.840.10008.1.2.1", true, true, "image/bmp", false),
	DEFLATED_EXPLICIT_VR_LITTLEENDIAN("1.2.840.10008.1.2.1.99", true, true, "image/bmp", false),	// RAW_EXPLICIT_VR_LITTLEENDIAN then Internet RFC 1951.
	RAW_IMPLICIT_VR_LITTLEENDIAN("1.2.840.10008.1.2", false, true, "image/bmp", false),
	JPEG_BASELINE( "1.2.840.10008.1.2.4.50", true, true, "image/jpg", true ),
	JPEG_EXTENDED( "1.2.840.10008.1.2.4.51", true, true, "image/jpg", true ),
	JPEG_LOSSLESS_NON_HIERARCHICAL( "1.2.840.10008.1.2.4.57", true, true, "image/jpg", true ),
	JPEG_LOSSLESS_NON_HIERARCHICAL_FIRST_ORDER( "1.2.840.10008.1.2.4.70", true, true, "image/jpg", true ),
	JPEG_LS_LOSSLESS("1.2.840.10008.1.2.4.80", true, true, "image/jpg", true),				// ISO/IS-14495-1
	JPEG_LS_NEAR_LOSSLESS("1.2.840.10008.1.2.4.81", true, true, "image/jpg", true),		// ISO/IS-14495-1
	J2K_PART1_REVERSIBLE("1.2.840.10008.1.2.4.90", true, true, "image/jp2", true),			// (ISO/IEC 15444-1)
	J2K_PART1_IRREVERSIBLE("1.2.840.10008.1.2.4.91", true, true, "image/jp2", true),		// may also be reversible, (ISO/IEC 15444-1)
	J2K_PART2_REVERSIBLE("1.2.840.10008.1.2.4.92", true, true, "image/jp2", true),			// (ISO/IEC 15444-2)
	J2K_PART2_IRREVERSIBLE("1.2.840.10008.1.2.4.93", true, true, "image/jp2", true),		// may also be reversible, (ISO/IEC 15444-2)
	JPIP_REFERENCED("1.2.840.10008.1.2.4.94", true, true, "image/jp2", true),
	DEFLATED_JPIP_REFERENCED("1.2.840.10008.1.2.4.95", true, true, "image/jp2", true),		// JPIP_REFERENCED then Internet RFC 1951.
	MPEG("1.2.840.10008.1.2.4.100", true, true, "video/mpeg", true),						// ISO/IEC 13818-2 
	LOSSLESS_RLE("1.2.840.10008.1.2.5", true, true, "image/bmp", true),
	PROPRIETARY(Pattern.compile("[[\\d]*[\\x2e]?]*"), true, true, "image/bmp", true); 		// this MUST be last.
	
	private final Pattern pattern;
	private final boolean explicitVR;
	private final boolean littleEndian;
	private final String mimeType;
	private final boolean encapsulated;
	
	TransferSyntaxUid(String unencodedRegex, boolean explicitVR, boolean littleEndian, String mimeType, boolean encapsulated)
	{
		// Encode the string pattern into a regular expression.
		// The purpose of this is to allow more readable patterns in the enum
		// definition.
		this( Pattern.compile(unencodedRegex.replace(".", "\\x2e")), explicitVR, littleEndian, mimeType, encapsulated );
	}
	
	// for complex patterns, this constructor is provided
	TransferSyntaxUid(Pattern pattern, boolean explicitVR, boolean littleEndian, String mimeType, boolean encapsulated)
	{
		this.pattern = pattern;
		this.explicitVR = explicitVR;
		this.littleEndian = littleEndian;
		this.mimeType = mimeType;
		this.encapsulated = encapsulated;
	}
	
	public boolean matches(String value)
	{
		return pattern.matcher(value).matches();
	}

	/**
     * @return the pattern
     */
    public Pattern getPattern()
    {
    	return pattern;
    }

	/**
     * @return the explicitVR
     */
    public boolean isExplicitVR()
    {
    	return explicitVR;
    }

	/**
     * @return the littleEndian
     */
    public boolean isLittleEndian()
    {
    	return littleEndian;
    }
    
    public String getMimeType()
    {
    	return mimeType;
    }

    public boolean isEncapsulated()
    {
    	return this.encapsulated;
    }
    
	public static TransferSyntaxUid getByUid(String uid)
    {
    	for(TransferSyntaxUid transferSyntax : TransferSyntaxUid.values())
    		if(transferSyntax.matches(uid))
    			return transferSyntax;
    	
    	return null;
    }
    
	/**
	 * Given a two byte array, form an unsigned long, taking into account whether
	 * the format is big or little endian.
	 * 
	 * @param argtmp
	 * @return
	 */
	public long makeUnsignedLongFrom2Bytes(byte[] argtmp)
	{
		return (long)makeUnsignedIntFrom2Bytes(argtmp);	
	}

	/**
	 * @param value
	 * @return
	 */
	public byte[] make2BytesFromLong(long value)
	{
		byte[] result = new byte[2];
		
		if( isLittleEndian() )
		{
			result[1] = (byte)((0x000000000000ff00 & value)>>8);
			result[0] = (byte)(0x00000000000000ff & value);
		}
		else
		{
			result[0] = (byte)((0x000000000000ff00 & value)>>8);
			result[1] = (byte)(0x00000000000000ff & value);
		}
		
		return result;
	}
	
	/**
	 * Given a two byte array, form an unsigned integer, taking into account whether
	 * the format is big or little endian.
	 * 
	 * Little Endian
	 * The least significant byte (LSB) value is at the lowest address. 
	 * The other bytes follow in increasing order of significance.
	 * 
	 * Big Endian
	 * The most significant byte (LSB) value is at the lowest address. 
	 * The other bytes follow in decreasing order of significance.
	 * 
	 * @param argtmp
	 * @return
	 */
	public int makeUnsignedIntFrom2Bytes(byte[] argtmp)
	{
		return makeUnsignedIntFrom2Bytes(argtmp, 0);
	}
	public int makeUnsignedIntFrom2Bytes(byte[] argtmp, int offset)
	{
		return isLittleEndian() ?
				( (0x000000ff & argtmp[1]) << 8 ) | (0x000000ff & argtmp[offset]) :
				( (0x000000ff & argtmp[0]) << 8 ) | (0x000000ff & argtmp[offset + 1]);
	}
	
	public byte[] make2BytesFromInt(int value)
	{
		byte[] result = new byte[2];
		
		if( isLittleEndian() )
		{
			result[1] = (byte)((0x0000ff00 & value)>>8);
			result[0] = (byte)(0x000000ff & value);
		}
		else
		{
			result[0] = (byte)((0x0000ff00 & value)>>8);
			result[1] = (byte)(0x000000ff & value);
		}
		
		return result;
	}
	
	public int makeUnsignedIntFrom4Bytes(byte[] rawValue)
	{
		return makeUnsignedIntFrom4Bytes(rawValue, 0);
	}    
	public int makeUnsignedIntFrom4Bytes(byte[] rawValue, int offset)
	{
		return (int)makeLongFromBytes(rawValue, offset, 4);
	}    
	
	public byte[] make4BytesFromInt(int value)
	{
		byte[] result = new byte[4];
		
		if( isLittleEndian() )
		{
			result[3] = (byte)((0xff000000 & value)>>24);
			result[2] = (byte)((0x00ff0000 & value)>>16);
			result[1] = (byte)((0x0000ff00 & value)>>8);
			result[0] = (byte) (0x000000ff & value);
		}
		else
		{
			result[0] = (byte)((0xff0000 & value)>>24);
			result[1] = (byte)((0x00ff0000 & value)>>16);
			result[2] = (byte)((0x0000ff00 & value)>>8);
			result[3] = (byte)(0x000000ff & value);
		}
		
		return result;
	}
	
	/**
	 * Given a four byte array, form an unsigned integer, taking into account whether
	 * the format is big or little endian.
	 * 
	 * @param argtmp
	 * @return
	 */
	public long makeUnsignedLongFrom4Bytes(byte[] argtmp)
	{
		return (long)makeUnsignedIntFrom4Bytes(argtmp, 0);
	}    
	public long makeUnsignedLongFrom4Bytes(byte[] rawValue, int offset)
	{
		return makeLongFromBytes(rawValue, offset, 4);
	}    
	
	public byte[] make4BytesFromLong(long value)
	{
		return make4BytesFromInt((int)value);
	}

	public long makeUnsignedLongFrom8Bytes(byte[] rawValue)
    {
		return makeUnsignedLongFrom8Bytes(rawValue, 0);
    }
	
	public long makeUnsignedLongFrom8Bytes(byte[] rawValue, int offset)
	{
		return makeLongFromBytes(rawValue, offset, 8);
	}
	
	/**
	 * 
	 * @param rawValue
	 * @param offset
	 * @param count
	 * @return
	 */
	public long makeLongFromBytes(byte[] rawValue, int offset, int count)
    {
		long result = 0l;
		
		if(isLittleEndian())
			for(int index=count-1; index >= 0; index--)
				result |= (0x000000ff & rawValue[offset + index]) << (8*index);
		else
			for(int index=0; index < count; ++index)
				result |= (0x000000ff & rawValue[offset + ((count-1)-index)]) << (8*index);
		
		return result;
    }

	/**
	 * 
	 * @param rawValue
	 * @return
	 */
	public Float makeFloatFrom4Bytes(byte[] rawValue)
	{
		return makeFloatFrom4Bytes(rawValue, 0);
	}
	
	/**
	 * 
	 * @param rawValue
	 * @param offset
	 * @return
	 */
	public Float makeFloatFrom4Bytes(byte[] rawValue, int offset)
	{
		int bits = makeUnsignedIntFrom4Bytes(rawValue, offset);
		return new Float( Float.intBitsToFloat(bits) );
	}
	
	/**
	 * Make a word array in transfer syntax defined order from a byte array.
	 * Each two bytes in the byte array makes up a word, the resulting array
	 * will be in the defined byte order (i.e. for big-endian does nothing,
	 * for little-endian, swaps every other byte).
	 * 
	 * To save memory, the resulting array is written to the array referenced 
	 * by the rawValue parameter (i.e. in place). 
	 * 
	 * The offset and length are both in terms of bytes.  The length MUST be an
	 * even number
	 * 
	 * @param rawValue
	 * @param offset
	 * @return
	 */
	public void intepretByteArrayAsWordArray(byte[] rawValue)
	{
		intepretByteArrayAsWordArray(rawValue, 0, rawValue.length);
	}
	
	public void intepretByteArrayAsWordArray(byte[] rawValue, int offset, int length)
	{
		byte[] result = new byte[length];
		
		if(isLittleEndian())
		{
			for(int index = 0; index < length; index += 2)
			{
				byte tmp = result[index];
				result[index] = rawValue[index+1];
				result[index+1] = tmp;
			}
		}
	}
}
