/**
 * Package: MAG - VistA Imaging
 * WARNING: Per VHA Directive 2004-038, this routine should not be modified.
 * Date Created: Aug 27, 2008
 * Site Name:  Washington OI Field Office, Silver Spring, MD
 * @author      DNS
 * @version 1.0
 *
 * ----------------------------------------------------------------
 * Property of the US Government.
 * No permission to copy or redistribute this software is given.
 * Use of unreleased versions of this software requires the user
 * to execute a written test agreement with the VistA Imaging
 * Development Office of the Department of Veterans Affairs,
 * telephone (DNS
 * 
 * The Food and Drug Administration classifies this software as
 * a Class II medical device.  As such, it may not be changed
 * in any way.  Modifications to this software may result in an
 * adulterated medical device under 21CFR820, the use of which
 * is considered to be a violation of US Federal Statutes.
 * ----------------------------------------------------------------
 */
package gov.va.med.imaging.dicom;

import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.DicomFormatPixelDataException;

/**
 * @author      DNS
 *
 */
public class RawDataToPixelTransformer
{
	private static final int BITS_PER_BYTE = 8;
	
	private boolean wordOriented = false;
	
	public RawDataToPixelTransformer()
	{
		this(false);
	}
	
	public RawDataToPixelTransformer(boolean wordOriented)
	{
		this.wordOriented = wordOriented;
	}
	
	public boolean isWordOriented()
	{
		return wordOriented;
	}
	
	public boolean isByteOriented()
	{
		return !wordOriented;
	}
	
	/**
	 * Transform an array of bytes into an array of pixel data.
	 * The bytes may be interpreted as single-byte or as double-byte elements,
	 * which we refer to as the 'word' size in this code (even though DICOM uses word to
	 * mean 16 bits).
	 * 
	 * Examples:
	 * rawData = 0x01 0x00
	 * bits (allocated=8, stored=8, highbit=7)
	 * VR 'OW'
	 * width=1, height=1
	 * result = {0x00000001}
	 * 
     * @param rawData
     * @param height
     * @param width
     * @param bitsAllocated
     * @return
     */
    public int[] transformRawByteDataToIntPixelArray(
    	byte[] rawData, 			// a raw byte array
    	int height, int width, 		// the destination pixel array height and width
    	int bitsAllocated, int bitsStored, int highBit)	// the storage characteristics of the raw data
    throws DicomFormatException
    {
    	if(isByteOriented() && bitsAllocated > 8)
    		throw new DicomFormatPixelDataException("The maximum number of bits allocated for an OB must be 8 or less.");
    	if(isWordOriented() && bitsAllocated > 16)
    		throw new DicomFormatPixelDataException("The maximum number of bits allocated for an OW must be 16 or less.");
    	if(bitsStored > bitsAllocated)
    		throw new DicomFormatPixelDataException("The number of bits stored must be less than or equal to bits allocated");
    	if(highBit > bitsAllocated || highBit < 0)
    		throw new DicomFormatPixelDataException("The high bit must be 0 or one less than bits allocated");
    	
    	RawDataAccessor rawDataAccessor = new RawDataAccessor(rawData, isByteOriented());
    	int storageMask = createIntegerSizedRightShiftedBitMask(bitsStored, 0);
    	int wordSize = isByteOriented() ? 8 : 16;
    	
	    int[] destination = new int[height * width];
		for(int heightIndex=0; heightIndex < height; ++heightIndex )
		{
			for(int widthIndex=0; widthIndex < width; ++widthIndex )
			{
				// the offset of the element in the destination buffer
				int destinationIndex = (heightIndex * width) + widthIndex;
				
				// the number of bits from the start of the source buffer to the first bit of the current pixel
				int sourceBitIndex = destinationIndex * bitsAllocated;
				
				// the offset of the first word of the source data and the offset
				// in the first word of the first bit
				int sourceWordIndex = sourceBitIndex / wordSize;
				int sourceBitStartOffset = sourceBitIndex % wordSize;

				// the last bit to take from the first word (may exceed the word boundary)
				int sourceBitEndOffset = sourceBitStartOffset + bitsAllocated;
				// the number of bits to take from the first byte
				int sourceBitsFromWord1 = bitsAllocated - Math.min(0, wordSize - sourceBitEndOffset);
				// the leftover bits past this pixel data
				int sourceBitsRightOffset = -1 * Math.min(0, sourceBitEndOffset - wordSize);
				// the mask that will get just this pixels bits
				int sourceWord1Mask = createIntegerSizedRightShiftedBitMask( sourceBitsFromWord1, sourceBitsRightOffset );

				// the pixel value eventually ends up in here
				int pixelValue = 0;
				
				// get the word that contains the first part of the pixel value
				// if the pixel is entirely contained within a word then this is the entire
				// pixel value (though possibly left shifted)
				pixelValue = rawDataAccessor.getRawDataValue(sourceWordIndex);
				//System.out.print("rawPixelValue = " + pixelValue);
				
				// mask it to remove what may be the low-order part of the previous pixel value
				// or the high order part of the next pixel
				pixelValue &= sourceWord1Mask;
				
				// if the current pixel spans the word (8 or 16 bit) boundary
				if(sourceBitEndOffset > wordSize)
				{
					int sourceBitsFromWord2 = sourceBitEndOffset-wordSize;
					
					// shift it to make room for the part of the value from the next word
					pixelValue = pixelValue << sourceBitsFromWord2;	// make room for the bits from the second byte
					
					// we know we are taking the leftmost bits
					byte sourceByte2Mask = createByteSizedLeftShiftedBitMask(sourceBitsFromWord2, 0);
					int pixelValue2 = rawDataAccessor.getRawDataValue(sourceWordIndex+1) & sourceByte2Mask;
					pixelValue2 = pixelValue2 >> (wordSize - sourceBitsFromWord2);
					
					pixelValue |= pixelValue2;
				}
				// else, the pixel value is entirely contained within one byte
				// simply shift it and we're done
				else
				{
					pixelValue = pixelValue >> sourceBitsRightOffset;
				}
				//System.out.println(" => " + pixelValue);

				destination[destinationIndex] = (int)( pixelValue & storageMask);
			}			
		}
	    return destination;
    }

    private class RawDataAccessor
    {
    	private final byte[] rawData;
    	private final boolean byteOriented;
    	
    	RawDataAccessor(byte[] rawData, boolean byteOriented)
    	{
    		this.rawData = rawData;
    		this.byteOriented = byteOriented;
    	}
    	
        /**
         * Get a value from the rawData array as an int.
         * If the array orientation is as a byte then simply return the 
         * index value, else the array orientation is a word so return 
         * the pair of bytes as an int.
         * 
         * @param rawData
         * @param index
         * @return
         */
    	int getRawDataValue(int index)
    	{
    		if(this.byteOriented)
    			return this.rawData[index];
    		else
    			return (this.rawData[index*2] << 8) + this.rawData[index*2 + 1]; 
    	}
    }
    
	
    
    /**
     * Create a bit mask that contains 'width' number of one's, offset from the left
     * by offset bits.
     * 
     * @param bits - the number of ON bits in the mask
     * @param offset - offset from the right
     * @return
     */
    public byte createByteSizedLeftShiftedBitMask(int width, int leftOffset)
    throws DicomFormatPixelDataException
    {
    	int mask = createIntegerSizedLeftShiftedBitMask(width, leftOffset);

    	mask &= 0xFF000000;
    	mask = mask >>> 24;
    	
    	return (byte)mask;
    }

    /**
     * Create a bit mask that contains 'width' number of one's, offset from the right
     * by offset bits.
     * 
     * @param bits - the number of ON bits in the mask
     * @param offset - offset from the right
     * @return
     * @throws DicomFormatPixelDataException 
     */
    public byte createByteSizedRightShiftedBitMask(int width, int rightOffset) 
    throws DicomFormatPixelDataException
    {
    	if(width + rightOffset > BITS_PER_BYTE)
    		throw new DicomFormatPixelDataException(
    			"Invalid request to generate bit mask, width='" + width + "' and rightOffset='" + rightOffset + "'.");

    	int mask = createIntegerSizedLeftShiftedBitMask(width, 0);

    	mask &= 0xFF000000;
    	mask = mask >>> 24 - rightOffset;
    	
    	return (byte)mask;
    }
    
    /**
     * 
     * @param width
     * @param leftOffset
     * @return
     * @throws DicomFormatPixelDataException
     */
    public int createIntegerSizedLeftShiftedBitMask(int width, int leftOffset)
    throws DicomFormatPixelDataException
    {
    	// build the mask of the correct width
    	int mask = createLeftAlignedMask(width);

    	mask = mask >>> leftOffset;
    	
    	return mask;
    }
    
    /**
     * 
     * @param width
     * @param leftOffset
     * @return
     * @throws DicomFormatPixelDataException
     */
    public int createIntegerSizedRightShiftedBitMask(int width, int rightOffset)
    throws DicomFormatPixelDataException
    {
    	// build the mask of the correct width
    	int mask = createLeftAlignedMask(width);

    	mask = mask >>> (32 - (rightOffset + width));
    	
    	return mask;
    }
    
    /**
     * Create an integer with the number of bits specified set in the leftmost
     * bits.
     * @param width
     * @return
     */
    private int createLeftAlignedMask(int width)
    {
    	// build the mask of the correct width
    	int mask = 0x80000000;
    	for(int index=0; index < (width-1); ++index)
    		mask = mask >> 1;
    	
    	return mask;
    }
}
