package gov.va.med.imaging.dicom;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * From the DICOM Specification, Part 5:
 * A unique identifier for a Data Element composed of an ordered pair of numbers 
 * (a Group Number followed by an Element Number).
 * 
 * A class that encapsulates the group and element fields
 * and deals with formatting.
 * 
 * @author      DNS
 *
 */
public class DataElementTag
implements Comparable<DataElementTag>
{
	private final int group;
	private final int element;

	private static String ElementRegex = "[0x28]?0[xX]([\\d]{4}),0[xX]([\\d]{4})[0x29]?";
	private Pattern elementPattern = Pattern.compile(ElementRegex);
	
	// static definitions of well known tag elements
	public static DataElementTag SEQUENCE_ITEM_TAG = new DataElementTag(0xFFFE, 0xE000);
	public static DataElementTag ITEM_DELIMITATION_TAG = new DataElementTag(0xFFFE, 0xE00D);
	public static DataElementTag SEQUENCE_DELIMITATION_TAG = new DataElementTag(0xFFFE, 0xE0DD);
	
	public static DataElementTag PIXEL_DATA_TAG = new DataElementTag(0x7FE0, 0x0010);	// the infamous "7 FE 10"
	public static DataElementTag TRANSFER_SYNTAX_UID_TAG = new DataElementTag(0x0002,0x0010);
	
	public static DataElementTag SAMPLES_PER_PIXEL = new DataElementTag(0x0028, 0x0002);
	public static DataElementTag SAMPLES_PER_PIXEL_USED = new DataElementTag(0x0028, 0x0003);
	public static DataElementTag PHOTOMETRIC_INTERPRETATION = new DataElementTag(0x0028, 0x0004);
	public static DataElementTag PLANAR_CONFIGURATION = new DataElementTag(0x0028, 0x0006);
	public static DataElementTag NUMBER_OF_FRAMES = new DataElementTag(0x0028, 0x0008);
	public static DataElementTag FRAME_INCREMENT_POINTER = new DataElementTag(0x0028, 0x0009);
	public static DataElementTag FRAME_DIMENSION_POINTER = new DataElementTag(0x0028, 0x000A);
	public static DataElementTag ROWS = new DataElementTag(0x0028, 0x0010);
	public static DataElementTag COLUMNS = new DataElementTag(0x0028, 0x0011);
	public static DataElementTag PLANES = new DataElementTag(0x0028, 0x0012);
	
	public static DataElementTag BITS_ALLOCATED = new DataElementTag(0x0028, 0x0100);
	public static DataElementTag BITS_STORED = new DataElementTag(0x0028, 0x0101);
	public static DataElementTag HIGH_BIT = new DataElementTag(0x0028, 0x0102);
	
	/**
	 * A constructor to maintain compatibility with the String based element keys
	 * used in the original version.
	 * 
	 * @param elementKeyAsString
	 * @throws IllegalArgumentException
	 */
	public DataElementTag(String elementKeyAsString)
	throws IllegalArgumentException
	{
	    super();
	    
		String elementAsString = elementKeyAsString.replace(" ", "");
		Matcher matcher = elementPattern.matcher(elementAsString);
		
		if( matcher.find() )
		{
			int group = Integer.parseInt( matcher.group(1) );
			int element = Integer.parseInt( matcher.group(2) );
		    this.group = group;
		    this.element = element;
		}
		else
			throw new IllegalArgumentException("The given String '" + elementKeyAsString + "' is not parsable as an element key.");
	}
	
	/**
	 * The preferred constructor
	 * 
	 * @param group
	 * @param element
	 */
	public DataElementTag(int group, int element)
    {
	    super();
	    this.group = group;
	    this.element = element;
    }

	/**
     * @return the group
     */
    public int getGroup()
    {
    	return group;
    }

	/**
     * @return the element
     */
    public int getElement()
    {
    	return element;
    }
	
    @Override
	public String toString()
	{
		return "(" + 
			DicomUtilities.toFourDigitHex(group) + "," + 
			DicomUtilities.toFourDigitHex(element) + 
			")";
	}
    
	/**
	 * Defines the natural ordering of DataElementTag as increasing by group and then element number.
	 * This implementation of compareTo() (and the Comparable interface) is consistent with equals().
	 * 
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     * @return a negative integer, zero, or a positive integer as this object is less than, equal to, 
     *         or greater than the specified object. 
     */
    @Override
    public int compareTo(DataElementTag that)
    {
    	return this.group != that.group ? this.group - that.group : this.element - that.element;
    }

	/**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode()
    {
	    final int prime = 31;
	    int result = 1;
	    result = prime * result + element;
	    result = prime * result + group;
	    return result;
    }

	/**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj)
    {
	    if (this == obj)
		    return true;
	    if (obj == null)
		    return false;
	    if (getClass() != obj.getClass())
		    return false;
	    final DataElementTag other = (DataElementTag) obj;
	    if (element != other.element)
		    return false;
	    if (group != other.group)
		    return false;
	    return true;
    }
}
