package gov.va.med.imaging.dicom;

import gov.va.med.imaging.dicom.dictionary.DicomDictionaryEntry;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInterpretationException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationValueLengthExceededException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationValueLengthInsufficientException;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * From the DICOM Specification, Part 5:
 * 
 * A Data Element is made up of fields. Three fields are common to all three Data Element structures; 
 * these are the Data Element Tag, Value Length, and Value Field. 
 * A fourth field, Value Representation, is only present in the two Explicit VR Data Element structures. 
 * The Data Element structures are defined in Sections 7.1.2. and 7.1.3. The definitions of the fields are:
 * 
 * Data Element Tag: An ordered pair of 16-bit unsigned integers representing the Group Number followed by Element Number.
 * 
 * Value Representation: A two-byte character string containing the VR of the Data Element. 
 * The VR for a given Data Element Tag shall be as defined by the Data Dictionary as specified in PS 3.6. 
 * The two character VR shall be encoded using characters from the DICOM default character set.
 * 
 * Value Length: 
 * Either:
 * - a 16 or 32-bit (dependent on VR and whether VR is explicit or implicit) unsigned integer 
 * containing the Explicit Length of the Value Field as the number of bytes (even) that make up the Value. 
 * It does not include the length of the Data Element Tag, Value Representation, and Value Length Fields.
 * - a 32-bit Length Field set to Undefined Length (FFFFFFFFH). Undefined Lengths may be used for Data Elements 
 * having the Value Representation (VR) Sequence of Items (SQ) and Unknown (UN). 
 * For Data Elements with Value Representation OW or OB Undefined Length may be used depending on the negotiated 
 * Transfer Syntax (see Section 10 and Annex A).
 * 
 * A unit of information as defined by a single entry in the data dictionary. 
 * An encoded Information Object Definition (IOD) Attribute that is composed of, at a minimum, 
 * three fields: a Data Element Tag, a Value Length, and a Value Field. For some specific 
 * Transfer Syntaxes, a Data Element also contains a VR Field where the Value Representation of 
 * that Data Element is specified explicitly.
 * 
 * REPEATING GROUP: Standard Data Elements within a particular range of Group Numbers where elements 
 * that have identical Element Numbers have the same meaning within each Group (and the same VR, VM, 
 * and Data Element Type). Repeating Groups shall only exist for Curves and Overlay Planes (Group Numbers 
 * (50xx,eeee) and (60xx,eeee), respectively) and are a remnant of versions of this standard prior to V3.0.
 * 
 * RETIRED DATA ELEMENT: A Data Element that is unsupported beginning with Version 3.0 of this standard. 
 * Implementations may continue to support Retired Data Elements for the purpose of backward compatibility 
 * with versions prior to V3.0, but this is not a requirement of this version of the standard.
 * 
 * A Data Element shall have one of three structures. 
 * Two of these structures contain the VR of the Data Element (Explicit VR) but differ in the way their 
 * lengths are expressed, while the other structure does not contain the VR (Implicit VR). 
 * All three structures contain the Data Element Tag, Value Length and Value for the Data Element. See Figure 7.1-1.
 * Implicit and Explicit VR Data Elements shall not coexist in a Data Set and Data Sets nested within it 
 * (see Section 7.5). Whether a Data Set uses Explicit or Implicit VR, among other characteristics, 
 * is determined by the negotiated Transfer Syntax (see Section 10 and Annex A).
 * Note: VRs are not contained in Data Elements when using DICOM Default Transfer Syntax (DICOM Implicit VR 
 * Little Endian Transfer Syntax).
 * 
 * This implementation includes a reference to a DataDictionaryEntry, which contains the implicit VR.
 * 
 * The methods equals(), hashCode() and compareTo() are all consistent. 
 * The methods equals(), hashCode() and compareTo() are all sensitive only to the dataElementTag member.
 * 
 * 
 * @author       DNS
 *
 */
public abstract class DataElement<T>
implements Comparable<DataElement<T>>
{
	private final DataElementFactory instantiatingFactory;	// retain a reference to our factory because we may need it to parse our own data
	private final DataElementTag dataElementTag;			// the actual group and element numbers
	private final DicomDictionaryEntry dictionaryEntry;		// null for explicit VR, not-null for implicit VR
	private final ValueRepresentation[] explicitVRField;	// not null for explicit VR, null for implicit VR
	private final long valueLength;							// will always be present, 16 or 32 bit unsigned, must be stored as long
	private byte[] rawValue;							//
	public final static long INDETERMINATE_LENGTH = -1L;
	
	private final Logger logger = Logger.getLogger(this.getClass().getName());
	
	/**
	 * The constructor of DataElement instances using explicit VR.  
	 * This should be called only from a DataElementFactory.
	 * 
	 * @param dataElementTag
	 * @param explicitVRField
	 * @param valueLength
	 * @param value
	 */
	protected DataElement(
			DataElementFactory instantiatingFactory, 
			DataElementTag dataElementTag, 
			ValueRepresentation explicitVRField, 
			long valueLength, 
			byte[] value)
	throws DicomFormatException
	{
		if(explicitVRField == null)
			throw new IllegalArgumentException("Explicit VR Field must be specified when encoding specifies that explicit VR is being used.");
		
		this.instantiatingFactory = instantiatingFactory;
		this.dataElementTag = dataElementTag;
		this.dictionaryEntry = null;
		this.explicitVRField = new ValueRepresentation[]{explicitVRField};
		this.valueLength = valueLength;
		this.rawValue = value;
		
		try{validateLengthValues();}
		catch(DicomFormatException dfX){logger.log(Level.WARNING, "DICOM compliance exception-" + dfX.getMessage() + "'.");}
		
		parseRawValue();
	}
	
	/**
	 * The constructor of DataElement instances using implicit VR.  
	 * This should be called only from a DataElementFactory.
	 * 
	 * @param dataElementTag
	 * @param dictionaryEntry
	 * @param valueLength
	 * @param value
	 * @param dataElementFactory 
	 */
	protected DataElement(
		DataElementFactory instantiatingFactory, 
		DataElementTag dataElementTag, 
		DicomDictionaryEntry dictionaryEntry, 
		long valueLength, 
		byte[] value)
	throws DicomFormatException
    {
		if(dictionaryEntry == null)
			throw new IllegalArgumentException("Dictionary Entry field must be specified when encoding specifies that implicit VR is being used.");
		
		this.instantiatingFactory = instantiatingFactory;
	    this.dataElementTag = dataElementTag;
	    this.dictionaryEntry = dictionaryEntry;
	    this.valueLength = valueLength;
	    this.rawValue = value;
		this.explicitVRField = null;

		try{validateLengthValues();}
		catch(DicomFormatException dfX){logger.log(Level.WARNING, "DICOM compliance exception-" + dfX.getMessage() + "'.");}
		
		parseRawValue();
    }

	/**
	 * Do basic validation of the length field and the length of the value against the
	 * specified values.
	 *  
	 * @throws ValueRepresentationInterpretationException
	 */
	protected void validateLengthValues() 
	throws ValueRepresentationInterpretationException
	{
		if(getValueLength() < getValueRepresentation()[0].getMinLengthOfValue() &&
				!getValueRepresentation()[0].isAllowUndefinedLength())
			throw new ValueRepresentationValueLengthInsufficientException(
				"Element '" + getDataElementTag() + "' " + getValueRepresentation()[0].toString() + 
				" length of " + valueLength + " is outside range of [" + 
				getValueRepresentation()[0].getMinLengthOfValue() + "-" + 
				getValueRepresentation()[0].getMaxLengthOfValue() + "].");
		
		if(getValueLength() > getValueRepresentation()[0].getMaxLengthOfValue() &&
				!getValueRepresentation()[0].isAllowUndefinedLength())
			throw new ValueRepresentationValueLengthExceededException(
				"Element '" + getDataElementTag() + "' " + getValueRepresentation()[0].toString() + 
				" length of " + valueLength + " is outside range of [" + 
				getValueRepresentation()[0].getMinLengthOfValue() + "-" + 
				getValueRepresentation()[0].getMaxLengthOfValue() + "].");
		
		if(getRawValue() == null)
			return;
		
		if(getRawValue().length < getValueRepresentation()[0].getMinLengthOfValue() &&
				!getValueRepresentation()[0].isAllowUndefinedLength())
			throw new ValueRepresentationValueLengthInsufficientException(
				"Element '" + getDataElementTag() + "' " + getValueRepresentation()[0].toString() + 
				" raw value length of " + getRawValue().length + " is outside range of [" + 
				getValueRepresentation()[0].getMinLengthOfValue() + "-" + 
				getValueRepresentation()[0].getMaxLengthOfValue() + "].");
		
		if(getValueLength() > getRawValue().length &&
				!getValueRepresentation()[0].isAllowUndefinedLength())
			throw new ValueRepresentationValueLengthExceededException(
				"Element '" + getDataElementTag() + "' " + getValueRepresentation()[0].toString() + 
				" raw value length of " + rawValue.length + " is outside range of [" + 
				getValueRepresentation()[0].getMinLengthOfValue() + "-" + 
				getValueRepresentation()[0].getMaxLengthOfValue() + "].");
	}
	
	/**
	 * Called from the constructor,
	 * The implementing class must take the raw value as a byte array and create a implememntation
	 * specific representation from it.
	 * The resulting value must be made available through the getValue() method.
	 * 
	 */
	protected abstract void parseRawValue()
	throws DicomFormatException;

    /**
     * Get the value of this field according to the interpretation of the
     * associated ValueRepresentation
     * @return
     * @throws ValueRepresentationInterpretationException 
     */
    public abstract T getValue() 
    throws ValueRepresentationInterpretationException;
    
    /**
     * 
     * @return
     */
	protected Logger getLogger()
    {
    	return logger;
    }

	/**
	 * 
	 * @return
	 */
	protected DataElementFactory getInstantiatingFactory()
    {
    	return instantiatingFactory;
    }

	/**
     * @return the tag
     */
    public DataElementTag getDataElementTag()
    {
    	return dataElementTag;
    }

	/**
	 * Will always return a non-null value.  If using explicit VR then this
	 * will return a single element array that contains the VR specified 
	 * when this instance was created.
	 * If using implicit VR then this will return an array of VR mapped to
	 * the entry in the DicomDictionary that the element tag values were
	 * mapped to when this instance was created.
	 * 
     * @return the ValueRepresentation array
     */
    public ValueRepresentation[] getValueRepresentation()
    {
    	if(isExplicitVR())
    		return this.explicitVRField;
    	else
    		return this.getDictionaryEntry().getVr();
    }
    
    public boolean isExplicitVR()
    {
    	return this.explicitVRField != null;
    }
    
	/**
	 * Will return null if isExplicitVR() is true.
	 * 
     * @return the dictionaryEntry
     */
    public DicomDictionaryEntry getDictionaryEntry()
    {
    	return dictionaryEntry;
    }


	/**
	 * Will return null if isExplicitVR() is false.
	 * 
     * @return the explicitVRField
     */
    public ValueRepresentation[] getExplicitVRField()
    {
    	return explicitVRField;
    }


	/**
     * @return the value
     */
    public byte[] getRawValue()
    {
    	return rawValue;
    }

    /**
     * In some cases the derived classes use the raw value as the interpreted
     * value, optionally having manipulated the contents.
     * @param rawValue
     */
    protected void setRawValue(byte[] rawValue)
    {
    	this.rawValue = rawValue;
    }
    
	/**
     * @return the valueLength
     */
    public long getValueLength()
    {
    	return valueLength;
    }
    
	/**
	 * The natural ordering of this class is the natural ordering of the DataElementTag within this class.
	 * 
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    @Override
    public int compareTo(DataElement<T> that)
    {
    	return this.dataElementTag.compareTo(that.dataElementTag);
    }

	@Override
    public int hashCode()
    {
	    final int prime = 31;
	    int result = 1;
	    result = prime * result + ((dataElementTag == null) ? 0 : dataElementTag.hashCode());
	    return result;
    }

	@Override
    public boolean equals(Object obj)
    {
	    if (this == obj)
		    return true;
	    if (obj == null)
		    return false;
	    if (getClass() != obj.getClass())
		    return false;
	    final DataElement other = (DataElement) obj;
	    if (dataElementTag == null)
	    {
		    if (other.dataElementTag != null)
			    return false;
	    } else if (!dataElementTag.equals(other.dataElementTag))
		    return false;
	    return true;
    }

	// ========================================================================================
	// Helper Methods to do various, usually bit or byte level, manipulations
	// ========================================================================================
    protected static String transformValueToStringRemoveNulls(byte[] rawValue, long valueLength)
    {
		for (int j = 0; j < valueLength; j++)
			if (rawValue[j] == 0) rawValue[j] = 20;
		return new String(rawValue);
    }
    
    protected String getRawValueCharacterRepresentation()
    {
    	StringBuffer sb = new StringBuffer();
    	
    	byte[] raw = getRawValue();
    	if(raw != null)
    		for(byte c : getRawValue())
    			sb.append("0x" + Integer.toHexString(c) + " ");
    	return sb.toString();
    }
}