package gov.va.med.imaging.mix.business.fhir;

import gov.va.med.MockDataGenerationField;
import gov.va.med.imaging.GUID;
import gov.va.med.imaging.mix.business.InstanceComparator;
import gov.va.med.imaging.exchange.enums.ObjectOrigin;

import java.io.Serializable;
import java.util.*;
/**
 * @author DNS DNS
 *
 * This Series class is for FHIR ImagingStudy model support (referring to the equivalent DICOM Series term)
 * Cardinality: Patient 1..* DiagnosticReport 0..* ImagingStudy 0..* Series 0..* Instance
 */
public class Series
implements Serializable, Iterable<Instance>, Comparable<Series>
{
	private static final long serialVersionUID = -7467740571635450273L;
	@MockDataGenerationField(pattern="[a-z0-9]{6,6}")
	private String seriesIen; 	// only for VA series
	@MockDataGenerationField(pattern="[0-9]{1,3}")
	private String number; 		// Numeric identifier of this series
	@MockDataGenerationField(pattern="[A-Z]{2,2}")
	private String modality; 	// string representation of a modality (CT, CR, MR, etc) that created this series
	@MockDataGenerationField(pattern=MockDataGenerationField.UID_PATTERN)
	private String uid;			// R! DICOM Series UID (64)
	@MockDataGenerationField(componentValueType="gov.va.med.imaging.mix.business.Instance")
	private String description;	// max 64/DICOM
	private String availability; // ONLINE | OFFLINE | NEARLINE | UNAVAILABLE -- only ONLINE data is maintained for exchange
	private String url; 		// Location of the referenced instance(s) - not maintained
	private String bodySite; 	// Body part examined - not maintained ENUM
	private String laterality; 	// 'L' or 'R': body part laterality 
	private String startDateTime;	// when series acquisition started (YYYYMMDD.HHMISS)
	private Integer numberOfInstances;	//R! -- supposed to match instances SET size
	private SortedSet<Instance> instances = new TreeSet<Instance>(new InstanceComparator());
	@MockDataGenerationField()
	private ObjectOrigin objectOrigin; // VA or DOD
	
	/**
	 * @param studyIen
	 * @param defaultSeriesNumber
	 * @return
	 */
	public static Series create(ObjectOrigin objectOrigin, String seriesIen, String seriesNumber)
	{
		Series series = new Series();
		
		series.setObjectOrigin(objectOrigin);
		series.setSeriesIen(seriesIen);
		series.setNumber(seriesNumber);
		
		return series;
	}
	
	public Series()
	{
		//images = new TreeSet<Instance>();
		uid = seriesIen = number = modality = url = bodySite = laterality = startDateTime = "";
		availability = "ONLINE";
		numberOfInstances=0;
		objectOrigin = ObjectOrigin.VA;
	}
	
	public int getNumberOfInstances()
	{
		return numberOfInstances; // this.instances.size();
	}
	
	@Deprecated
	public Set<Instance> getImages() 
	{
		return Collections.unmodifiableSet(instances);
	}
		
	public void replaceInstance(Instance oldInstance, Instance newInstance)
	{
		instances.remove(oldInstance);
		instances.add(newInstance);
	}
	
	/**
	 * Add instance to the Series as a child.
	 * @param image
	 */
	public void addInstance(Instance instance) 
	{
		synchronized(instances)
		{
			instances.add(instance);
			numberOfInstances++;
		}
	}

	/**
	 * Add all of the instances in the given Set to the Series as children.
	 * @param images
	 */
	public void addInstances(SortedSet<Instance> instances) 
	{
		synchronized(this.instances)
		{
			this.instances.addAll(instances);
			numberOfInstances += instances.size();
		}
	}

	@Override
	public Iterator<Instance> iterator()
	{
		return instances.iterator();
	}

	/**
	 * @return the seriesIen
	 */
	public String getSeriesIen() {
		return seriesIen;
	}

	/**
	 * @param seriesIen the seriesIen to set
	 */
	public void setSeriesIen(String seriesIen) {
		this.seriesIen = seriesIen;
	}

	public static Series createMockSeries(String number) {
		Series series = new  Series();
		series.seriesIen = new GUID().toShortString();
		series.uid = new GUID().toShortString();
		series.number = number;
		return series;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		String output = "Series Details [" + this.seriesIen + "] \t Series UID [" + this.uid + "]\t Series Number [" + this.number + "]\n"
			+ "modality [" + this.modality + "]\t Contains " + instances.size() + " images\n";
		Iterator<Instance> imageIter = instances.iterator();
		while(imageIter.hasNext()) {
			Instance instance = imageIter.next();
			output += "\t IEN[" + (instance == null ? "<null instance>" : instance.getIen()) + 
				"] UID[" + (instance == null ? "<ni>" : instance.getUid()) + 
				"] \t Instance#[" + (instance == null ? "<ni>" : instance.getNumber()) + 
//				"] \t Disp Seq # [" + (instance == null ? "<ni>" : instance.getDicomSequenceNumberForDisplay()) + 
				"] \n";
		}
		
		return output;
	}

	public String getNumber() {
		return number;
	}

	public void setNumber(String number) {
		this.number = number;
	}

	/**
	 * Returns the string representation of a modality (CT, CR, MR, etc)
	 */
	public String getModality() {
		return modality;
	}

	/**
	 * @param modality Sets the string representation of a modality (CT, CR, MR, etc)
	 */
	public void setModality(String modality) {
		this.modality = modality;
	}

	public String getUid() {
		return uid;
	}

	public void setUid(String uid) {
		this.uid = uid;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getAvailability() {
		return availability;
	}

	public void setAvailability(String availability) {
		this.availability = availability;
	}

	public String getUrl() {
		return url;
	}

	public void setUrl(String url) {
		this.url = url;
	}

	public String getBodySite() {
		return bodySite;
	}

	public void setBodySite(String bodySite) {
		this.bodySite = bodySite;
	}

	public String getLaterality() {
		return laterality;
	}

	public void setLaterality(String laterality) {
		this.laterality = laterality;
	}

	public String getStartDateTime() {
		return startDateTime;
	}

	public void setStartDateTime(String startDateTime) {
		this.startDateTime = startDateTime;
	}

	/**
	 * @return the objectOrigin
	 */
	public ObjectOrigin getObjectOrigin() {
		return objectOrigin;
	}

	/**
	 * @param objectOrigin the objectOrigin to set
	 */
	public void setObjectOrigin(ObjectOrigin objectOrigin) {
		this.objectOrigin = objectOrigin;
	}

	public SortedSet<Instance> getInstances() {
		return instances;
	}

	public void setInstances(SortedSet<Instance> instances) {
		this.instances = instances;
	}
	@Override
	public int compareTo(Series that) 
	{
		return this.uid.compareTo(that.uid);
	}

}
