package gov.va.med.imaging.mix.business.fhir;

import gov.va.med.imaging.mix.business.SeriesComparator;

import java.io.Serializable;
import java.util.Iterator;
// import java.util.Date;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * @author DNS DNS
 *
 * This ImagingStudy class is for FHIR ImagingStudy model support (referring to the equivalent DICOM Study term)
 * Cardinality: Patient 1..* DiagnosticReport 0..* ImagingStudy 0..* Series 0..* Instance
 */
public class ImagingStudy implements Serializable, Iterable<Series>, Comparable<ImagingStudy>
{	
	private static final long serialVersionUID = -5185851367113539916L;
	private String SiteNumber;			// where the study is created (to/from ImagingStudy's additional text extension element)
	private String SiteAbbr;			// where the study is created (to/from ImagingStudy's additional text extension element)
	private String SiteName;			// where the study is created (to/from ImagingStudy's additional text extension element)
	private String startDateTime;		// when study started (YYYYMMDD.HHMISS)
	private String patientID;			// R! patient ID from creator enterprise (ICN for VA, EDI_PI for DOD)
	private String uid;					// R! DICOM Study UID
	private String accessionNumber;		// hooks studies to one report, that is one order (max 16/DICOM)
	private String studyID;				// the Study's ID string (Max 16 chars)
	private String order;				// Order(s) that caused this study to be performed -- ? old HL7 term (Placer Order#) -- not used 
	private String modalitiesInStudy;	// One or more Modalities separated by '\'
	private String referrer; 			// Referring physician - not maintained
	private String availability; 		// ONLINE | OFFLINE | NEARLINE | UNAVAILABLE -- only ONLINE data is maintained for exchange
	private String url; 				// Location of the referenced study - not used
    private String procedure;			// Type of procedure performed - not maintained for exchange
    private String interpreter;			// Who interpreted images - not maintained for exchange
	private String description;			// Max 64 chars free text
	// here are the Read Only attributes; they have Getters only and must be set only internally by DB API call
	private Integer numberOfSeries;		// R! -- supposed to match series SET size
	private Integer numberOfInstances;	// R! -- supposed to match sum of Instance SET sizes from all series SET
	private SortedSet<Series> series = new TreeSet<Series>(new SeriesComparator());

	/**
	 * Create a new ImagingStudy
	 * @param sSiteNum StationNumber for VA, else ???)
	 * @param sSiteAbbr Site abbreviation (3+ ch) for VA, else ???)
	 * @param sSiteName Site name for VA, else ???)
	 * @param sDT when study started (YYYYMMDD.HHMISS)
	 * @param sPatID patient ID of study creator enterprise (ICN for VA, EDI_PI for DOD)
	 * @param sIUID DICOM Study UID (64)
	 * @param sAccN hooks studies to one report, that is one order (max 16/DICOM)
	 * @param sID the Study's ID string (max 16/DICOM)
	 * @param sOrder Placer Order number - not maintained
	 * @param mtysInStd One or more Modalities separated by '\'
	 * @param sRef Referring physician - not maintained
	 * @param sAvail ONLINE | OFFLINE | NEARLINE | UNAVAILABLE -- only ONLINE data is maintained for exchange 
	 * @param sUrl location of the referenced study - not used
	 * @param sProc Type of procedure performed - not maintained for exchange
	 * @param sInterp Who interpreted images - not maintained for exchange
	 * @param description free text (64/DICOM)
	 */
	public ImagingStudy(String sSiteNum, String sSiteAbbr, String sSiteName, String sDT, String sPatID, String sIUID, String sAccN, String sID, 
			     String sOrder, String mtysInStd,  String sRef, String sAvail, String sUrl, String sProc, String sInterp, String sDesc)
	{
		this.SiteNumber = sSiteNum;
		this.SiteAbbr = sSiteAbbr;
		this.SiteName = sSiteName;
		this.startDateTime = sDT;
		this.patientID = sPatID;
		this.uid = sIUID;
		this.accessionNumber = sAccN;
		this.studyID = sID;
		this.order = sOrder;
		this.modalitiesInStudy = mtysInStd;
		this.referrer = sRef;
		this.availability = sAvail;
		this.description = sDesc;
		
		this.numberOfSeries = 0;
		this.numberOfInstances = 0;
	}

	public String getSiteNumber() {
		return SiteNumber;
	}

	public void setSiteNumber(String siteNumber) {
		SiteNumber = siteNumber;
	}

	public String getSiteAbbr() {
		return SiteAbbr;
	}

	public void setSiteAbbr(String siteAbbr) {
		SiteAbbr = siteAbbr;
	}

	public String getSiteName() {
		return SiteName;
	}

	public void setSiteName(String siteName) {
		SiteName = siteName;
	}

	public String getStartDateTime() {
		return startDateTime;
	}

	public void setStartDateTime(String startDateTime) {
		this.startDateTime = startDateTime;
	}

	public String getPatientID() {
		return patientID;
	}

	public void setPatientID(String patientID) {
		this.patientID = patientID;
	}

	public String getUid() {
		return uid;
	}

	public void setUid(String studyUID) {
		this.uid = studyUID;
	}

	public String getAccessionNumber() {
		return accessionNumber;
	}

	public void setAccessionNumber(String accessionNumber) {
		this.accessionNumber = accessionNumber;
	}

	public String getStudyID() {
		return studyID;
	}

	public void setStudyID(String studyID) {
		this.studyID = studyID;
	}

	public String getOrder() {
		return order;
	}

	public void setOrder(String order) {
		this.order = order;
	}

	public String getModalitiesInStudy() {
		return modalitiesInStudy;
	}

	public void setModalitiesInStudy(String modalitiesInStudy) {
		this.modalitiesInStudy = modalitiesInStudy;
	}

	public void setReferrer(String referrer) {
		this.referrer = referrer;	// DicomUtils.reformatDicomName(name);
	}
	public String getReferrer() {
		return referrer;
	}

	public String getAvailability() {
		return availability;
	}

	public void setAvailability(String availability) {
		this.availability = availability;
	}

	public String getUrl() {
		return url;
	}

	public void setUrl(String url) {
		this.url = url;
	}

	public String getProcedure() {
		return procedure;
	}

	public void setProcedure(String procedure) {
		this.procedure = procedure;
	}
	
	public String getInterpreter() {
		return interpreter;
	}

	public void setInterpreter(String interpreter) {
		this.interpreter = interpreter;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public Integer getNumberOfSeries() {
		return numberOfSeries;
	}

	public Integer getNumberOfInstances() {
		return numberOfInstances;
	}

	public SortedSet<Series> getSeries() {
		return this.series;
	}

	public void setSeries(SortedSet<Series> series) {
		this.series = series;
	}
	
	/**
	 * Add a series to the Study as a child; increments counters
	 * @param series as a single series
	 */
	public void addSeries(Series series) 
	{
		synchronized(this.series)
		{
			this.series.add(series);
			numberOfSeries++;
			numberOfInstances += series.getInstances().size();
		}
	}

	/**
	 * Add all of the series in the given Set to the Study as children; increments counters
	 * @param series as a SortedSet
	 */
	public void addSeries(SortedSet<Series> series) 
	{
		synchronized(this.series)
		{
			this.series.addAll(series);
		}
	}

	@Override
	public Iterator<Series> iterator()
	{
		return series.iterator();
	}



	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() 
	{
		return this.uid + " of patient[" + this.patientID + "]; accessionNum=" + this.accessionNumber + "; ID=" + this.studyID + 
				"; Desc=" + this.description + "; Modalities=" + this.modalitiesInStudy + 
				"; StudyDateTime=" + this.startDateTime + " -- #Series=" + this.numberOfSeries + " #instances=" + this.numberOfInstances +
				"; of Site [" + this.SiteNumber + "/" + this.SiteAbbr + "/" + this.SiteName;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((uid == null) ? 0 : uid.hashCode());
		result = prime * result
				+ ((patientID == null) ? 0 : patientID.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final ImagingStudy other = (ImagingStudy) obj;
		if (uid == null) {
			if (other.uid != null)
				return false;
		} else if (!uid.equals(other.uid))
			return false;
		if (patientID == null) {
			if (other.patientID != null)
				return false;
		} else if (!patientID.equals(other.patientID))
			return false;
		return true;
	}	
	
	@Override
	public int compareTo(ImagingStudy that) 
	{
		return this.uid.compareTo(that.uid);
	}
}
