/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.memory;

import gov.va.med.imaging.GUID;
import gov.va.med.imaging.storage.cache.*;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.impl.CacheManagerImpl;
import gov.va.med.imaging.storage.cache.impl.eviction.StorageThresholdEvictionStrategyMemento;
import gov.va.med.imaging.storage.cache.impl.memory.MemoryCache;
import gov.va.med.server.ServerLifecycleEvent;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;

import junit.framework.TestCase;

import org.apache.log4j.Logger;

/**
 * @author       DNS
 *
 * This test creates an in-memory cache with a single region and an eviction strategy
 * that deletes groups when free space is too low.
 */
public class TestStorageThresholdEviction 
extends TestCase
{
	private static final int maxCacheInstanceCount = 32;
	private static final int instanceDataLength = 4096;
	private static final long maxCacheSize = instanceDataLength * maxCacheInstanceCount;
	private static final long minInstanceCountThreshold = 8;
	private static final long targetInstanceCountThreshold = 16;
	
	private Cache cache = null;

	private static byte[] sampleData = null;
	private Logger logger = Logger.getLogger(this.getClass());

	// create some list of bytes we can use for data
	// the data should be repeatable so tests are consistent
	static
	{
		sampleData = new byte[instanceDataLength];
		for(int index=0; index<instanceDataLength; ++index)
			sampleData[index] = (byte)(index % 256);
	}
	
	protected static byte[] getSampleData()
	{
		return sampleData;
	}


	
	protected URI getCacheUri() 
	throws URISyntaxException
	{
		return new URI(MemoryCache.protocol + "://" + this.getName() + "?" + "maxSize=" + maxCacheSize + "&maxInstanceCount=" + maxCacheInstanceCount);
	}

	/**
	 * @return the cache
	 */
	public Cache getCache()
	{
		return this.cache;
	}

	private void setCache(Cache cache)
	{
		this.cache = cache;
	}

	/* (non-Javadoc)
	 * @see junit.framework.TestCase#setUp()
	 */
	protected void setUp() 
	throws Exception
	{
		AbstractCacheTest.initializeLogging(this.getName());
		Logger.getRootLogger().info("Starting unit test '" + this.getName() + "' =======================================================");
		
		CacheManagerImpl cacheManager = CacheManagerImpl.getSingleton();
		setCache( cacheManager.createCache(this.getName(), getCacheUri(), (String)null) );
		
		StorageThresholdEvictionStrategyMemento evictionStrategyMemento = new StorageThresholdEvictionStrategyMemento();
		
		evictionStrategyMemento.setName("storageThresholdEvictionStrategy");
		evictionStrategyMemento.setDelay(0L);			// scan immediately
		evictionStrategyMemento.setInterval(1000L);		// scan every second
		//evictionStrategyMemento.setMaxUsedSpaceThreshold(maxUsedSpaceThreshold);
		evictionStrategyMemento.setMinFreeSpaceThreshold(minInstanceCountThreshold * instanceDataLength);
		evictionStrategyMemento.setTargetFreeSpaceThreshold(targetInstanceCountThreshold * instanceDataLength);
		evictionStrategyMemento.setInitialized(true);
		cacheManager.createEvictionStrategy(getCache(), evictionStrategyMemento);

		cacheManager.createRegion(getCache(), "region", new String[]{"storageThresholdEvictionStrategy"} );
		
		getCache().setInitialized(true);
		getCache().setEnabled(true);
		cacheManager.serverLifecycleEvent(new ServerLifecycleEvent(ServerLifecycleEvent.EventType.START));
	}

	/* (non-Javadoc)
	 * @see junit.framework.TestCase#tearDown()
	 */
	protected void tearDown() throws Exception
	{
		CacheManagerImpl cacheManager = CacheManagerImpl.getSingleton();
		
		cacheManager.serverLifecycleEvent(new ServerLifecycleEvent(ServerLifecycleEvent.EventType.STOP));
		
		super.tearDown();
	}

	/**
	 * Simply test that the cache regions's freespace property is functional.
	 * @throws CacheException 
	 * @throws IOException 
	 */
	public void testCacheFreespace() 
	throws CacheException, IOException
	{
		int iterations = 64;
		String[] imageIds = new String[iterations];
		
		for(int index=0; index < iterations; ++index)
		{
			imageIds[index] = (new GUID()).toShortString();
			createAndPopulateInstance( imageIds[index] );
			
			System.out.println( "Available region free space is " + getCache().getRegion("region").getFreeSpace() + " bytes." );
			try{Thread.sleep(1000L);}catch(InterruptedException iX){}		// give the eviciton thread a chance to keep up
		}
		
	}

	private void createAndPopulateInstance(String instanceId) 
	throws CacheException, IOException
	{
		String[] path = new String[]{instanceId};
		Instance instance = getCache().getOrCreateInstance("region", path, instanceId);
		
		InstanceWritableByteChannel writeChannel = instance.getWritableChannel();
		
		java.nio.ByteBuffer src = ByteBuffer.wrap( getSampleData() );
		
		writeChannel.write(src);

		writeChannel.close();
	}
}
