/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.memory;

import gov.va.med.imaging.storage.cache.InstanceByteChannelFactory;
import gov.va.med.imaging.storage.cache.InstanceByteChannelListener;
import gov.va.med.imaging.storage.cache.InstanceReadableByteChannel;
import gov.va.med.imaging.storage.cache.InstanceWritableByteChannel;
import gov.va.med.imaging.storage.cache.TracableComponent;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.PersistenceIOException;
import gov.va.med.imaging.storage.cache.impl.memory.data.MemoryInstanceData;
import gov.va.med.imaging.storage.cache.impl.memory.data.MemoryInstanceDataFactory;
import gov.va.med.imaging.storage.cache.memento.ByteChannelFactoryMemento;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.HashSet;
import java.util.Set;
import java.util.zip.Checksum;

/**
 * @author       DNS
 *
 */
public class MemoryInstanceByteChannelFactory 
implements InstanceByteChannelFactory<Integer>, InstanceByteChannelListener
{
	public final static int maxInstanceSize = 4096;
	private long sweepTime = 1000;
	private long maxChannelOpenDuration = 10000;
	private boolean traceChannelInstantiation;
	private int currentlyOpenReadableByteChannels = 0;
	private int currentlyOpenWritableByteChannels = 0;
	private MemoryInstanceDataFactory instanceFactory;
	
	public MemoryInstanceByteChannelFactory(MemoryInstanceDataFactory instanceFactory)
	{
		this.instanceFactory = instanceFactory;
	}
	
	public MemoryInstanceDataFactory getInstanceFactory()
	{
		return instanceFactory;
	}

	public int getCurrentlyOpenReadableByteChannels()
	{
		return currentlyOpenReadableByteChannels;
	}
	
	public int getCurrentlyOpenWritableByteChannels()
	{
		return currentlyOpenWritableByteChannels;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.InstanceByteChannelFactory#getInstanceReadableByteChannel(java.lang.Object, gov.va.med.imaging.storage.cache.InstanceByteChannelListener)
	 */
	public InstanceReadableByteChannel getInstanceReadableByteChannel(Integer instanceId, InstanceByteChannelListener timeoutListener)
	throws PersistenceIOException, CacheException
	{
		MemoryInstanceData instance = getInstanceFactory().get(instanceId);
		if(instance != null)
		{
			InstanceReadableByteChannel readable = new MockInstanceReadableByteChannel(instance);
			++currentlyOpenReadableByteChannels;
			return readable;
		}
		
		return null;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.InstanceByteChannelFactory#getInstanceWritableByteChannel(java.lang.Object, gov.va.med.imaging.storage.cache.InstanceByteChannelListener)
	 */
	public InstanceWritableByteChannel getInstanceWritableByteChannel(Integer instanceId, InstanceByteChannelListener timeoutListener)
	throws PersistenceIOException, CacheException
	{
		MemoryInstanceData instanceData = getInstanceFactory().getOrCreate(instanceId, maxInstanceSize);
		InstanceWritableByteChannel writable = new MockInstanceWritableByteChannel(instanceData);
		++currentlyOpenReadableByteChannels;
		return writable;
	}

	/**
	 * @param id
	 */
	public void remove(Integer instanceId)
	{
		getInstanceFactory().instanceRemoved(instanceId);
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.InstanceByteChannelFactory#getMaxChannelOpenDuration()
	 */
	public long getMaxChannelOpenDuration()
	{
		return maxChannelOpenDuration;
	}

	public void setMaxChannelOpenDuration(long max)
	{
		this.maxChannelOpenDuration = max;
	}

	public long getSweepTime()
	{
		return sweepTime;
	}

	public void setSweepTime(long sweep)
	{
		this.sweepTime = sweep;
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.InstanceByteChannelFactory#isTraceChannelInstantiation()
	 */
	public boolean isTraceChannelInstantiation()
	{
		return traceChannelInstantiation;
	}

	public void setTraceChannelInstantiation(boolean traceChannelInstantiation)
	{
		this.traceChannelInstantiation = traceChannelInstantiation;
	}

	// ==========================================================================================================
	// InstanceByteChannelListener
	// ==========================================================================================================
	public void readChannelClose(InstanceReadableByteChannel readable)
	{
		--currentlyOpenReadableByteChannels;
	}

	public void readChannelIdleTimeout(InstanceReadableByteChannel readable)
	{
		--currentlyOpenReadableByteChannels;
	}

	public void writeChannelClose(InstanceWritableByteChannel writable)
	{
		--currentlyOpenWritableByteChannels;
	}

	public void writeChannelIdleTimeout(InstanceWritableByteChannel writable)
	{
		--currentlyOpenWritableByteChannels;
	}

	// ==========================================================================================================
	// State Serialization
	// ==========================================================================================================
	public ByteChannelFactoryMemento createMemento()
	{
		return null;
	}

	public void restoreMemento(ByteChannelFactoryMemento memento)
	{
	}
	
	// ==========================================================================================================
	// ByteChannel Realizations
	// ==========================================================================================================
	/**
	 * 
	 * @author       BECKEC
	 *
	 */
	class MockInstanceReadableByteChannel
	implements InstanceReadableByteChannel, TracableComponent
	{
		private MemoryInstanceData instanceData;
		private StackTraceElement[] instantiatingStackTrace = null;
		int position = 0;
		private boolean open = true;		// the channel is open when it is created
		
		public MockInstanceReadableByteChannel(MemoryInstanceData instanceData)
		{
			this.instanceData = instanceData;
			
			instantiatingStackTrace = Thread.currentThread().getStackTrace();
		}
		
		@Override
		public StackTraceElement[] getInstantiatingStackTrace()
		{
			return instantiatingStackTrace;
		}
		
		public Checksum getChecksum()
		{
			return null;
		}

		public int read(ByteBuffer dst) 
		throws IOException
		{
			if( !open )
				throw new IOException("Invalid state, channel is already closed.");
			
			int bytesRead = Math.min(instanceData.getLimit() - position, dst.remaining() );
			if(bytesRead > 0)
				dst.put(instanceData.read(position, bytesRead));
			
			position += bytesRead;
			
			return bytesRead;
		}

		public void close() 
		throws IOException
		{
			open = false;
			notifyListeners(false);
		}

		public void error() 
		throws IOException
		{
			open = false;
			notifyListeners(true);
		}

		public boolean isOpen()
		{
			return open;
		}
		
		private Set<InstanceByteChannelListener> listeners = new HashSet<InstanceByteChannelListener>();
		public void addListener(InstanceByteChannelListener listener)
		{
			listeners.add(listener);
		}
		public void removeListener(InstanceByteChannelListener listener)
		{
			listeners.remove(listener);
		}
		private void notifyListeners(boolean errorClose)
		{
			for(InstanceByteChannelListener listener : listeners)
				if( errorClose )
					listener.readChannelIdleTimeout(this);
				else
					listener.readChannelClose(this);
		}

		/* (non-Javadoc)
		 * @see gov.va.med.imaging.storage.cache.InstanceReadableByteChannel#getLastAccessedTime()
		 */
		@Override
		public long getLastAccessedTime()
		{
			return System.currentTimeMillis();
		}

		public File getCacheFile()
		{
			return null;
		}
	}

	/**
	 * 
	 * @author       BECKEC
	 *
	 */
	class MockInstanceWritableByteChannel
	implements InstanceWritableByteChannel, TracableComponent
	{
		private MemoryInstanceData instanceData;
		int position = 0;
		private boolean open = true;		// the channel is open when it is created
		private Checksum checksum = new java.util.zip.Adler32();
		private StackTraceElement[] instantiatingStackTrace = null;
		
		public MockInstanceWritableByteChannel(MemoryInstanceData instanceData)
		{
			this.instanceData = instanceData;
			
			instantiatingStackTrace = Thread.currentThread().getStackTrace();
		}
		
		@Override
		public StackTraceElement[] getInstantiatingStackTrace()
		{
			return instantiatingStackTrace;
		}
		

		public boolean isOpen()
		{
			return open;
		}
		
		public Checksum getChecksum()
		{
			return checksum;
		}

		public int write(ByteBuffer src) 
		throws IOException
		{
			if( ! isOpen() )
				throw new IOException("Invalid state, channel is already closed.");
			
			// assure that we do not overwrite the buffer
			if( src.remaining() > (instanceData.getCapacity() - position) )
				throw new IOException("Error writing to channel, storage overflow");
			int bytesWritten = Math.min(instanceData.getCapacity() - position, src.remaining() );
			if(bytesWritten > 0)
			{
				byte[] tmp = new byte[src.remaining()];
				src.get(tmp);
				checksum.update(tmp, 0, tmp.length);
				instanceData.write(position, tmp);
			}
			
			position += bytesWritten;
			
			return bytesWritten;
		}

		public void error() 
		throws IOException
		{
			open = false;
			notifyListeners(true);
		}

		public void close() 
		throws IOException
		{
			open = false;
			notifyListeners(false);
		}

		private Set<InstanceByteChannelListener> listeners = new HashSet<InstanceByteChannelListener>();
		public void addListener(InstanceByteChannelListener listener)
		{
			listeners.add(listener);
		}
		public void removeListener(InstanceByteChannelListener listener)
		{
			listeners.remove(listener);
		}
		private void notifyListeners(boolean errorClose)
		{
			for(InstanceByteChannelListener listener : listeners)
				if( errorClose )
					listener.writeChannelIdleTimeout(this);
				else
					listener.writeChannelClose(this);
		}
		/* (non-Javadoc)
		 * @see gov.va.med.imaging.storage.cache.InstanceReadableByteChannel#getLastAccessedTime()
		 */
		@Override
		public long getLastAccessedTime()
		{
			return System.currentTimeMillis();
		}
	}

}
