/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.memory;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import gov.va.med.imaging.storage.cache.*;
import gov.va.med.imaging.storage.cache.events.GroupLifecycleEvent;
import gov.va.med.imaging.storage.cache.events.GroupLifecycleListener;
import gov.va.med.imaging.storage.cache.events.LifecycleEvent;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;

/**
 * @author       DNS
 *
 */
public class MemoryGroup 
implements Group
{
	private final String name;
	private final GroupFactory groupFactory;
	private final InstanceFactory instanceFactory;
	private final GroupSet groups;
	private final InstanceSet instanceSet;
	private Date lastAccessed = new Date();

	/**
	 * 
	 * @param name
	 * @param groupFactory
	 * @param instanceFactory
	 */
	public MemoryGroup(String name, GroupFactory groupFactory, InstanceFactory instanceFactory)
	{
		super();
		this.name = name;
		this.groupFactory = groupFactory;
		this.instanceFactory = instanceFactory;
		this.groups = new GroupSet(getGroupFactory());
		this.instanceSet = new InstanceSet(getInstanceFactory());
	}

	@Override
	public String getName()
	{
		return name;
	}

	/**
	 * @return the groupFactory
	 */
	public GroupFactory getGroupFactory()
	{
		return this.groupFactory;
	}

	/**
	 * @return the instanceFactory
	 */
	public InstanceFactory getInstanceFactory()
	{
		return this.instanceFactory;
	}

	@Override
	public Date getLastAccessed() 
	throws CacheException
	{
		return lastAccessed;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Group#getSize()
	 */
	@Override
	public long getSize() 
	throws CacheException
	{
		return 0;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.Group#remove()
	 */
	@Override
	public void delete(boolean forceDelete) 
	throws CacheException
	{
		deleteAllChildGroups(forceDelete);
		deleteAllChildInstances(forceDelete);
		notifyListeners(LifecycleEvent.DELETE);
	}
	
	// ================================================================================================================
	// Group Management
	// ================================================================================================================
	@Override
	public Group getChildGroup(String groupName) 
	throws CacheException
	{
		return groups.getByName(groupName);
	}

	@Override
	public Group getGroup(String[] groupPath) 
	throws CacheException
	{
		return groups.getByName(groupPath);
	}

	@Override
	public Iterator<? extends Group> getGroups() 
	throws CacheException
	{
		return groups.iterator();
	}

	@Override
	public Group getOrCreateGroup(String[] groupPath) 
	throws CacheException
	{
		return groups.getOrCreateByName(groupPath);
	}

	@Override
	public Group getOrCreateChildGroup(String groupName) 
	throws CacheException
	{
		return groups.getOrCreateByName(groupName);
	}

	/**
	 * Delete a Group.  The groupName is an ordered array of the
	 * instances ancestor groups, starting from the progeny of this
	 * group.
	 */
	@Override
	public void deleteGroup(String[] groupName, boolean forceDelete) 
	throws CacheException
	{
		// if the group is a child of ours then delete it
		if(groupName.length == 1)
		{
			this.deleteChildGroup(groupName[0], forceDelete);
			return;
		}
		
		Group childGroup = getChildGroup(groupName[0]);

		String[] progenyGroupNames = new String[groupName.length-1];
		System.arraycopy(groupName, 1, progenyGroupNames, 0, progenyGroupNames.length);
		childGroup.deleteGroup(progenyGroupNames, forceDelete);
		return;
	}

	public void deleteChildGroup(String groupName, boolean forceDelete) 
	throws CacheException
	{
		Group childGroup = getChildGroup(groupName);
		if(childGroup != null)
			deleteChildGroup(childGroup, forceDelete);
	}

	@Override
	public void deleteAllChildGroups(boolean forceDelete) 
	throws CacheException
	{
		for(Group group : groups)
			group.delete(forceDelete);
		
		while(groups.size() > 0)
			groups.clear();
	}

	@Override
	public void deleteChildGroup(Group childGroup, boolean forceDelete) 
	throws CacheException
	{
		if( groups.remove(childGroup) )
			childGroup.delete(forceDelete);
	}

	@Override
	public int evaluateAndEvictChildGroups(EvictionJudge<Group> judge) 
	throws CacheException
	{
		List<Group> killList = new ArrayList<Group>();
		int evictedCount = 0;

		// kill the lowest progeny first
		for(Group group : groups)
			evictedCount += group.evaluateAndEvictChildGroups(judge);
		
		for(Group group : groups)
			if(judge.isEvictable(group))
				killList.add(group);
		
		for(Group group : killList)
		{
			deleteChildGroup(group, false);		// eviction passes do not delete while another thread is using the group
			++evictedCount;
		}
		
		return evictedCount;
	}

	// ================================================================================================================
	// Instance Management
	// ================================================================================================================
	@Override
	public Instance getChildInstance(String key) 
	throws CacheException
	{
		return instanceSet.getByName(key);
	}

	public void deleteChildInstance(String key, boolean forceDelete) 
	throws CacheException
	{
		Instance childInstance = instanceSet.getByName(key);
		if(childInstance != null)
			this.deleteChildInstance(childInstance, forceDelete);
	}
	
	@Override
	public Instance getInstance(String[] path, String key) 
	throws CacheException
	{
		return getGroup(path).getChildInstance(key);
	}

	@Override
	public void deleteInstance(String[] path, String key, boolean forceDelete) 
	throws CacheException
	{
		getGroup(path).deleteChildInstance(key, forceDelete);
	}

	@Override
	public Iterator<? extends Instance> getInstances() 
	throws CacheException
	{
		return instanceSet.iterator();
	}

	@Override
	public Instance getOrCreateChildInstance(String key) 
	throws CacheException
	{
		return instanceSet.getOrCreateByName(key);
	}

	@Override
	public Instance getOrCreateInstance(String[] path, String key) 
	throws CacheException
	{
		return getOrCreateGroup(path).getOrCreateChildInstance(key);
	}

	@Override
	public void deleteAllChildInstances(boolean forceDelete) 
	throws CacheException
	{
		synchronized(instanceSet)
		{
			for(Instance instance : instanceSet)
				instance.delete(forceDelete);
			instanceSet.clear();
		}
	}

	public void deleteChildInstance(Instance childInstance, boolean forceDelete) 
	throws CacheException
	{
		instanceSet.remove(childInstance);
		childInstance.delete(forceDelete);
	}

	// ======================================================================================================
	// Listener Management
	// ======================================================================================================
	private List<GroupLifecycleListener> listeners = new ArrayList<GroupLifecycleListener>();
	@Override
	public void registerListener(GroupLifecycleListener listener)
	{
		listeners.add(listener);
	}
	
	@Override
	public void unregisterListener(GroupLifecycleListener listener)
	{
		listeners.remove(listener);
	}
	
	protected void notifyListeners(LifecycleEvent event)
	{
		GroupLifecycleEvent lifecycleEvent = new GroupLifecycleEvent(event, getName());
		for(GroupLifecycleListener listener : listeners)
			listener.notify(lifecycleEvent);
	}
	
}
