package gov.va.med.imaging.storage.cache.impl.jcifs;

import gov.va.med.imaging.storage.cache.Group;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.CacheInitializationException;
import gov.va.med.imaging.storage.cache.exceptions.CacheInternalException;
import gov.va.med.imaging.storage.cache.exceptions.RegionInitializationException;
import gov.va.med.imaging.storage.cache.exceptions.RegionNotInitializedException;
import gov.va.med.imaging.storage.cache.impl.PersistentGroupSet;
import gov.va.med.imaging.storage.cache.impl.PersistentRegion;
import gov.va.med.imaging.storage.cache.impl.memento.PersistentRegionMemento;

import java.util.Iterator;

import jcifs.smb.SmbException;
import jcifs.smb.SmbFile;

import org.apache.log4j.Logger;

/**
 * The JCIFS based implementation of a cache region.
 * 
 */
public class JcifsRegion
extends PersistentRegion
implements JcifsRegionMBean
{
	private Logger log = Logger.getLogger(this.getClass());
	private SmbFile regionDirectory = null;
	private JcifsGroupSet childGroups = null;

	// ======================================================================================================
	// Factory Methods
	// ======================================================================================================
	
	/**
	 * Create a FileSystemCacheRegion, restoring the state from a memento (as much as possible)
	 * 
	 * @param memento
	 * @param instanceFactoryChannel
	 * @param evictionStrategy
	 * @return
	 * @throws CacheInitializationException 
	 */
	public static JcifsRegion create(
		JcifsCache parentCache,
		PersistentRegionMemento memento) 
	throws RegionInitializationException
	{
		return create(
			parentCache, 
			memento.getName(),
			memento.getEvictionStrategyNames(),
			memento.getSecondsReadWaitsForWriteCompletion(), 
			memento.isSetModificationTimeOnRead());
	}
	
	/**
	 * Create a FileSysytemRegion instance
	 * 
	 * @param cacheRootDirectory
	 * @param name
	 * @param instanceFactoryChannel
	 * @param evictionStrategy
	 * @param secondsReadWaitsForWriteCompletion
	 * @param setModificationTimeOnRead
	 * @return
	 */
	public static JcifsRegion create(
		JcifsCache parentCache,
		String name,
		String[] evictionStrategyNames, 
		int secondsReadWaitsForWriteCompletion, 
		boolean setModificationTimeOnRead )
	throws RegionInitializationException
	{
		try
		{
			return new JcifsRegion(
					parentCache, 
					name, 
					evictionStrategyNames, 
					secondsReadWaitsForWriteCompletion, 
					setModificationTimeOnRead);
		} 
		catch (CacheException x)
		{
			Logger.getLogger(JcifsRegion.class).error(x);
			throw new RegionInitializationException(x);
		}
	}

	// ======================================================================================================
	// Constructors
	// ======================================================================================================
	
	/**
	 * 
	 * @param cacheRootDirectory
	 * @param name
	 * @param instanceFactoryChannel
	 * @param evictionStrategy
	 * @param secondsReadWaitsForWriteCompletion
	 * @param setModificationTimeOnRead
	 * @throws CacheException
	 */
	private JcifsRegion(
			JcifsCache parentCache,
			String name, 
			String[] evictionStrategyNames, 
			int secondsReadWaitsForWriteCompletion, 
			boolean setModificationTimeOnRead) 
	throws CacheException
	{
		super(parentCache, name, evictionStrategyNames, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
	}
	
	/**
	 * A type-converted accessor of the parent cache
	 * @return
	 */
	private JcifsCache getParentJcifsCache()
	{
		return (JcifsCache)getParentCache();
	}

	// ======================================================================================================
	// Core Accessors
	// ======================================================================================================
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.impl.jcifs.JcifsRegionMBean#getRegionDirectory()
	 */
	@Override
	public SmbFile getRegionDirectory()
	throws RegionNotInitializedException
	{
		if(! isInitialized().booleanValue())
			throw new RegionNotInitializedException("Region Directory is not set until initialization is complete");
		return regionDirectory;
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.impl.jcifs.JcifsRegionMBean#getFreeSpace()
	 */
	@Override
	public long getFreeSpace()
	{
		try
		{
			SmbFile regionRoot = getRegionDirectory();
			return regionRoot.getDiskFreeSpace();
		} 
		catch (RegionNotInitializedException x)
		{
			log.error(x);
		} 
		catch (SmbException x)
		{
			log.error(x);
		}
		return 0L;
	}

	/**
	 * The JCIFS lib does not support getting the total available space
	 */
	@Override
	public long getTotalSpace()
	{
		return 0L;
	}

	@Override
	protected PersistentGroupSet getPersistentGroupSet() 
	throws RegionNotInitializedException
	{
		if(! isInitialized().booleanValue())
			throw new RegionNotInitializedException("Region Directory is not set until initialization is complete");
		return childGroups;
	}
	
	/**
	 * Regions cannot be removed, so throw an error if someone tries.
	 */
	@Override
	public void delete(boolean forceDelete) 
	throws CacheException
	{
		throw new CacheInternalException("Illegal attempt to remove a Region.");
	}

	@Override
	protected void initialize()
	throws RegionInitializationException
	{
		log.debug("'" + this.getName() + "' initializing...");
		
		try
		{
			SmbFile cacheRootFile = getParentJcifsCache().getRootDirectory();
			
			if(this.getName().lastIndexOf('/') == this.getName().length()-1)
				this.regionDirectory = new SmbFile(cacheRootFile, this.getName() );
			else
				this.regionDirectory = new SmbFile(cacheRootFile, this.getName() + "/" );
			
			if(! regionDirectory.exists())
			{
				log.debug("'" + this.getName() + "' initializing - directory does not exist, creating...");
				regionDirectory.mkdirs();
			}
			log.debug("'" + this.getName() + "' initializing - directory exists");
			
			this.childGroups = new JcifsGroupSet(regionDirectory, getInstanceFactoryChannel(), getSecondsReadWaitsForWriteCompletion(), isSetModificationTimeOnRead());
			log.debug("'" + this.getName() + "' initialized.");
		} 
		catch (Exception x)
		{
			String msg = "Group directory '" + regionDirectory.getPath() + "' could not be accessed, " + x.getClass().getSimpleName();
			log.error(msg, x);
			throw new RegionInitializationException(msg, x);
		} 
	}
	
	// ======================================================================================================
	// Child Group Management
	// ======================================================================================================
	/**
	 * 
	 */
	@Override
	public Group getChildGroup(String groupName) 
	throws CacheException
	{
		return childGroups.getChild(groupName, false);
	}

	/**
	 * 
	 */
	@Override
	public Group getOrCreateChildGroup(String groupName) 
	throws CacheException
	{
		return childGroups.getChild(groupName, true);
	}
	
	/**
	 * All subdirectories of ourselves are child groups.
	 * @throws CacheException 
	 */
	@Override
	public Iterator<? extends Group> getGroups() 
	throws CacheException
	{
		return childGroups.hardReferenceIterator();
	}

	@Override
	public void deleteChildGroup(Group childGroup, boolean forceDelete)
	throws CacheException 
	{
		childGroups.deleteChild(childGroup, forceDelete);
	}

	@Override
	public void deleteAllChildGroups(boolean forceDelete) 
	throws CacheException 
	{
		childGroups.deleteAll(forceDelete);
	}


}
