package gov.va.med.imaging.storage.cache.impl.jcifs;

import gov.va.med.imaging.storage.cache.Instance;
import gov.va.med.imaging.storage.cache.InstanceByteChannelFactory;
import gov.va.med.imaging.storage.cache.InstanceByteChannelListener;
import gov.va.med.imaging.storage.cache.InstanceReadableByteChannel;
import gov.va.med.imaging.storage.cache.InstanceWritableByteChannel;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.InstanceInaccessibleException;
import gov.va.med.imaging.storage.cache.exceptions.PersistenceException;
import gov.va.med.imaging.storage.cache.exceptions.PersistenceIOException;
import gov.va.med.imaging.storage.cache.impl.PersistentInstance;

import java.io.*;
import java.net.MalformedURLException;
import java.util.Date;

import jcifs.smb.SmbException;
import jcifs.smb.SmbFile;
import jcifs.smb.SmbFileInputStream;
import jcifs.smb.SmbFileOutputStream;

/**
 * This class is the binding from an abstract PersistentInstance to a 
 * file system implementation.
 *  
 */
public class JcifsInstance 
extends PersistentInstance
implements Comparable<Instance>
{
	public final static String checksumFileExtension = ".checksum";
	private SmbFile instanceFile = null;

	// =======================================================================================================================================
	/**
	 * @param instanceFile
	 * @param instanceFactoryChannel
	 * @return
	 * @throws CacheException
	 */
	public static JcifsInstance getInstance(
		SmbFile instanceFile, 
		InstanceByteChannelFactory instanceFactoryChannel,
		int secondsReadWaitsForWriteCompletion,
		boolean setModificationTimeOnRead
	) 
	throws CacheException
	{
		try
		{
			if(instanceFile.exists())
				return new JcifsInstance(instanceFile, false, instanceFactoryChannel, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Unable to determine the existence of '" + instanceFile.getPath() + "'", x);
		}
		
		return null;
	}
	
	/**
	 * 
	 * @param instanceFile
	 * @param instanceFactoryChannel
	 * @return
	 * @throws CacheException
	 */
	public static JcifsInstance getOrCreateInstance(
		SmbFile instanceFile, 
		InstanceByteChannelFactory instanceFactoryChannel,
		int secondsReadWaitsForWriteCompletion,
		boolean setModificationTimeOnRead
	) 
	throws CacheException
	{
		return new JcifsInstance(instanceFile, true, instanceFactoryChannel, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
	}
	
	// =======================================================================================================================================
	/**
	 * @param instanceFile
	 * @param createIfNotExist
	 * @param instanceFactoryChannel
	 * @throws CacheException
	 */
	private JcifsInstance(
		SmbFile instanceFile, 
		boolean createIfNotExist, 
		InstanceByteChannelFactory instanceFactoryChannel,
		int secondsReadWaitsForWriteCompletion,
		boolean setModificationTimeOnRead
	)
	throws CacheException
	{
		super(instanceFactoryChannel, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
		this.instanceFile = instanceFile;
		if(!isPersistent() && !createIfNotExist )
			throw new InstanceInaccessibleException();
		
		// load the checksum value if it exists
		loadChecksum();
	}

	/**
	 * 
	 */
	public String getName()
	{
		return instanceFile.getName();
	}
	
	// ================================================================================================================================
	// The type specific methods to tie this class to the persistent storage
	// ================================================================================================================================
	@SuppressWarnings("unchecked")
	@Override
	public InstanceByteChannelFactory<SmbFile> getInstanceChannelFactory()
	{
		return super.getInstanceChannelFactory();
	}

	protected InstanceReadableByteChannel createInstanceReadableByteChannel() 
	throws PersistenceIOException, CacheException
	{
		return getInstanceChannelFactory().getInstanceReadableByteChannel(this.getFile(), this);
	}

	protected InstanceWritableByteChannel createInstanceWritableByteChannel() 
	throws PersistenceIOException, CacheException
	{
		return getInstanceChannelFactory().getInstanceWritableByteChannel(this.getFile(), this);
	}
	
	public boolean isPersistent() 
	throws PersistenceException
	{
		try
		{
			return getFile().exists();
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Unable to determine the existence of '" + instanceFile.getPath() + "'", x);
		}
	}
	
	protected void createPersistent() 
	throws PersistenceIOException
	{
		try
		{
			getFile().createNewFile();
		} 
		catch (IOException ioX)
		{
			throw new PersistenceIOException("Error creating new file '" + getFile().getCanonicalPath() + "'.", ioX);
		}
	}
	
	@Override
	protected void removePersistent()
	throws PersistenceIOException
	{
		try
		{
			getFile().delete();
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Error removing SMB file '" + getFile().getPath() + "'.", x);
		}
	}
	
	@Override
	protected void setLastModified(long date) 
	throws PersistenceIOException
	{
		try
		{
			instanceFile.setLastModified(date);
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Error setting modification time for SMB file '" + getFile().getPath() + "'.", x);
		}
	}
	
	
	public SmbFile getFile()
	{
		return instanceFile;
	}
	
	public String getMediaType()
	{
		return null;
	}
	
	public Date getLastAccessed() 
	throws PersistenceIOException
	{
		try
		{
			return new Date(getFile().lastModified());
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Error getting last accessed time for SMB file '" + getFile().getPath() + "'.", x);
		}
	}

	public long getSize() 
	throws PersistenceIOException
	{
		try
		{
			return getFile().length();
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Error getting size of SMB file '" + getFile().getPath() + "'.", x);
		}
	}

	public int compareTo(Instance o)
	{
		if(o instanceof JcifsInstance)
		{
			JcifsInstance that = (JcifsInstance)o;
			if( this.instanceFile.equals(that.instanceFile) )
				return 0;
			return this.instanceFile.getCanonicalPath().compareTo(that.instanceFile.getCanonicalPath());
		}
		return -1;
	}
	
	// ================================================================================================================================
	// The type specific methods to tie this class to the persistent storage for storing checksum
	// ================================================================================================================================
	private String createChecksumFileName()
	{
		return instanceFile.getPath() + checksumFileExtension;
	}

	private SmbFile getChecksumFile() 
	throws MalformedURLException
	{
		String checksumFileName = createChecksumFileName();
		return new SmbFile(checksumFileName);
	}
	
	@Override
	protected InputStream openChecksumInputStream()
	throws IOException
	{
		SmbFile checksumFile = getChecksumFile();
		if( ! checksumFile.exists() )
			return null;
		return new SmbFileInputStream( checksumFile );
	}

	@Override
	protected OutputStream openChecksumOutputStream()
	throws IOException
	{
		SmbFile checksumFile = getChecksumFile();
		return new SmbFileOutputStream( checksumFile );
	}

	@Override
	protected boolean persistentChecksumExists() 
	throws IOException 
	{
		return getChecksumFile().exists();
	}

	@Override
	protected void removeChecksumPersistent() 
	throws PersistenceIOException 
	{
		try 
		{
			if(persistentChecksumExists())
				getChecksumFile().delete();
		} 
		catch (Exception e) 
		{
			throw new PersistenceIOException(e);
		}
	}
}
