/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: 
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:        DNS
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (301) 734-0100.
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.awiv.server;

import gov.va.med.PatientArtifactIdentifier;
import gov.va.med.PatientIdentifier;
import gov.va.med.RoutingToken;
import gov.va.med.RoutingTokenImpl;
import gov.va.med.SERIALIZATION_FORMAT;
import gov.va.med.URN;
import gov.va.med.URNFactory;
import gov.va.med.exceptions.RoutingTokenFormatException;
import gov.va.med.imaging.AbstractImagingURN;
import gov.va.med.imaging.BhieStudyURN;
import gov.va.med.imaging.ImageURN;
import gov.va.med.imaging.StudyURN;
import gov.va.med.imaging.awiv.AWIVClientContext;
import gov.va.med.imaging.awiv.AWIVClientRouter;
import gov.va.med.imaging.awiv.business.AwivArtifactResults;
import gov.va.med.imaging.awiv.business.AwivStudyDetails;
import gov.va.med.imaging.awiv.client.PatientStudyService;
import gov.va.med.imaging.awiv.exceptions.AwivConnectionException;
import gov.va.med.imaging.awiv.exceptions.AwivInsufficientPatientSensitivityException;
import gov.va.med.imaging.awiv.exceptions.AwivMethodException;
import gov.va.med.imaging.awiv.server.translator.AwivClientTranslator;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.InsufficientPatientSensitivityException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.RoutingTokenHelper;
import gov.va.med.imaging.exchange.business.ArtifactResults;
import gov.va.med.imaging.exchange.business.Study;
import gov.va.med.imaging.exchange.business.StudyFilter;
import gov.va.med.imaging.transactioncontext.TransactionContext;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;

public class PatientStudyServiceImpl 
extends AbstractAwivClientRemoteServiceServlet 
implements PatientStudyService 
{

	private static final long serialVersionUID = 2256886039895143898L;

	@Override
	public AwivArtifactResults getPatientStudies(String transactionId, String patientIcn, 
			String routingTokenString, int maximumAllowedSensitivity, String cls)
	throws AwivConnectionException, AwivMethodException, AwivInsufficientPatientSensitivityException
	{
		try
		{
			AWIVClientRouter router = AWIVClientContext.getRouter();
			// The routingTokenString is a VA Radiology routing token created based on the site number from the 
			// treating facility list
			RoutingToken routingToken = RoutingTokenImpl.parse(routingTokenString);
			// convert this routing token to the site appropriate one, for VA sites this will be the same as
			// routingToken, for DoD this will be a DoD radiology routingToken which is more consistent with
			// how things work for Clinical Display.
			RoutingToken siteAppropriateRoutingToken = RoutingTokenHelper.createSiteAppropriateRoutingToken(routingToken.getRepositoryUniqueId());
			StudyFilter filter = new StudyFilter();		
			filter.setStudy_class(cls);
			filter.setMaximumAllowedLevel(AwivClientTranslator.translatePatientSensitiveCode(maximumAllowedSensitivity));
			setTransactionContextProperties(transactionId, "getPatientStudies", patientIcn);
			ArtifactResults artifactResults = 
				router.getShallowArtifactResultsForPatientFromSiteWithCaching(siteAppropriateRoutingToken, 
						PatientIdentifier.icnPatientIdentifier(patientIcn),
						filter, true, true);
						
			AwivArtifactResults results = AwivClientTranslator.translate(artifactResults);
			setEntriesReturned(results == null ? 0 : results.size());
			return results;
		}
		catch(RoutingTokenFormatException rtfX)
		{
			throw new AwivMethodException(rtfX);
		}
		catch(URNFormatException urnfX)
		{
			throw new AwivMethodException(urnfX);
		}
		catch(InsufficientPatientSensitivityException ipsX)
		{			
			throw new AwivInsufficientPatientSensitivityException(ipsX.getSensitiveValue().getSensitiveLevel().getCode(), 
					ipsX.getSensitiveValue().getWarningMessage());
		}
		catch(ConnectionException cX)
		{
			throw new AwivConnectionException(cX);
		}
		catch(MethodException mX)
		{
			throw new AwivMethodException(mX);
		}		
	}

	@Override
	public String getStudyReport(String transactionId, String studyIdentifier)
	throws AwivConnectionException, AwivMethodException
	{
		try
		{
			AWIVClientRouter router = AWIVClientContext.getRouter();
			StudyURN studyUrn = null;
			URN urn = URNFactory.create(studyIdentifier, SERIALIZATION_FORMAT.CDTP);		
			TransactionContext transactionContext = TransactionContextFactory.get();
			if(urn instanceof StudyURN)
			{
				studyUrn = (StudyURN)urn;
			}
			else if(urn instanceof ImageURN)
			{
				ImageURN imageUrn = (ImageURN)urn;
				studyUrn = imageUrn.getParentStudyURN();
			}
			else
			{
				// set the transaction context properties so the transaction properties are set
				if(urn instanceof PatientArtifactIdentifier)
				{
					PatientArtifactIdentifier paid = (PatientArtifactIdentifier)urn;
					setTransactionContextProperties(transactionId, "getStudyReport", paid.getPatientIdentifier(), studyIdentifier);	
				}
				else
				{
					setTransactionContextProperties(transactionId, "getStudyReport");
					transactionContext.setUrn(studyIdentifier);
				}				
				return "1^^\nReport is not available for this study";
			}
			setTransactionContextProperties(transactionId, "getStudyReport", studyUrn.getPatientId(), studyIdentifier);
			
			Study study = router.getPatientStudy(studyUrn);
			if(study != null)
			{
				transactionContext.setFacadeBytesSent(study.getRadiologyReport() == null ? 0L : study.getRadiologyReport().length());
				return study.getRadiologyReport();
			}
			getLogger().warn("Got null study back from router for study Id '" + studyIdentifier + "', returning null report.");
			return null;			
		}
		catch(URNFormatException urfnX)
		{
			throw new AwivMethodException(urfnX);
		}
		catch(ConnectionException cX)
		{
			throw new AwivConnectionException(cX);
		}
		catch(MethodException mX)
		{
			throw new AwivMethodException(mX);
		}
	}

	@Override
	public AwivStudyDetails getStudyDetails(String transactionId,
			String studyIdentifier) 
	throws AwivConnectionException, AwivMethodException
	{
		StudyURN studyUrn = null;
		try
		{
			// create either a StudyURN or a BhieStudyURn depending on the community ID
			// found in the study ID string
			studyUrn = URNFactory.create(studyIdentifier, SERIALIZATION_FORMAT.CDTP, StudyURN.class);
		}
		catch (Throwable x)
		{
			getLogger().error(x);
			throw new AwivMethodException(x);
		}
		try
		{
			Study study = null;
			
			AWIVClientRouter rtr = AWIVClientContext.getRouter();
			if(studyUrn instanceof StudyURN)
			{
				// update the transaction context with patientId
				setTransactionContextProperties(transactionId, "getStudyDetails", 
						studyUrn.getPatientId(), studyIdentifier);
				study = rtr.getPatientStudyWithDeletedImages((StudyURN)studyUrn, 
						false);
			}
			else if(studyUrn instanceof BhieStudyURN)
			{
				// update the transaction context with patientId
				setTransactionContextProperties(transactionId, "getStudyDetails", ((BhieStudyURN)studyUrn).getPatientId(), studyIdentifier);
				study = rtr.getPatientStudy((BhieStudyURN)studyUrn);
			}
			
			return AwivClientTranslator.translate(study);
		}
		catch(URNFormatException urfnX)
		{
			throw new AwivMethodException(urfnX);
		}
		catch(ConnectionException cX)
		{
			throw new AwivConnectionException(cX);
		}
		catch(MethodException mX)
		{
			throw new AwivMethodException(mX);
		}
	}

	@Override
	public String getStudyInformation(String transactionId,
			String studyIdentifier) 
	throws AwivConnectionException, AwivMethodException
	{
		AbstractImagingURN urn = null;
		try
		{
			// create either a StudyURN or a BhieStudyURn depending on the community ID
			// found in the study ID string
			urn = URNFactory.create(studyIdentifier, AbstractImagingURN.class);
		}
		catch (Throwable x)
		{
			getLogger().error(x);
			throw new AwivMethodException(x);
		}
		try
		{
			if(urn instanceof BhieStudyURN)
			{
				return "<br>Image information not available for DoD studies";
			}			
			
			setTransactionContextProperties(transactionId, "getStudyInformation", 
					urn.getPatientId(), studyIdentifier);
			AWIVClientRouter rtr = AWIVClientContext.getRouter();
			String imageInformation = rtr.getImageInformation(urn);
			imageInformation = AwivClientTranslator.translateNewLinesToBreaks(imageInformation);
			TransactionContextFactory.get().setFacadeBytesSent(imageInformation == null ? 0L : imageInformation.length());
			return imageInformation;
		}
		catch(ConnectionException cX)
		{
			throw new AwivConnectionException(cX);
		}
		catch(MethodException mX)
		{
			throw new AwivMethodException(mX);
		}
	}
}
