/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Jan 19, 2012
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:        DNS
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (DNS
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.awiv.client;

import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.Window.Location;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.smartgwt.client.util.SC;

import gov.va.med.imaging.awiv.business.AwivServerInformation;
import gov.va.med.imaging.awiv.business.AwivTimer;
import gov.va.med.imaging.awiv.business.AwivUserInformation;
import gov.va.med.imaging.awiv.client.events.AwivTimerRunEvent;
import gov.va.med.imaging.awiv.client.ui.widgets.AwivConstants;
import gov.va.med.imaging.awiv.client.ui.widgets.AwivExceptionHandler;
import gov.va.med.imaging.awiv.client.ui.widgets.AwivViewerManager;
import gov.va.med.imaging.awiv.client.ui.widgets.JavaScriptMethods;
import gov.va.med.imaging.awiv.client.ui.widgets.dialogs.SessionExpiredDialog;
import gov.va.med.imaging.awiv.client.ui.widgets.dialogs.TimeoutWarningDialog;

/**
 * @author       DNS
 *
 */
public class AwivHelper
{
	private static Logger logger = Logger.getLogger("");
	private static AwivServerInformation awivServerInformation = null;
	
	/**
	 * This is the proper way to reload the AWIV and force the user to authenticate again. 
	 * This ensures the AWIV is closed before refreshing the page
	 */
	public static void reloadAwiv()
	{
		AwivViewerManager.closeAwivViewer();
		JavaScriptMethods.reloadPage();
	}
	
	//private static Timer sessionTimer = null;
	private static AwivTimer sessionTimer = null;
	
	private static void startSessionTimer()
	{
		/*
		sessionTimer = new Timer()
		{
			@Override
			public void run()
			{
				TimeoutWarningDialog.show();				
			}
		};*/
		sessionTimer = new AwivTimer(new AwivTimerRunEvent()
		{
			
			@Override
			public void OnTimerRun()
			{
				TimeoutWarningDialog.show();
			}
		});
		
		int timeout = 1000 * 60 * getSessionTimeout();
		logger.info("Restarting session timer to '" + timeout + "' ms");
		sessionTimer.scheduleRepeating(timeout); 
	}
	
	public static void resetSessionTimer()
	{
		stopSessionTimer();
		startSessionTimer();
	}
	
	public static void stopSessionTimer()
	{
		if(sessionTimer != null)
		{
			sessionTimer.cancel();
		}
		sessionTimer = null;
	}
	
	/**
	 * This terminates the session on the server forcing a re-authentication
	 */
	public static void logout()
	{		
		AwivViewerManager.closeAwivViewer();
		// call service to invalidate the session
		clearUserCookie();		
		UserSessionServiceAsync userService = UserSessionService.Util.getInstance();
		userService.logout(new AsyncCallback<Boolean>()
		{
			@Override
			public void onFailure(Throwable arg0)
			{
				//Window.alert("Error calling logout on server, " + arg0.getMessage());
				// don't do anything, just force a logout
				stopSessionTimer();
				reloadAwiv();
			}

			@Override
			public void onSuccess(Boolean result)
			{
				// logout was successful
				stopSessionTimer();
				reloadAwiv();
			}					
		});
	}
	
	public static void clearUserCookie()
	{
		//Cookies.removeCookie(AwivConstants.userCookieName);
		// set the cookie with no value and expired in 1970
		Cookies.setCookie(AwivConstants.userCookieName, "", new Date(0), null, "/Awiv", false);
	}
	
	/**
	 * This method pings the server to keep the session alive
	 * @deprecated This method is no longer needed because sessions are no longer used
	 */
	public static void pingServer()
	{
		UserSessionServiceAsync userService = UserSessionService.Util.getInstance();
		AwivHelper.resetSessionTimer();
		userService.isSessionStillActive(new AsyncCallback<Boolean>()
		{
			@Override
			public void onFailure(Throwable arg0)
			{
				if(!AwivExceptionHandler.handleServiceException(arg0))
				{
					logger.severe("Error determining if session is still active, " + arg0.getMessage());
					//Window.alert("Error determining if session is still active, " + arg0.getMessage());
					stopSessionTimer();
					AwivViewerManager.closeAwivViewer();
					SessionExpiredDialog.DisplaySessionExpiredDialog();
				}
			}

			@Override
			public void onSuccess(Boolean result)
			{
				if(result)
				{
					// all is fine, do nothing
				}
				else
				{
					stopSessionTimer();
					AwivViewerManager.closeAwivViewer();
					SessionExpiredDialog.DisplaySessionExpiredDialog();
				}						
			}					
		});
	}
	
	public static boolean isDebugMode()
	{
		String debugParameter = Window.Location.getParameter("debug");
		boolean debugEnabled = (debugParameter != null && "true".equalsIgnoreCase(debugParameter));
		return debugEnabled;
	}
	
	/**
	 * Determine if the user has the necessary keys/permission to view patient images. 
	 * @param awivUserInformation
	 * @return
	 */
	public static boolean canUserViewPatients(AwivUserInformation awivUserInformation)
	{
		if(awivUserInformation != null)
		{
			if(awivUserInformation.isClaimsAuthentication())
			{
				logger.info("User is authenticated to the CLAIMS system, allowed to view patient without security keys");
				return true;
			}
			
			// if the user has the MAG SYSTEM key they can view images regardless of the photo ID key
			if(awivUserInformation.isImagingSystemUser())
				return true;
			
			// if the user has this key then they can only view the photo ID image
			if(awivUserInformation.userHasKey(AwivConstants.magPatPhotoOnlySecurityKey))
			{				
				// user has the key
				logger.info("User has the '" + AwivConstants.magPatPhotoOnlySecurityKey + "', cannot view patient images");
				return false;
			}
			return true;
		}
		logger.log(Level.INFO, "Null user information when determining if user can view patient - not sure how that happened.");
		return false;
	}

	/**
	 * Get the timeout for the session
	 * @return
	 */
	public static int getSessionTimeout()
	{
		if(awivServerInformation == null)
			return AwivConstants.idleTimeout;
		return awivServerInformation.getIdleTimeoutMinutes();
	}
	
	public static void setAwivServerInformation(AwivServerInformation awivServerInformation)
	{
		AwivHelper.awivServerInformation = awivServerInformation;
		logger.info("Idle timeout set to '" + AwivHelper.awivServerInformation.getIdleTimeoutMinutes() + "' minutes.");
	}
	
	/**
	 * Determine if the user is using an IE browser
	 * @return
	 */
	public static boolean isBrowserIE()
	{
		String userAgent = JavaScriptMethods.getUserAgent();
		if(userAgent.contains(AwivConstants.ieUserAgent))
		{
			return true;
		}
		return false;
	}
	
	/**
	 * Warning message to show to the user when they try to view a study with an unsupported browser 
	 */
	public static void displayUnsupportedBrowserWarning()
	{
		String userAgent = JavaScriptMethods.getUserAgent();
		SC.say("Unsupported Browser",
				"Your browser <i>" + userAgent + "</i> is not supported for viewing images, only Microsoft Internet Explorer is supported");
	}
	
	/**
	 * Warning message to display to users who are not allowed to view a specific study
	 */
	public static void displayUnallowedToViewStudyWarning()
	{
		SC.say("Cannot View Images",
			"You are not authorized to view these images.<br>Clinical images require the <i>MAGDISP CLIN</i> key.<br>");
	}
	
	/**
	 * Get the URL of the AWIV using SSL (to use with redirection)
	 * @return
	 */
	public static String getSslUrl()
	{
		StringBuilder url = new StringBuilder();
		url.append(AwivConstants.sslHttpProtocol);
		url.append("//");
		url.append(Location.getHostName());
		url.append(":");
		url.append(getSslPort());
		url.append(Location.getPath());		
		return url.toString();
	}
	
	private static String getSslPort()
	{
		String sslPort = Cookies.getCookie(AwivConstants.sslPortCookieName);
		if(sslPort == null || sslPort.length() <= 0)
			sslPort = AwivConstants.defaultSslPort;
		return sslPort;
	}
	
	public static String getInsufficientPermissionsWarningMessage()
	{
		StringBuilder sb = new StringBuilder();
		sb.append("You do not have sufficient permissions to access patient data through the AWIV<br>");
		sb.append("You must meet one of the following:");
		sb.append("<ul>");
		sb.append("<li>Have the MAGDISP CLIN security key</li>");
		sb.append("<li>Have the MAGDISP ADMIN security key</li>");
		sb.append("<li>Have the MAG PAT PHOTO ONLY security key</li>");
		sb.append("<li>Be a CLAIMS user</li>");
		sb.append("</ul>");	
		
		return sb.toString();
	}
	
	public static String getUserViewableImageClass(AwivUserInformation awivUserInformation)
	{
		String result = "";
		if(awivUserInformation.isClaimsAuthentication())
			return result; // everything
		/*
		if(awivUserInformation.userHasKey(AwivConstants.magSystemSecurityKey))
			return result; // everything
			*/
		// JMW 3/6/2012 - the MAG SYSTEM key does NOT allow a user to view everything (based on 
		// how Clinical Display works).
		if(awivUserInformation.userHasKey(AwivConstants.magDispAdminSecurityKey) && 
				awivUserInformation.userHasKey(AwivConstants.magDispClinSecurityKey))
		{
			// user has both keys, show them everything
			return result;
		}
				
		if(awivUserInformation.userHasKey(AwivConstants.magDispAdminSecurityKey))
		{
			result = "ADMIN,ADMIN/CLIN,CLIN/ADMIN";
		}
		else if(awivUserInformation.userHasKey(AwivConstants.magDispClinSecurityKey))
		{
			result = "CLIN,CLIN/ADMIN,ADMIN/CLIN";
		}
		return result;
	}
	
	/**
	 * Display the error message in a warning dialog
	 * @param errorDescription Description of the problem
	 * @param t The exception that caused the error
	 */
	public static void displayErrorMessage(String errorDescription, Throwable t)
	{
		StringBuilder error = new StringBuilder();
		error.append(errorDescription);
		error.append(", please try again. If the error persists contact your administrator<hr><b>Error Details:</b><br> " + t.getMessage());
		SC.warn(error.toString());
	}
	
	/**
	 * Determine if the user has the necessary keys to override a 'Needs Review' study and view it
	 * 
	 * @param awivUserInformation
	 * @return
	 */
	public static boolean canUserViewNeedsReviewStudies(AwivUserInformation awivUserInformation)
	{
		return (awivUserInformation.userHasKey(AwivConstants.magSystemSecurityKey) || 
				awivUserInformation.userHasKey(AwivConstants.magEditSecurityKey));
	}
}
