/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Dec 7, 2011
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:        DNS
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (DNS
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.awiv.client;

import java.util.logging.Level;
import java.util.logging.Logger;

import gov.va.med.imaging.awiv.business.AwivUserInformation;
import gov.va.med.imaging.awiv.business.Patient;
import gov.va.med.imaging.awiv.business.UUID;
import gov.va.med.imaging.awiv.client.dao.PatientLookupServiceDao;
import gov.va.med.imaging.awiv.client.dao.UserServiceDao;
import gov.va.med.imaging.awiv.client.events.AboutDialogEvent;
import gov.va.med.imaging.awiv.client.events.LogVisibleEvent;
import gov.va.med.imaging.awiv.client.events.PatientSelectedEvent;
import gov.va.med.imaging.awiv.client.events.ShowPatientSelectEvent;
import gov.va.med.imaging.awiv.client.events.UserAuthenticatedEvent;
import gov.va.med.imaging.awiv.client.events.UserLogoutEvent;
import gov.va.med.imaging.awiv.client.ui.widgets.AwivConstants;
import gov.va.med.imaging.awiv.client.ui.widgets.AwivExceptionHandler;
import gov.va.med.imaging.awiv.client.ui.widgets.AwivViewerManager;
import gov.va.med.imaging.awiv.client.ui.widgets.Masthead;
import gov.va.med.imaging.awiv.client.ui.widgets.TabArea;
import gov.va.med.imaging.awiv.client.ui.widgets.dialogs.AuthenticateUserDialog;
import gov.va.med.imaging.awiv.client.ui.widgets.dialogs.WaitDialog;
import gov.va.med.imaging.awiv.exceptions.AwivUserNotLoggedInException;

import com.google.gwt.core.client.EntryPoint;
import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.resources.client.ClientBundle;
import com.google.gwt.resources.client.CssResource;
import com.google.gwt.resources.client.CssResource.NotStrict;
import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.Window.ClosingEvent;
import com.google.gwt.user.client.Window.Location;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.RootLayoutPanel;
import com.smartgwt.client.types.VerticalAlignment;
import com.smartgwt.client.util.BooleanCallback;
import com.smartgwt.client.util.SC;
import com.smartgwt.client.widgets.Label;
import com.smartgwt.client.widgets.layout.HLayout;
import com.smartgwt.client.widgets.layout.VLayout;

/**
 * This is the main entry point for the application
 * 
 * @author       DNS
 *
 */
public class Awiv
implements EntryPoint, PatientSelectedEvent, LogVisibleEvent, ShowPatientSelectEvent, 
GWT.UncaughtExceptionHandler, UserLogoutEvent, Window.ClosingHandler, 
AboutDialogEvent, UserAuthenticatedEvent
{
	//private static final int NORTH_HEIGHT = ApplicationMenu.APPLICATION_MENU_HEIGHT + Masthead.MASTHEAD_HEIGHT;// 85; // MASTHEAD_HEIGHT + APPLICATION_MENU_HEIGHT
	private static final int NORTH_HEIGHT = Masthead.MASTHEAD_HEIGHT;

	private VLayout mainLayout;
	private HLayout northLayout;
	private HLayout southLayout;
	private TabArea tabArea;
	private Masthead mastHead = null;
	private static Logger logger = Logger.getLogger("");
	private AuthenticateUserDialog authenticateUserDialog = null;
	
	private AwivUserInformation awivUserInformation = null;
	
	  
	interface GlobalResources
			extends ClientBundle
	{
		@NotStrict
		@Source("Awiv.css")
		CssResource css();
	}

	public void onModuleLoad()
	{
		GWT.<GlobalResources> create(GlobalResources.class).css()
			.ensureInjected();
		
	    // get rid of scroll bars, and clear out the window's built-in margin,
	    // because we want to take advantage of the entire client area
	    Window.enableScrolling(false);
	    Window.setMargin("0px");
	    
	    // initialize the main layout container
	    mainLayout = new VLayout();
	    mainLayout.setWidth100();  
	    mainLayout.setHeight100();  
	    
	    // initialize the North layout container
	    northLayout = new HLayout();  
	    northLayout.setHeight(NORTH_HEIGHT); 
	    VLayout vLayout = new VLayout(); 
	    // add the Masthead to the nested layout container
	    mastHead = new Masthead(this, this, this);
	    vLayout.addMember(mastHead);
	    // add the Application menu to the nested layout container
	    //vLayout.addMember(new ApplicationMenu(this, this));
	    // add the nested layout container to the  North layout container
	    northLayout.addMember(vLayout);
	    // initialize the East layout container
	    tabArea = new TabArea(this); 
	    // initialize the South layout container
	    southLayout = new HLayout(); 
	    // set the Navigation Pane and ContextArea as members of the South 
	    // layout container 
	    southLayout.setMembers(tabArea);  
	    // add the North and South layout containers to the main layout container
	    mainLayout.addMember(northLayout);  
	    mainLayout.addMember(southLayout); 

	    // add the main layout container to GWT's root panel 
	    RootLayoutPanel.get().add(mainLayout);	    
	    
	    GWT.setUncaughtExceptionHandler(this);	    
	    
	    mainLayout.setRedrawOnResize(false);
	    Window.addWindowClosingHandler(this);

	    // start by loading information about the user
	    //loadUserInformation();
	    
	    //authenticateUser();
	    Scheduler.get().scheduleDeferred(new Scheduler.ScheduledCommand () {
	        public void execute () 
	        {
	        	authenticateUser();
	        }
	    });
	}
	
	/**
	 * Check if the user is authenticated, if not request credentials. Also ensure using SSL connection 
	 */
	private void authenticateUser()
	{
		if(!AwivConstants.sslHttpProtocol.equalsIgnoreCase(Location.getProtocol()))
		{
			SC.warn("The VistA Imaging AWIV requires a secure connection, click OK to redirect with SSL", new BooleanCallback()
			{				
				@Override
				public void execute(Boolean value)
				{
					String newUrl = AwivHelper.getSslUrl();
					Location.assign(newUrl);
				}
			});
		}
		else
		{			
			String userCookie = Cookies.getCookie(AwivConstants.userCookieName);
			if(userCookie == null || userCookie.length() <= 0)
			{
				// authenticate the user
				authenticateUserDialog = new AuthenticateUserDialog(this, getSpecifiedSiteNumber());
				authenticateUserDialog.show();				
			}
			else
			{
				// the cookie is validated when getting user information
				loadUserInformation();
			}
		}
	}
	
	/**
	 * Retrieve user information from the server
	 */
	private void loadUserInformation()
	{
		final com.smartgwt.client.widgets.Window waitDialog = new com.smartgwt.client.widgets.Window();
		
		waitDialog.setWidth(360);  
		waitDialog.setHeight(115);  
		waitDialog.setTitle("Loading User Information");  
		waitDialog.setShowMinimizeButton(false);
		waitDialog.setShowCloseButton(false);
		waitDialog.setIsModal(true);  
		waitDialog.setShowModalMask(true);  
		waitDialog.centerInPage();  
        
        Label label = new Label("Please wait while loading user information");  
        label.setHeight100();  
        label.setPadding(5);  
        label.setValign(VerticalAlignment.TOP);  
        
        waitDialog.addItem(label);  
        
        waitDialog.show();
	
        String transactionId = UUID.uuid();
		logger.info("Retrieving user information with transaction Id '" + transactionId + "'.");
		
		UserServiceDao.userServiceDao.getUserInformation(transactionId, new AsyncCallback<AwivUserInformation>()
		{
			
			@Override
			public void onSuccess(AwivUserInformation arg0)
			{
				waitDialog.hide();
				setUserInformation(arg0);
				if(!isUserAllowedAccess())
				{
					logger.warning("User does not have sufficient permissions to access AWIV, alerting user and logging out.");
					SC.warn(AwivHelper.getInsufficientPermissionsWarningMessage(), new BooleanCallback()
					{						
						@Override
						public void execute(Boolean value)
						{							
							OnLogoutUser();
						}
					});
				}

				// if the user set the debug parameter and has the MAG SYSTEM key, allow the debug tab
				// JMW 2/27/2012 - the debug tab doesn't have anything useful on it right now so disabling it
				/*
			    if(AwivHelper.isDebugMode() && arg0.userHasKey(AwivConstants.magSystemSecurityKey))
			    	tabArea.displayDebugTab();
			    	*/
				AwivHelper.resetSessionTimer();
				
				if(isPatientSpecified())
				{
					String icn = getSpecifiedPatientIcn();
					logger.info("Patient Id '" + icn + "' has been specified, will only allow viewing this patient.");
					// kill the cookie!
					Cookies.removeCookie(AwivConstants.patientCookieName);
					getPatientInformation(icn);
				}
				else
				{				
					if(arg0.isClaimsAuthentication())
					{
						logger.info("User is authenticated to the claims system, will request the user select a site to lookup patient");
						tabArea.displayPatientSiteLookup();
					}
					else
					{
						logger.info("User is authenticated to VistA site, patient lookup will be done against site '" + arg0.getPatientLookupSiteNumber() + "'.");
						tabArea.displayPatientSelectionTab();	
					}
				}
			}
			
			@Override
			public void onFailure(Throwable arg0)
			{
				waitDialog.hide();
				if(!AwivExceptionHandler.handleServiceException(arg0))
				{
					logger.log(Level.SEVERE, "Error retriving user information, " + arg0.getMessage());
					if(arg0 instanceof AwivUserNotLoggedInException)
					{
						authenticateUser(); // try again
					}					
				}
			}
		});
	}
	
	/**
	 * Determines if the user has permission to access the AWIV
	 * @return
	 */
	private boolean isUserAllowedAccess()
	{
		// if this is a claims user then they can view any data
		if(this.awivUserInformation.isClaimsAuthentication())
		{
			return true;			
		}
		return this.awivUserInformation.userHasNecessaryKeys();
	}
	
	private boolean isPatientSpecified()
	{
		String patientIcn = getSpecifiedPatientIcn();
		if(patientIcn != null && patientIcn.length() > 0)
			return true;
		return false;
	}
	
	/**
	 * Find the (optional) specified patient ICN from the URL parameter
	 * @return
	 */
	private String getSpecifiedPatientIcn()
	{
		return Cookies.getCookie(AwivConstants.patientCookieName);
	}
	
	/**
	 * Find the (optional) specified site number from the URL parameter
	 * @return
	 */
	private String getSpecifiedSiteNumber()
	{
		return Window.Location.getParameter("siteNumber");
	}
	
	/**
	 * Retrieve user information given the patient ICN
	 * @param patientIcn
	 */
	private void getPatientInformation(final String patientIcn)
	{
		String transactionId = UUID.uuid();
		String siteNumber = awivUserInformation.getPatientLookupSiteNumber();
		logger.info("Retrieving patient information  for patient '" + patientIcn + "' with transaction Id '" + transactionId + "'.");
		WaitDialog.displayWaitDialog("Loading Patient Information", "Loading patient information");
		PatientLookupServiceDao.patientLookupServiceDao.getPatientInformation(transactionId, siteNumber, patientIcn, new AsyncCallback<Patient>()
		{
			/* (non-Javadoc)
			 * @see com.google.gwt.user.client.rpc.AsyncCallback#onFailure(java.lang.Throwable)
			 */
			@Override
			public void onFailure(Throwable arg0)
			{
				WaitDialog.hideWaitDialog();
				if(!AwivExceptionHandler.handleServiceException(arg0))
				{
					logger.log(Level.SEVERE, "Error retriving patient information, " + arg0.getMessage());
					SC.warn("Error finding information for patient '" + patientIcn + "' from site '" + awivUserInformation.getPatientLookupSiteNumber() + "'.");
				}
			}

			/* (non-Javadoc)
			 * @see com.google.gwt.user.client.rpc.AsyncCallback#onSuccess(java.lang.Object)
			 */
			@Override
			public void onSuccess(Patient patient)
			{
				WaitDialog.hideWaitDialog();
				logger.info("Got information about patient");
				onPatientSelected(patient, false); // only allow viewing this patient				
			}			
		});
	}
	
	private void setUserInformation(AwivUserInformation awivUserInformation)
	{
		
		this.awivUserInformation = awivUserInformation;
		tabArea.setAwivUserInformation(awivUserInformation);
		mastHead.setUserInformation(awivUserInformation);
		AwivHelper.setAwivServerInformation(awivUserInformation.getAwivServerInformation());
		logger.info("Received user information from site '" + this.awivUserInformation.getSiteName() + " [" + this.awivUserInformation.getSiteNumber() + "].");
		if(this.awivUserInformation.getKeys() == null || this.awivUserInformation.getKeys().length <= 0)
		{
			logger.info("User has no security keys");
		}
		else
		{
			for(String key : this.awivUserInformation.getKeys())
			{
				logger.info("User has key [" + key + "]");
			}
		}
	}

	@Override
	public void onPatientSelected(Patient patient, boolean canClosePatient)
	{
		if(AwivHelper.canUserViewPatients(this.awivUserInformation))
		{
			tabArea.displayPatient(patient, canClosePatient);
		}
	}	

	@Override
	public void OnDisplayLog(boolean visible)
	{
		tabArea.displayLog(visible);
	}

	@Override
	public void OnShowPatientSelect()
	{
		tabArea.displayPatientSelectionTab();		
	}

	@Override
	public void onUncaughtException(Throwable arg0)
	{
		String msg = "UncaughtException: " +arg0.getMessage();
		if(arg0.getCause() != null)
			msg += "<br>" + arg0.getCause().toString();
		Window.alert(msg);
		
		//logger.log(Level.SEVERE, msg);
		logger.log(Level.SEVERE, msg, arg0);
	}

	@Override
	public void OnLogoutUser()
	{
		AwivHelper.logout();
	}

	/* (non-Javadoc)
	 * @see com.google.gwt.user.client.Window.ClosingHandler#onWindowClosing(com.google.gwt.user.client.Window.ClosingEvent)
	 */
	@Override
	public void onWindowClosing(ClosingEvent closingEvent)
	{
		try
		{
			// ensure the AWIV window is closed
			AwivViewerManager.closeAwivViewer();
			AwivHelper.clearUserCookie(); // just to be sure it gets removed
		}
		catch(Exception ex)
		{
			// do nothing with the exception since we are closing
		}
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.awiv.client.events.AboutDialogEvent#onShowAboutDialog()
	 */
	@Override
	public void onShowAboutDialog()
	{
		StringBuilder sb = new StringBuilder();
		sb.append("<b>VistA Imaging AWIV</b>");
		sb.append("<br>");
		sb.append("Version: " + awivUserInformation.getAwivServerInformation().getServerVersion());
		
		if(AwivHelper.isDebugMode())
		{		
			sb.append("<br>");
			sb.append("GWT Version: " + GWT.getVersion());
			sb.append("<br>");
			sb.append("SmartGWT Version: " + com.smartgwt.client.Version.getVersion());
		}
		
		SC.say("About", sb.toString());
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.awiv.client.events.UserAuthenticatedEvent#onUserAuthenticated()
	 */
	@Override
	public void onUserAuthenticated()
	{
		authenticateUserDialog.destroy();
		loadUserInformation();
	}
}
