/*
 * $Id: IDCheckFilter.java,v 1.1 2011/03/01 16:24:08 vhahacbergri Exp $
 */
package gov.va.med.hac.edi.ewv2.security;

import java.io.*;
import java.net.URL;
import java.net.URLEncoder;

import javax.servlet.*;
import javax.servlet.http.*;

import org.jboss.seam.annotations.Logger;
import org.jboss.seam.log.Log;

/**
 * IDCheckFilter runs as a filter to intercept web app accesses. If the session
 * does not include a user stored under SEC_USER, then this filter will
 * overwrite the response so that an ASP page will be accessed. That ASP page
 * will figure out the NT username, and post back to the IDCheckServlet, which
 * will put the NT username into the session under SEC_USER
 * 
 * @author vhahacbergri (altered from code by vhahacnguyes)
 */
public final class IDCheckFilter implements Filter
{
    
    @Logger
    Log                        log;
    
    // The name in the session under which the NT user is stored
    public static final String SEC_USER   = "user";
    
    // The name in the session under which the error page is stored
    public static final String ERROR_PAGE = "gov.va.med.hac.sec.ErrorPage";
    
    // Finish of the postBack URL
    private String             postBackSuffix;
    
    // Where the ASP should post back to, ie. the location of the IDCheckServlet
    private String             postBackWithToken;
    
    // The asp page URL
    private String             aspPageUrl;
    
    // Where to go if this doesn't work out
    private String             errorPage;
    
    // I put a debug flag into the web.xml file because I had to log
    // out of this filter using System.out.println. (The seam logger does
    // not seem to be available when this runs, at least the first time or two.)
    public static boolean      doDebugLogging;
    
    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
     */
    public void init(FilterConfig conf) throws ServletException
    {
        // Get configurable entries from web.xml
        aspPageUrl = conf.getInitParameter("aspCheckPage");
        errorPage = conf.getInitParameter("errorPage");
        postBackSuffix = conf.getInitParameter("postBackSuffix");
        doDebugLogging = "true".equals(conf.getInitParameter("filterDebug"));
        
        // Put together the entire postBack string
        // postBackWithToken = postBackPrefix + postBackSuffix;
        
        logDebug(new StringBuffer("Security Filter Parameters [").append("Post Back Suffix:" + postBackSuffix).append(
                "Post Back With Token:" + postBackWithToken).append(", Asp Page:" + aspPageUrl).append(
                ", Error Page:" + errorPage).append("]").toString());
    }
    
    // Write text into the HttpServletResponse that will cause a redirect to
    // an ASP page
    private void overwriteResponse(HttpServletRequest hreq, HttpServletResponse hres, String sId, String capturedUrlString)
            throws IOException
    {
        logDebug("In postAsp with capturedUrlString: " + capturedUrlString);
        
        // Put together the host and port from the request. These are used to
        // replace the token in the postBack string
        String hostOnly = hreq.getLocalName();
        int port = hreq.getLocalPort();
        String hostAndPort = new StringBuffer(hostOnly).append(':').append(port).toString();
        logDebug("In postAsp, hostAndPort; " + hostAndPort);
        
        // Pick apart the incoming URL
        URL capturedURL = new URL(capturedUrlString);
        logDebug("Captured URL path: " + capturedURL.getPath());
        String capturedProtocol = capturedURL.getProtocol();
        logDebug("Captured URL protocol: " + capturedProtocol);
        String capturedHost = capturedURL.getHost();
        logDebug("Captured URL host: " + capturedHost);
        int capturedPort = capturedURL.getPort();
        logDebug("Captured URL port: " + capturedPort);
        
        String postBackUrl = "";
        String encodedOriginalUrl = "";
		try {
			// The setting of postBack differs depending on whether we are
	        // connecting
	        // to localhost or not, so we calculate it here
	        // String postBackUrl = MessageFormat.format(postBackWithToken,
	        // hostAndPort);
			postBackUrl = IDCheckAESEncryption.encrypt(capturedProtocol + ':' + '/' + '/' + hostAndPort + '/' + postBackSuffix);
			logDebug("encodedPostBackUrl: " + postBackUrl);
			// The capturedUrl needs its host replaced with the fully qualified host
			encodedOriginalUrl = IDCheckAESEncryption.encrypt(capturedUrlString.replace(capturedHost, hostOnly));
			logDebug("In postAsp, encodedOriginalUrl after replacement: " + encodedOriginalUrl);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
        
        
        // Overwrite the response with instructions to redirect to the ASP page
        PrintWriter w = hres.getWriter();
        w.println("<html><body onload=\"document.getElementById('postA').submit();\">");
        w.println("Please wait...  Reload the application if your browser is not redirected.");
        w.println("<form id=\"postA\" method=\"POST\" action=\"" + aspPageUrl + "\">");
        w.println("<input type=\"hidden\" name=\"s\" value=\"" + sId + "\">");
        w.println("<input type=\"hidden\" name=\"c\" value=\"" + URLEncoder.encode(encodedOriginalUrl, "UTF-8") + "\">");
        w.println("<input type=\"hidden\" name=\"u\" value=\"" + URLEncoder.encode(postBackUrl, "UTF-8") + "\">");
        w.println("</form></body>");
        w.println("</html>");
        
    }
    
    // Required to implement interface Filter
    public void destroy()
    {
    }
    
    // Intercept all web accesses. If the HTTP Session does not contain a user
    // under SEC_USER, redirect to an ASP page by overwriting the HTTP response
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException,
            ServletException
    {
        
        // Get the HTTP session, and see whether there is a user stored there
        // yet.
        HttpServletRequest hreq = (HttpServletRequest) request;
        HttpSession session = hreq.getSession();
        logDebug("In doFilter with session: " + session.getId());
        String userBean = (String) session.getAttribute(SEC_USER);
        
        // If there is a UserBean in session, user has been to the ASP page,
        // and we can shortcircuit this filter
        if (isNull(userBean))
        {
            logInfo("No user in session");
            
            // Get the URL of the request
            String url = hreq.getRequestURL().toString();
            logDebug("In doFilter, url: " + url);
            
            // Skip certain url's so that we don't get into an infinite loop.
            // Especially, skip if the URL is the servlet called by the ASP page
            boolean skipThisUrl = isSkip(url);
            
            // If we are not skipping this URL, we end up going to an ASP page
            if (!skipThisUrl)
            {
                logDebug("Not skipping this url: " + url);
                
                // This session id probably is not necessary, but the ASP page
                // expects it.
                String id = session.getId() + session.hashCode();
                
                // If we don't pass around the queryString along with the url,
                // then
                // bookmarking of pages that use parameters will not work
                String queryString = hreq.getQueryString();
                String urlWithQuery = "" + url;
                if (queryString != null)
                {
                    urlWithQuery += "?" + queryString;
                }
                
                // Overwrite the Response so that control will pass to an ASP
                // page.
                // That page will redirect to the IDCheckServlet
                overwriteResponse(hreq, (HttpServletResponse) response, id, urlWithQuery);
                
                logDebug("In dofilter, going to ASP page");
                
                // Don't go through the remainder of the filter chain. This
                // prevents
                // the request from getting to the server. Instead, the
                // response,
                // which was filled out by overwriteResponse, is used. That
                // response is
                // a redirect to the ASP page.
                return;
            }
            else
            {
                logDebug("In IDCheckFilter.dofilter, not going to Asp" + " because URL can be skipped");
            }
        }
        else
        {
            logDebug("In IDCheckFilter.dofilter, not going to Asp" + " because a session user exists");
        }
        request.setAttribute(ERROR_PAGE, errorPage);
        
        // Carry on with the other filters as normal
        chain.doFilter(request, response);
    }
    
    /**
     * Determine whether the given URL should be skipped by this filter, that
     * is, should let the filter be short-circuited so that it doesn't go to the
     * ASP page
     * 
     * @param url
     *            the url
     * 
     * @return true, if checks if is skip
     */
    private boolean isSkip(String url)
    {
        logDebug("In isSkip, url: " + url);
        String uppercaseUrl = url.toUpperCase();
        // Skip under one of these circumstances:
        // The url is the IDCheckServlet that the ASP page has forwarded to
        // The url is the error page
        // The url is a style sheet
        // The url is an image
        boolean doSkip = url.endsWith(postBackSuffix) || contains(uppercaseUrl, errorPage.toUpperCase())
                || uppercaseUrl.endsWith(".CSS") || uppercaseUrl.endsWith(".GIF") ? true : false;
        logDebug("Returning " + doSkip + " from isSkip");
        return doSkip;
    }
    
    /**
     * Is the given string null or blank?
     * 
     * @param s
     *            the string
     * 
     * @return true, if the given string is null or blank
     */
    private boolean isNull(String s)
    {
        return s == null || s.length() == 0 ? true : false;
    }
    
    /**
     * Does the first string contain the second string?
     * 
     * @param s1
     *            the string 1
     * @param s2
     *            the string 2
     * 
     * @return true, if successful
     */
    private boolean contains(String s1, String s2)
    {
        return s1.indexOf(s2) > -1 ? true : false;
    }
    
    private void logDebug(String s)
    {
        if (doDebugLogging)
        {
            System.out.println("IDCheckFilter:" + s);
        }
    }
    
    private void logInfo(String s)
    {
        if (log != null)
        {
            log.info(s);
        }
        else
        {
            System.out.println("Nolog: " + s);
        }
    }
}
