package gov.va.med.hac.edi.ewv2.actions;

import gov.va.med.hac.edi.ewv2.beans.claim.CobAmountsBean;
import gov.va.med.hac.edi.ewv2.beans.codes.RemittanceCode;
import gov.va.med.hac.edi.ewv2.beans.ohi.*;
import gov.va.med.hac.edi.ewv2.beans.ohi.claimlevel.*;
import gov.va.med.hac.edi.ewv2.beans.ohi.linelevel.*;
import gov.va.med.hac.edi.ewv2.persistence.*;
import gov.va.med.hac.edi.ewv2.persistence.base.*;
import gov.va.med.hac.edi.ewv2.utils.*;
import gov.va.med.hac.edi.ewv2.exceptions.*;
import java.util.*;

import org.jboss.seam.log.Log;

public class OhiDataBinder extends DTOBase
{
    private String primaryPayerID = null;
    private ClaimType claimType = null;
    
    public OhiInfoContainer getOhiData(EwvClaim claimBean, Log log)
    throws NonUniqueRecordsException
    {
        // Get the necessary pieces from the database
        Set<EwvServiceLine> serviceLineSet = claimBean.getEwvServiceLines();
        Set<EwvClaimOhi> claimOhiSet = claimBean.getEwvClaimOhis();
        claimType = ClaimType.typeOf(claimBean.getClaimType());
        
        // Loop through the claim ohis, adding a data structure for each one to a list
        List<ClaimOhiContainer> claimOhiList = new ArrayList<ClaimOhiContainer>(0);
        for (EwvClaimOhi nextClaimOhi : claimOhiSet)
        {
            Set<EwvClaimLevelAdjustment> claimOhiAdjustments = nextClaimOhi.getEwvClaimLevelAdjustments();
            ClaimOhiContainer newClaimOhiContainer = getClaimOhi(nextClaimOhi, claimOhiAdjustments, log);
            claimOhiList.add(newClaimOhiContainer);
        }
        Collections.sort(claimOhiList);
        
        // Determine whether there are any problems with payer identification
        // Three different problems might exist that cause a change in our Quickview layout
        boolean hasMultiplePrimaryPayers = getHasMultiplePrimaryPayers(claimBean, log);
        boolean hasMultipleSecondaryPayers = getHasMultipleSecondaryPayers(claimBean, log);
        boolean hasMultiplePayersWithSamePayerId = getHasMultiplePayersWithSamePayerId(claimBean, log);
        boolean hasPayerIdProblem = false;
        if (hasMultiplePrimaryPayers || hasMultipleSecondaryPayers  ||
                     hasMultiplePayersWithSamePayerId) {
          hasPayerIdProblem = true;
        }
        
        // A fourth type of payer id problem is not as serious.
        // Is this flag even necessary?
        boolean hasMultipleAdjudicationsFromSamePayer = getHasMultipleAdjudicationsFromSamePayer(claimBean, log);

        String primaryClaimOhiPayerId = null;
        String secondaryClaimOhiPayerId = null;
        ClaimOhi primaryClaimOhi = null;
        ClaimOhi secondaryClaimOhi = null;
        Boolean tertiaryOhiExists = false;
        List<QuickViewServiceLineOhiOutputRow> quickviewServiceLineOhiOutputRows = null;
        
        // Only do some of this stuff if there is no payer id problem
        if (!hasPayerIdProblem) {
          // Put together the primary and secondary Claim Ohi records, for use in subtables
          primaryClaimOhi = populateClaimOhi(claimOhiSet, serviceLineSet, "P");
          secondaryClaimOhi = populateClaimOhi(claimOhiSet, serviceLineSet, "S");
          tertiaryOhiExists = getTertiaryOhiExists(claimOhiSet);
          
          // Phase Two Stuff
          // Put together a data structure to hold service lines plus ohis plus adjustment info
          // This is used for the table near the bottom of the QuickView tab, which would be
          //   difficult to do as three separate tables because of the line breaks due to
          //   multiple primary or secondary adjustments.
          if (primaryClaimOhi != null) {
            primaryClaimOhiPayerId = primaryClaimOhi.getOhiPayerId();
          }
          if (secondaryClaimOhi != null) {
            secondaryClaimOhiPayerId = secondaryClaimOhi.getOhiPayerId();
          }
  
          quickviewServiceLineOhiOutputRows
            = populateQuickviewServiceLineOhiOutputRows(serviceLineSet,
                                            primaryClaimOhiPayerId,
                                            secondaryClaimOhiPayerId,
                                            log);
        }
        
        List<DetailServiceLineOhiOutputRow> detailServiceLineOhiOutputRows
          = populateDetailServiceLineOutputRows(serviceLineSet,
                                                hasMultiplePayersWithSamePayerId,
                                                claimOhiSet,
                                                log);
        
        // End of Phase Two Stuff

        OhiInfoContainer ohiInfoContainer
          = new OhiInfoContainer(claimOhiList,
  		                          primaryClaimOhi,
  		                          secondaryClaimOhi,
  		                          tertiaryOhiExists,
  		                          quickviewServiceLineOhiOutputRows,
  		                          detailServiceLineOhiOutputRows,
  		                          hasPayerIdProblem,
  		                          hasMultiplePrimaryPayers,
  		                          hasMultipleSecondaryPayers,
  		                          hasMultiplePayersWithSamePayerId,
  		                          hasMultipleAdjudicationsFromSamePayer);
        
        return ohiInfoContainer;
    }
    
    private ClaimOhiContainer getClaimOhi (EwvClaimOhi claimOhi,
                                           Set<EwvClaimLevelAdjustment> adjustmentSet,
                                           Log log)
    {
        MedicareInPatientAdjudicationBean medicareInPatientAdjudicationBean
          = createMedicareInPatientAdjudicationBean(claimOhi, log);
        MedicareOutPatientAdjudicationBean medicareOutPatientAdjudicationBean
          = createMedicareOutPatientAdjudicationBean(claimOhi);
        
        CobAmountsBean claimOhiAmountsBean = createCOBAmountsBean(claimOhi);
        
        ClaimOhiHeader claimOhiHeader = createClaimOhiHeaderBean(claimOhi);
        
        // Adjustments
        ClaimOhiAdjustmentsBean claimOhiAdjustmentsBean
          = createClaimOhiAdjustmentsBean(claimOhi.getOhiPayerId(), adjustmentSet, log);
         
        ClaimOhiContainer claimOhiContainer
          = new ClaimOhiContainer(claimOhiHeader,
                                  claimOhiAdjustmentsBean,
        		                  claimOhiAmountsBean,
        		                  medicareInPatientAdjudicationBean,
        		                  medicareOutPatientAdjudicationBean);

        
        return claimOhiContainer;
    }  
    
    private MedicareOutPatientAdjudicationBean createMedicareOutPatientAdjudicationBean(EwvClaimOhi claim)
    {
        String ohiPayerId = claim.getOhiPayerId();
        List<RemittanceCode> remittanceRemarkCodes = createOutpatientRemittanceRemarkCodes(claim);
        //        String reimbursementRate = ((claim.getOptReimbursementRate() != null) ? (claim.getOptReimbursementRate()+"%") : " ");
        // Phase Two Change
        //String reimbursementRate = ((claim.getOptReimbursementRate() != null) ? (claim.getOptReimbursementRate()) : " ");
        String reimbursementRate = claim.getOptReimbursementRate();
       
        String claimHcpcsPayableAmount = claim.getOptClaimHcpcsPayableAmt();
        String esrdPaidAmount = claim.getOptEsrdPaidAmt();
        String nonPayableProfessionalComponentAmount = claim.getOptNonpayableProfComponent();
        
        MedicareOutPatientAdjudicationBean ret = new MedicareOutPatientAdjudicationBean(ohiPayerId, remittanceRemarkCodes, reimbursementRate, claimHcpcsPayableAmount, esrdPaidAmount, nonPayableProfessionalComponentAmount);
        
        boolean ok = Ewv2Utils.okToRender(claim.getOptRemittanceRemarkCode1(),claim.getOptRemittanceRemarkCode2(),claim.getOptRemittanceRemarkCode3(),claim.getOptRemittanceRemarkCode4(),claim.getOptRemittanceRemarkCode5(),reimbursementRate, claimHcpcsPayableAmount, esrdPaidAmount, nonPayableProfessionalComponentAmount);
        
        ret.setShouldRender(ok);
        
        return ret;
    }
    
    private List<RemittanceCode> createOutpatientRemittanceRemarkCodes(EwvClaimOhi claim)
    {
        List<RemittanceCode> listOf = new ArrayList<RemittanceCode>(0);
        
        if(claim.getOptRemittanceRemarkCode1() != null) {
           listOf.add(new RemittanceCode(claim.getOptRemittanceRemarkCode1(),claim.getOptRemittanceRemarkDesc1(),Ewv2Utils.shorten(claim.getOptRemittanceRemarkDesc1(), 60)));
        }
        if(claim.getOptRemittanceRemarkCode2() != null) {
           listOf.add(new RemittanceCode(claim.getOptRemittanceRemarkCode2(),claim.getOptRemittanceRemarkDesc2(),Ewv2Utils.shorten(claim.getOptRemittanceRemarkDesc2(), 60)));
        }
        if(claim.getOptRemittanceRemarkCode3() != null) {
           listOf.add(new RemittanceCode(claim.getOptRemittanceRemarkCode3(),claim.getOptRemittanceRemarkDesc3(),Ewv2Utils.shorten(claim.getOptRemittanceRemarkDesc3(), 60))); 
        }
        if(claim.getOptRemittanceRemarkCode4() != null) {
           listOf.add(new RemittanceCode(claim.getOptRemittanceRemarkCode4(),claim.getOptRemittanceRemarkDesc4(),Ewv2Utils.shorten(claim.getOptRemittanceRemarkDesc4(), 60)));
        }
        if(claim.getOptRemittanceRemarkCode5() != null) {
           listOf.add(new RemittanceCode(claim.getOptRemittanceRemarkCode5(),claim.getOptRemittanceRemarkDesc5(),Ewv2Utils.shorten(claim.getOptRemittanceRemarkDesc5(), 60)));
        }
        
        return listOf;
    }
    
    private MedicareInPatientAdjudicationBean createMedicareInPatientAdjudicationBean
        (EwvClaimOhi claim, Log log)
    {
        String ohiPayerId = claim.getOhiPayerId();
        String coveredDays = claim.getInpCoveredDays();
        //log.info("Covered days: " + coveredDays);
        
        // Defect #UAT-20 fix
        String ppsOperatingFederalSpecDrugAmount = claim.getInpPpsOperFedSpecDrgAmt();
        String lifetimeReserveDays = claim.getInpLifetimeReserveDays();
        String ppsCapitalOutlierAmount = claim.getInpPpsCapitalOutlrAmt();
        String lifetimePsychiatricDays = claim.getInpLifetimePsychiatricDays();
        String claimIndirectTeachingAmount = claim.getInpClaimIndirectTeachAmt();
        String claimDrugAmount = claim.getInpClaimDrgAmt();
        String nonPayableProfessionalComponent = claim.getInpNonpayableProfComponent();
        String claimDisproportionateShare = claim.getInpClaimDisproporShare();
        String ppsCapitalExceptionAmount = claim.getInpPpsCapitalExceptionAmt();
        String mspPassThruAmount = claim.getInpMspPassThroughAmt();
        String costReportDayCount = claim.getInpCostReportDayCount();
        String claimPpsCapitalAmount = claim.getInpClaimPpsCapitalAmt();
        String ppsCapitalFspDrugAmount = claim.getInpPpsCapitalFspDrgAmt();
        String ppsCapitalHspDrugAmount = claim.getInpPpsCapitalHspDrgAmt();
        String ppsCapitalDshDrugAmount = claim.getInpPpsCapitalDshDrgAmt();
        String oldCapitalAmount = claim.getInpOldCapitalAmt();
        List<RemittanceCode> remittanceRemarkCodes = createInPatientRemittanceRemarkCodes(claim);
        String ppsCapitalImeAmount = claim.getInpPpsCapitalImeAmt();
        String ppsOperHospitalSpecicalDrugAmount = claim.getInpPpsOperHospSpecDrgAmt();
        
        MedicareInPatientAdjudicationBean ret
          = new MedicareInPatientAdjudicationBean(ohiPayerId,
                                                  coveredDays, 
                                                  ppsOperatingFederalSpecDrugAmount, 
                                                  lifetimeReserveDays,
                                                  ppsCapitalOutlierAmount,
                                                  lifetimePsychiatricDays,
                                                  claimIndirectTeachingAmount,
                                                  claimDrugAmount, 
                                                  nonPayableProfessionalComponent, 
                                                  claimDisproportionateShare, 
                                                  ppsCapitalExceptionAmount,
                                                  mspPassThruAmount,
                                                  costReportDayCount,
                                                  claimPpsCapitalAmount,
                                                  ppsCapitalFspDrugAmount,
                                                  ppsCapitalHspDrugAmount,
                                                  ppsCapitalDshDrugAmount,
                                                  oldCapitalAmount,
                                                  remittanceRemarkCodes,
                                                  ppsCapitalImeAmount,
                                                  ppsOperHospitalSpecicalDrugAmount);
        
        boolean ok = Ewv2Utils.okToRender(claim.getInpRemittanceRemarkCode1(),
                                          claim.getInpRemittanceRemarkCode2(),
                                          claim.getInpRemittanceRemarkCode3(),
                                          claim.getInpRemittanceRemarkCode4(),
                                          claim.getInpRemittanceRemarkCode5(),
                                          coveredDays,
                                          ppsOperatingFederalSpecDrugAmount,
                                          lifetimeReserveDays, 
                                          ppsCapitalOutlierAmount, 
                                          lifetimePsychiatricDays,
                                          claimIndirectTeachingAmount,
                                          claimDrugAmount,
                                          nonPayableProfessionalComponent,
                                          claimDisproportionateShare,
                                          ppsCapitalExceptionAmount,
                                          mspPassThruAmount, 
                                          costReportDayCount,
                                          claimPpsCapitalAmount,
                                          ppsCapitalFspDrugAmount,
                                          ppsCapitalHspDrugAmount, 
                                          ppsCapitalDshDrugAmount,
                                          oldCapitalAmount, 
                                          ppsCapitalImeAmount, 
                                          ppsOperHospitalSpecicalDrugAmount);

        //log.info("Should render Medicare Inpatient: " + ok);

        ret.setShouldRender(ok);
        
        return ret;
    }
    
    private List<RemittanceCode> createInPatientRemittanceRemarkCodes(EwvClaimOhi claim)
    {
        List<RemittanceCode> listOf = new ArrayList<RemittanceCode>(0);
        
        if(claim.getInpRemittanceRemarkCode1() != null)
        {
           listOf.add(new RemittanceCode(claim.getInpRemittanceRemarkCode1(),claim.getInpRemittanceRemarkDesc1(),Ewv2Utils.shorten(claim.getInpRemittanceRemarkDesc1(), 60)));
        }
        
        if(claim.getInpRemittanceRemarkCode2() != null)
        {
           listOf.add(new RemittanceCode(claim.getInpRemittanceRemarkCode2(),claim.getInpRemittanceRemarkDesc2(),Ewv2Utils.shorten(claim.getInpRemittanceRemarkDesc2(), 60)));
        }
        
        if(claim.getInpRemittanceRemarkCode3() != null)
        {
           listOf.add(new RemittanceCode(claim.getInpRemittanceRemarkCode3(),claim.getInpRemittanceRemarkDesc3(),Ewv2Utils.shorten(claim.getInpRemittanceRemarkDesc3(), 60))); 
        }
        
        if(claim.getInpRemittanceRemarkCode4() != null)
        {
           listOf.add(new RemittanceCode(claim.getInpRemittanceRemarkCode4(),claim.getInpRemittanceRemarkDesc4(),Ewv2Utils.shorten(claim.getInpRemittanceRemarkDesc4(), 60)));
        }
        
        if(claim.getInpRemittanceRemarkCode5() != null)
        {
           listOf.add(new RemittanceCode(claim.getInpRemittanceRemarkCode5(),claim.getInpRemittanceRemarkDesc5(),Ewv2Utils.shorten(claim.getInpRemittanceRemarkDesc5(), 60)));
        }
        
        return listOf;
    }
    
    private CobAmountsBean createCOBAmountsBean(EwvClaimOhi claim)
    {
        String ohiPayerId = claim.getOhiPayerId();
        String cobSubmittedCharges = claim.getCobSubmittedCharges();
        String cobTotalAllowedAmount = claim.getCobTotalAllowedAmt();
        String cobApprovedAmount = claim.getCobApprovedAmt();
        String cobTotalNonCoveredAmount = claim.getCobTotalNoncoveredAmt();
        String cobPayerPaidAmount = claim.getCobPayerPaidAmt();
        String cobTotalDeniedAmount = claim.getCobTotalDeniedAmt();
        String cobPatientPaidAmount = claim.getCobPatientPaidAmt();
        String cobTotalMedicarePaidAmount = claim.getCobTotalMedicarePaidAmt();
        String cobPatientResponsibility = claim.getCobPatientResponsibility();
        String medicarePaidAmt = claim.getCobTotalMedicarePaidAmt();
        String cobMedicareATrustFund = claim.getCobMedicareATrustFund();
        String cobMedicareBTrustFund = claim.getCobMedicareBTrustFund();
        
        String medicarePaidEightyPercentAmount = claim.getCobMedicarePaidAmt80();
        String medicarePaidOneHundredPercentAmount = claim.getCobMedicarePaidAmt100();
        
        CobAmountsBean ret = new CobAmountsBean(ohiPayerId, 
                                                cobSubmittedCharges, 
                                                cobTotalAllowedAmount, 
                                                cobApprovedAmount, 
                                                cobTotalNonCoveredAmount, 
                                                cobPayerPaidAmount, 
                                                cobTotalDeniedAmount, 
                                                cobPatientPaidAmount, 
                                                cobTotalMedicarePaidAmount, 
                                                medicarePaidOneHundredPercentAmount, 
                                                medicarePaidEightyPercentAmount, 
                                                cobPatientResponsibility, 
                                                medicarePaidAmt, 
                                                cobMedicareATrustFund, 
                                                cobMedicareBTrustFund);
        
        return ret;
    }
    
    private ClaimOhiAdjustmentsBean createClaimOhiAdjustmentsBean
      (String theOhiPayerId, Set<EwvClaimLevelAdjustment> adjustmentSet, Log log)
    {
        String ohiPayerId = theOhiPayerId;
        
        List<Adjustment> adjustmentList = new ArrayList<Adjustment>(); 
        
        for(EwvClaimLevelAdjustment adjust : adjustmentSet)
        {
          adjustmentList.add(createClaimOhiAdjustment(ohiPayerId,adjust));
        }
        
        // sort by amounts, OA, PI, PR etc...
        Collections.sort(adjustmentList);

        return new ClaimOhiAdjustmentsBean(ohiPayerId, adjustmentList);
    }
    
    private ClaimOhiAdjustment createClaimOhiAdjustment(String ohiPayerId,
    		                                                EwvClaimLevelAdjustment adjust)
    {
        String claimAdjustmentReasonCode = adjust.getClaimAdjustmentReasonCode();
        String quantity = adjust.getQuantity();
        String descriptionLong = adjust.getDescription();
        String descriptionShort = Ewv2Utils.shorten(descriptionLong, 15);
        
        String coAmount = adjust.getCoAmt();
        String crAmount = adjust.getCrAmt();
        String oaAmount = adjust.getOaAmt();
        String piAmount = adjust.getPiAmt();
        String prAmount = adjust.getPrAmt();
        
        ClaimOhiAdjustment ret = new ClaimOhiAdjustment(ohiPayerId,
                                                        claimAdjustmentReasonCode, 
                                                        descriptionLong, 
                                                        descriptionShort, 
                                                        quantity, 
                                                        coAmount,
                                                        crAmount,
                                                        oaAmount,
                                                        piAmount, 
                                                        prAmount);
    
        if (coAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.CO);
        }
        else if (crAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.CR);
        }
        else if (oaAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.OA);
        }
        else if (piAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.PI);
        }
        else if (prAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.PR);
        }
    
        return ret;
    }
    
    private ClaimOhiHeader createClaimOhiHeaderBean(EwvClaimOhi claim)
    {
        String paymentSequenceIndicator = claim.getPaymentSequenceIndicator();
        String ohiGroupName = claim.getOhiGroupName();
        String filingIndicator = claim.getFilingIndicator();
        String benefitsAssignmentIndicator = claim.getBenefitsAssignmentIndicator();
        String insuranceType = claim.getInsuranceType();
        String claimAdjustmentDate = claim.getClaimAdjustmentDate();
        String ohiPayerID = claim.getOhiPayerId();
        String otherPayerName = claim.getOtherPayerName();
        
        // TODO where is this coming from?
        String ohiOne = "XXX";
        
        ClaimOhiHeader ret
          = new ClaimOhiHeader(paymentSequenceIndicator,
                                             ohiGroupName,
                                             filingIndicator,
                                             benefitsAssignmentIndicator,
                                             insuranceType,
                                             claimAdjustmentDate,
                                             ohiPayerID,
                                             otherPayerName,
                                             ohiOne);
        
        boolean ok = Ewv2Utils.okToRender(paymentSequenceIndicator, ohiGroupName, filingIndicator, benefitsAssignmentIndicator, insuranceType, claimAdjustmentDate, ohiPayerID, otherPayerName);
        
        ret.setShouldRender(ok);
        
        return ret;
    }
    
    private ClaimOhi populateClaimOhi(Set<EwvClaimOhi> claimOhis,
  		                               Set<EwvServiceLine> serviceLines,
  		                               String ohiType)
    {
      ClaimOhi returnBean = null;
      
      // Find the pertinent ohi
      EwvClaimOhi sourceOhi = null;
      for (EwvClaimOhi claimOhi : claimOhis)
      {
        if (ohiType.equals(claimOhi.getPaymentSequenceIndicator())) {
          sourceOhi = claimOhi;
          break;
        }
      }
      
      // If after looking for the pertinent ohi, we did not find one, give up
      if (sourceOhi == null) {
    	return null;
      }
      
      // Start up an object to return
      returnBean = new ClaimOhi();
      
      // Get the string fields
      returnBean.setOtherPayerName(sourceOhi.getOtherPayerName());
      returnBean.setOhiGroupName(sourceOhi.getOhiGroupName());
      returnBean.setOhiPayerId(sourceOhi.getOhiPayerId());
      returnBean.setPayerPaidAmount(Ewv2Utils.formatAsCurrency(sourceOhi.getCobPayerPaidAmt()));
      
      // Get the Claim OHI adjustments
      Set<EwvClaimLevelAdjustment> sourceClaimOhiAdjustments
        = sourceOhi.getEwvClaimLevelAdjustments();
      List<ClaimOhiAdjustment> returnClaimOhiAdjustments = new ArrayList<ClaimOhiAdjustment>();
      for (EwvClaimLevelAdjustment sourceClaimOhiAdjustment : sourceClaimOhiAdjustments)
      {
    	ClaimOhiAdjustment claimOhiAdjustment
    	  = createClaimOhiAdjustment(sourceOhi.getOhiPayerId(),sourceClaimOhiAdjustment);
    	returnClaimOhiAdjustments.add(claimOhiAdjustment);
      }
      returnBean.setClaimOhiAdjustments(returnClaimOhiAdjustments);
      
      // Get the remittance remark code stuff
      List<RemittanceCode> remittanceCodes = getRemittanceCodesForOhi(sourceOhi);
      returnBean.setRemittanceRemarkCodes(remittanceCodes);
      
      // Get the service line stuff
      
      
      return returnBean;
    }
    
    private boolean getTertiaryOhiExists(Set<EwvClaimOhi> claimOhis)
    {
      boolean returnVal = false;

	  // Find the pertinent ohi
	  for (EwvClaimOhi claimOhi : claimOhis)
	  {
	    if ("T".equals(claimOhi.getPaymentSequenceIndicator())) {
	      returnVal = true;
          break;
	    }
	  }
      return returnVal;
    }

    private List<RemittanceCode> getRemittanceCodesForOhi(EwvClaimOhi ohi) {
      List<RemittanceCode> remittanceCodes = new ArrayList<RemittanceCode>();
      RemittanceCode nextRemittanceCode = null;
    	
      // Look through both inpatient and outpatient, for now
      // For this purpose, we do not need short names
      if (ohi.getInpRemittanceRemarkCode1() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getInpRemittanceRemarkCode1(),
    		   ohi.getInpRemittanceRemarkDesc1(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      if (ohi.getInpRemittanceRemarkCode2() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getInpRemittanceRemarkCode2(),
    		   ohi.getInpRemittanceRemarkDesc2(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }

      if (ohi.getInpRemittanceRemarkCode3() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getInpRemittanceRemarkCode3(),
    		   ohi.getInpRemittanceRemarkDesc3(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      if (ohi.getInpRemittanceRemarkCode4() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getInpRemittanceRemarkCode4(),
    		   ohi.getInpRemittanceRemarkDesc4(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      if (ohi.getInpRemittanceRemarkCode5() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getInpRemittanceRemarkCode5(),
    		   ohi.getInpRemittanceRemarkDesc5(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      if (ohi.getOptRemittanceRemarkCode1() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getOptRemittanceRemarkCode1(),
    		   ohi.getOptRemittanceRemarkDesc1(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      if (ohi.getOptRemittanceRemarkCode2() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getOptRemittanceRemarkCode2(),
    		   ohi.getOptRemittanceRemarkDesc2(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      if (ohi.getOptRemittanceRemarkCode3() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getOptRemittanceRemarkCode3(),
    		   ohi.getOptRemittanceRemarkDesc3(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      if (ohi.getOptRemittanceRemarkCode4() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getOptRemittanceRemarkCode4(),
    		   ohi.getOptRemittanceRemarkDesc4(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      if (ohi.getOptRemittanceRemarkCode5() != null) {
    	  nextRemittanceCode = new RemittanceCode
    	      (ohi.getOptRemittanceRemarkCode5(),
    		   ohi.getOptRemittanceRemarkDesc5(),
    		   null);
    	  remittanceCodes.add(nextRemittanceCode);
      }
      
      return remittanceCodes;
    }

    private List<DetailServiceLineOhiOutputRow> populateDetailServiceLineOutputRows
            (Set<EwvServiceLine> serviceLines,
             boolean hasMultiplePayersWithSamePayerId,
             Set<EwvClaimOhi> claimOhiSet,
             Log log)
    {
      List<DetailServiceLineOhiOutputRow> returnList = new ArrayList<DetailServiceLineOhiOutputRow>();

      List<EwvServiceLine> serviceLineList = new ArrayList<EwvServiceLine>(serviceLines);

      Collections.sort(serviceLineList);
      
      // Go through the service lines.  
      for (EwvServiceLine nextServiceLine : serviceLineList) {
        
        // Go through the ohi records for this service line
        Set<EwvServiceLineOhi> serviceLineOhiSet = nextServiceLine.getEwvServiceLineOhis();

        for (EwvServiceLineOhi serviceLineOhi : serviceLineOhiSet) {
          // Create an output row.  (Each service line ohi will get at least one output row.)
          DetailServiceLineOhiOutputRow newOutputRow = new DetailServiceLineOhiOutputRow();
          
          // Put the service-line-level info into the output row
          newOutputRow.setLineNumber(Ewv2Utils.formatAsInteger(nextServiceLine.getLineNumber()));
          newOutputRow.setProcedureCode(nextServiceLine.getProcedureCode());
          newOutputRow.setLineCharge(Ewv2Utils.formatAsCurrency(nextServiceLine.getCharge()));
          newOutputRow.setBeginDateOfService(Ewv2Utils.formatAsDate(nextServiceLine.getBeginDateOfService()));

          // Put the ohi-level info into the output row
          newOutputRow.setAdjudicationDate(Ewv2Utils.formatAsDate(serviceLineOhi.getOhiAdjudicatedDate()));
          newOutputRow.setBundledUnbundledLineNumber(Ewv2Utils.formatAsInteger(serviceLineOhi.getBundledUnbundledLineNumber()));
          newOutputRow.setRevenueCode(serviceLineOhi.getRevenueCode());
          newOutputRow.setModifiers(Ewv2Utils.combineModifiers(
             ", ",
             serviceLineOhi.getModifier1(),
             serviceLineOhi.getModifier2(),
             serviceLineOhi.getModifier3(),
             serviceLineOhi.getModifier4()));
          newOutputRow.setPayerPaidAmount(Ewv2Utils.formatAsCurrency(serviceLineOhi.getOhiPaid()));
          
          // The payerId used to need special attention
          //String payerId = serviceLineOhi.getPayerId();
          //log.info("Going to determinePayerId from Detail");
          String payerId = determinePayerId(serviceLineOhi,log);
          //log.info("payerId for service line ohi: " + payerId);
          newOutputRow.setPayerId(payerId);
          
          // The payment sequence indicator still needs special attention
          String paymentSequenceIndicator = "?";
          if (payerId != null && !hasMultiplePayersWithSamePayerId) {
            paymentSequenceIndicator = determinePaymentSequenceIndicator(payerId, claimOhiSet, log);
            //log.info("Got payment sequence indicator: " + paymentSequenceIndicator);
          }
          newOutputRow.setPaymentSequenceIndicator(paymentSequenceIndicator);
          
          // Get the adjustments for this ohi
          Set<EwvServiceLineOhiAdjtmt> adjustmentsSet = serviceLineOhi.getEwvServiceLineOhiAdjtmts();
          List<EwvServiceLineOhiAdjtmt> adjustmentsList = new ArrayList<EwvServiceLineOhiAdjtmt> (adjustmentsSet);
          
          // Put the info from the first adjustment into the output row
          if (adjustmentsList.size() > 0) {
            EwvServiceLineOhiAdjtmt firstAdjustmentDb = adjustmentsList.get(0);
            ServiceLineOhiAdjustment firstAdjustment
               = createServiceLineOhiAdjustment(firstAdjustmentDb);
            
            newOutputRow.setAdjustment(firstAdjustment);
          }
          // Finish with this output row 
          returnList.add(newOutputRow);
          
          // Loop through any remaining adjustments, putting together more output rows
          // Keep adding output rows until there are no more adjustments of either type
          int adjustmentListSize = adjustmentsList.size();
          for (int rowCounter = 1; rowCounter < adjustmentListSize; rowCounter++) {
        
            // Create an output row.  
            DetailServiceLineOhiOutputRow nextOutputRow = new DetailServiceLineOhiOutputRow();
            
            // Put the adjustment info into the output row
            if (adjustmentListSize >= rowCounter + 1) {
              EwvServiceLineOhiAdjtmt nextAdjustmentDb = adjustmentsList.get(rowCounter);
              ServiceLineOhiAdjustment nextAdjustment
                  = createServiceLineOhiAdjustment(nextAdjustmentDb);
              nextOutputRow.setAdjustment(nextAdjustment);
            }
            
            // Finish with this output row 
            returnList.add(nextOutputRow);
          }
        }
      }
      
      return returnList;
    }
    
    private List<QuickViewServiceLineOhiOutputRow> populateQuickviewServiceLineOhiOutputRows
        (Set<EwvServiceLine> serviceLines,
         String primaryPayerId, 
         String secondaryPayerId,
         Log log)
    {
      //log.info("In populateQuickviewServiceLineOhiOutputRows with primary: " + primaryPayerId);
      
      // Return value
      List<QuickViewServiceLineOhiOutputRow> returnList = new ArrayList<QuickViewServiceLineOhiOutputRow>();
      
      // Local variables - service line level
      List<EwvServiceLine> serviceLineList = null;
      QuickViewServiceLineOhiOutputRow newOutputRow = null;
      
      // Local variables - service line adjudication level
      Set<EwvServiceLineOhi> serviceLineOhiSet = null;
      List<EwvServiceLineOhi> primaryServiceLineOhiList = null;
      List<EwvServiceLineOhi> secondaryServiceLineOhiList = null;
      EwvServiceLineOhi thisRowPrimaryAdjudication = null;
      EwvServiceLineOhi thisRowSecondaryAdjudication = null;
      
      // Local variables - service line adjudication adjustment level
      Set<EwvServiceLineOhiAdjtmt> primaryAdjustmentsSet = null;
      Set<EwvServiceLineOhiAdjtmt> secondaryAdjustmentsSet = null;
      List<EwvServiceLineOhiAdjtmt> primaryAdjustmentsList = null;
      List<EwvServiceLineOhiAdjtmt> secondaryAdjustmentsList = null;
      EwvServiceLineOhiAdjtmt thisRowPrimaryAdjustment = null;
      EwvServiceLineOhiAdjtmt thisRowSecondaryAdjustment = null;
      
      // Sort the service lines by criteria described in class EwvServiceLine
      serviceLineList = new ArrayList<EwvServiceLine>(serviceLines);
      Collections.sort(serviceLineList);
      //log.info("Service lines: " + serviceLineList.size());
      
      // Go through the service lines, putting together output rows 
      for (EwvServiceLine nextServiceLine : serviceLineList) {
        //log.info("Next service line");
        
        // Get the ohi records (actually adjudications) for this service line
        serviceLineOhiSet = nextServiceLine.getEwvServiceLineOhis();    
        if (serviceLineOhiSet != null && serviceLineOhiSet.size() > 0) {
          //log.info("Adjudications: " + serviceLineOhiSet.size());
          
          // Split the adjudications into two lists, one from the primary, one from the secondary
          // Ignore any tertiary adjudications, which are not shown in the Quickview
          // Check the payer id to determine which list to put each adjudication in
          // In a case of payers with the same id, this method would not have been called
          primaryServiceLineOhiList = new ArrayList<EwvServiceLineOhi> ();
          secondaryServiceLineOhiList = new ArrayList<EwvServiceLineOhi> ();
          for (EwvServiceLineOhi nextOhi : serviceLineOhiSet) {
            //log.info("ON SERVICE LINE OHI: " + nextOhi.getGuid());
            // If the payer ID has not been set (during dev or testing), use the old
            //   location, in the adjustment
            //log.info("Going to determinePayerId from Quickview");
            String nextOhiPayerId = determinePayerId(nextOhi, log);
            //log.info("GOT PAYER ID: " + nextOhiPayerId);
            
            if (primaryPayerId != null && primaryPayerId.equals(nextOhiPayerId)) {
              primaryServiceLineOhiList.add(nextOhi);
            } else if (secondaryPayerId != null && secondaryPayerId.equals(nextOhiPayerId)) {
              secondaryServiceLineOhiList.add(nextOhi);
            } else {
              // Must be a tertiary payer, which can be ignored for quickview
            }
          }
          //log.info("Primary adjudications: " + primaryServiceLineOhiList.size());
          //log.info("Secondary adjudications: " + secondaryServiceLineOhiList.size());
          
          // Find out how many adjudication rows will be made for this service line
          // One adjudication row shows a primary and/or secondary adjudication,
          //   as well as the first primary adjustment and/or first secondary adjustment
          int primaryOhiAdjudicationCount = primaryServiceLineOhiList.size();
          int secondaryOhiAdjudicationCount = secondaryServiceLineOhiList.size();
          int adjudicationRowCount = Math.max(primaryOhiAdjudicationCount, secondaryOhiAdjudicationCount);
          //log.info("Adjudication rows: " + adjudicationRowCount);
          
          // If there are adjudication rows, loop through them
          if (adjudicationRowCount > 0) {
          
            for (int i = 0; i < adjudicationRowCount; i++) {
              //log.info("Adjudication row: " + i);
              
              // Get the current primary and secondary adjudications
              if (i < primaryServiceLineOhiList.size()) {
                thisRowPrimaryAdjudication = primaryServiceLineOhiList.get(i);
              } else {
                thisRowPrimaryAdjudication = null;
              }
              if (i < secondaryServiceLineOhiList.size()) {
                thisRowSecondaryAdjudication = secondaryServiceLineOhiList.get(i);
              } else {
                thisRowSecondaryAdjudication = null;
              }
              //log.info("Primary Adjudication: " + thisRowPrimaryAdjudication);
              //log.info("Secondary Adjudication: " + thisRowSecondaryAdjudication);
            
              // Get the adjustments for the current primary and secondary adjudications
              if (thisRowPrimaryAdjudication != null) {
                primaryAdjustmentsSet = thisRowPrimaryAdjudication.getEwvServiceLineOhiAdjtmts();
                //log.info("Primary adjustments Set size: " + primaryAdjustmentsSet.size());
                primaryAdjustmentsList = new ArrayList(primaryAdjustmentsSet);
              } else {
                primaryAdjustmentsList = new ArrayList();
              }
            
              if (thisRowSecondaryAdjudication != null) {
                secondaryAdjustmentsSet = thisRowSecondaryAdjudication.getEwvServiceLineOhiAdjtmts();
                secondaryAdjustmentsList = new ArrayList(secondaryAdjustmentsSet);
              } else {
                secondaryAdjustmentsList = new ArrayList();
              }
              //log.info("Primary adjustments List size: " + primaryAdjustmentsList.size());
              //log.info("Secondary adjustments: " + secondaryAdjustmentsList.size());
            
              // Determine the maximum number of adjustments for this adjudication
              int primaryAdjustmentCount = primaryAdjustmentsList.size();
              int secondaryAdjustmentCount = secondaryAdjustmentsList.size();
              int adjustmentRowCount = Math.max(primaryAdjustmentCount, secondaryAdjustmentCount);
              //log.info("adjustmentRowCount: " + adjustmentRowCount);
              
              // If there are adjustment rows, loop through them
              if (adjustmentRowCount > 0) {
              
                // Loop up to the number of adjustment rows
                for (int j = 0; j < adjustmentRowCount; j++) {
                  //log.info("Adjustment Row #" + j);
                  // Get the current primary and secondary adjustments
                  if (j < primaryAdjustmentsList.size()) {
                    thisRowPrimaryAdjustment = primaryAdjustmentsList.get(j);
                  } else {
                    thisRowPrimaryAdjustment = null;
                  }
                  if (j < secondaryAdjustmentsList.size()) {
                    thisRowSecondaryAdjustment = secondaryAdjustmentsList.get(j);
                  } else {
                    thisRowSecondaryAdjustment = null;
                  }
                
                  // For the first adjudication and adjustment, include the service line stuff
                  EwvServiceLine serviceLineToPass = null;
                  
                  //log.info("Making output row, i: " + i + ", j: " + j);
                  
                  if (i == 0 && j == 0) {
                    serviceLineToPass = nextServiceLine;
                  }
                  
                  // For the first adjustment, include the adjudication stuff
                  EwvServiceLineOhi primaryAdjudicationToPass = null;
                  EwvServiceLineOhi secondaryAdjudicationToPass = null;
                  if (j == 0) {
                    primaryAdjudicationToPass = thisRowPrimaryAdjudication;
                    secondaryAdjudicationToPass = thisRowSecondaryAdjudication;
                  }
                  
                  newOutputRow = popQuickviewServiceLineOhiOutputRow(
                      serviceLineToPass,
                      primaryAdjudicationToPass,
                      secondaryAdjudicationToPass,
                      thisRowPrimaryAdjustment,
                      thisRowSecondaryAdjustment,
                      log);
                    
                  returnList.add(newOutputRow);
                  //log.info("Added an output row, size now: " + returnList.size());
                } // end of looping through the adjustment rows
              } else {
                // If there are no adjustments, make a single row anyway
                //   with just the service-line and adjudication stuff
                // For the first adjudication and adjustment, include the service line stuff
                EwvServiceLine serviceLineToPass = null;
                
                //log.info("Making output row, i: " + i);
                
                if (i == 0) {
                  serviceLineToPass = nextServiceLine;
                }
                
                newOutputRow = popQuickviewServiceLineOhiOutputRow(
                   serviceLineToPass,
                   thisRowPrimaryAdjudication,
                   thisRowSecondaryAdjudication,
                   null,
                   null,
                   log);

                returnList.add(newOutputRow);
              }
            } // end of looping through the adjudication rows
          } else {
            // If there are no adjudications, make a row anyway, with just the service-line stuff
            newOutputRow = popQuickviewServiceLineOhiOutputRow(nextServiceLine,
                                                               null,
                                                               null,
                                                               null,
                                                               null,
                                                               log);

            returnList.add(newOutputRow);
          }
        }
          
      } // End of service lines
      
      return returnList;
    }

  private QuickViewServiceLineOhiOutputRow popQuickviewServiceLineOhiOutputRow
    (EwvServiceLine inServiceLine,
     EwvServiceLineOhi inPrimaryOhi, 
     EwvServiceLineOhi inSecondaryOhi, 
     EwvServiceLineOhiAdjtmt inPrimaryAdjustment,
     EwvServiceLineOhiAdjtmt inSecondaryAdjustment,
     Log log)
  {
    QuickViewServiceLineOhiOutputRow outputRow = new QuickViewServiceLineOhiOutputRow();
    
    // Service-line-level fields
    if (inServiceLine != null) {
      outputRow.setLineNumber(Ewv2Utils.formatAsInteger(inServiceLine.getLineNumber()));
      outputRow.setBeginDateOfService(Ewv2Utils.formatAsDate(inServiceLine.getBeginDateOfService()));
      outputRow.setCharge(Ewv2Utils.formatAsCurrency(inServiceLine.getCharge()));
    }
    
    // Primary Service-line-ohi-level fields
    if (inPrimaryOhi != null) {
      outputRow.setPrimaryPayerPaidAmount(Ewv2Utils.formatAsCurrency(inPrimaryOhi.getOhiPaid()));
      outputRow.setPrimaryPayerProcedureCode(inPrimaryOhi.getProcedureCode());
      outputRow.setOhiModifiers(Ewv2Utils.combineModifiers(
              ", ",
              inPrimaryOhi.getModifier1(),
              inPrimaryOhi.getModifier2(),
              inPrimaryOhi.getModifier3(),
              inPrimaryOhi.getModifier4()));
    }
    
    // Secondary Service-line-ohi-level fields
    if (inSecondaryOhi != null) {
      outputRow.setSecondaryPayerPaidAmount(Ewv2Utils.formatAsCurrency(inSecondaryOhi.getOhiPaid()));
      outputRow.setSecondaryPayerProcedureCode(inSecondaryOhi.getProcedureCode());
    }
    
    // Primary Service-line-ohi-adjustment-level fields
    if (inPrimaryAdjustment != null) {
      ServiceLineOhiAdjustment primaryAdjustment = createServiceLineOhiAdjustment(inPrimaryAdjustment);
      outputRow.setPrimaryServiceLineAdjustment(primaryAdjustment);
    }

    // Secondary Service-line-ohi-adjustment-level fields
    if (inSecondaryAdjustment != null) {
      ServiceLineOhiAdjustment secondaryAdjustment
        = createServiceLineOhiAdjustment(inSecondaryAdjustment);
      outputRow.setSecondaryServiceLineAdjustment(secondaryAdjustment);
    }

    return outputRow;
  }
    
//  private List<QuickViewServiceLineOhiOutputRow> popQuickviewServiceLineOhiOutputRowsOrig
//    (Set<EwvServiceLine> serviceLines,
//     String primaryPayerId, 
//     String secondaryPayerId,
//     Log log)
//  {
//  List<QuickViewServiceLineOhiOutputRow> returnList = new ArrayList<QuickViewServiceLineOhiOutputRow>();
//  
//  // Sort the service lines by criteria described in class EwvServiceLine
//  List<EwvServiceLine> serviceLineList = new ArrayList<EwvServiceLine>(serviceLines);
//  Collections.sort(serviceLineList);
//  
//  // Go through the service lines, putting together output rows 
//  for (EwvServiceLine nextServiceLine : serviceLineList) {
//    
//    // Get the ohi records (actually adjudications) for this service line
//    Set<EwvServiceLineOhi> serviceLineOhiSet = nextServiceLine.getEwvServiceLineOhis();
//    
//    // If there is an ohi for this service line, make at least one row for it
//    if (serviceLineOhiSet != null && serviceLineOhiSet.size() > 0) {
//      // Decide which ohi is primary, which secondary
//      EwvServiceLineOhi primaryOhi = determineServiceLineOhiForPayer
//          (serviceLineOhiSet, primaryPayerId, log);
//      EwvServiceLineOhi secondaryOhi = determineServiceLineOhiForPayer
//          (serviceLineOhiSet, secondaryPayerId, log);
//      
//      // Get the primary adjustments (adjustments from the primary payer)
//      List<EwvServiceLineOhiAdjtmt> primaryAdjustmentsList = null;
//      if (primaryOhi != null) {
//        Set<EwvServiceLineOhiAdjtmt> primaryAdjustmentsSet = primaryOhi.getEwvServiceLineOhiAdjtmts();
//        primaryAdjustmentsList = new ArrayList(primaryAdjustmentsSet);
//      }
//      
//      // Get the secondary adjustments (adjustments from the secondary payer)
//      List<EwvServiceLineOhiAdjtmt> secondaryAdjustmentsList = null;
//      if (secondaryOhi != null) {
//        Set<EwvServiceLineOhiAdjtmt> secondaryAdjustmentsSet = secondaryOhi.getEwvServiceLineOhiAdjtmts();
//        secondaryAdjustmentsList = new ArrayList(secondaryAdjustmentsSet);
//      }
//      
//      // Create an output row.  (Each service line with ohis will get at least one output row.)
//      QuickViewServiceLineOhiOutputRow newOutputRow = new QuickViewServiceLineOhiOutputRow();
//      
//      // Put the service-line-level info into the output row
//      newOutputRow.setLineNumber(Ewv2Utils.formatAsInteger(nextServiceLine.getLineNumber()));
//      newOutputRow.setBeginDateOfService(Ewv2Utils.formatAsDate(nextServiceLine.getBeginDateOfService()));
//      newOutputRow.setCharge(Ewv2Utils.formatAsCurrency(nextServiceLine.getCharge()));
//      
//      // Put primary-specific info, if any, into the output row
//      if (primaryAdjustmentsList != null && primaryAdjustmentsList.size() > 0) {
//        newOutputRow.setPrimaryPayerPaidAmount(Ewv2Utils.formatAsCurrency(primaryOhi.getOhiPaid()));
//        newOutputRow.setPrimaryPayerProcedureCode(primaryOhi.getProcedureCode());
//
//        // And adjustments
//        EwvServiceLineOhiAdjtmt firstPrimaryAdjustmentDb = primaryAdjustmentsList.get(0);
//        ServiceLineOhiAdjustment firstPrimaryAdjustment
//            = createServiceLineOhiAdjustment(primaryPayerId, firstPrimaryAdjustmentDb);
//        newOutputRow.setPrimaryServiceLineAdjustment(firstPrimaryAdjustment);
//      }
//      
//      // Put secondary-specific info, if any, into the output row
//      if (secondaryAdjustmentsList != null && secondaryAdjustmentsList.size() > 0) {
//        newOutputRow.setSecondaryPayerPaidAmount(Ewv2Utils.formatAsCurrency(secondaryOhi.getOhiPaid()));
//        newOutputRow.setPrimaryPayerProcedureCode(secondaryOhi.getProcedureCode());
//
//        // And adjustments
//        EwvServiceLineOhiAdjtmt firstSecondaryAdjustmentDb = secondaryAdjustmentsList.get(0);
//        ServiceLineOhiAdjustment firstSecondaryAdjustment
//            = createServiceLineOhiAdjustment(secondaryPayerId, firstSecondaryAdjustmentDb);
//        newOutputRow.setSecondaryServiceLineAdjustment(firstSecondaryAdjustment);
//      }
//      
//      // If we could not determine a primary or secondary, punt
//      if (primaryOhi == null && secondaryOhi == null) {
//        newOutputRow.setCharge("Error");
//      }
//      
//      // Finish with this output row 
//      returnList.add(newOutputRow);
//    
//      // Keep adding output rows until there are no more adjustments of either type
//      int primaryListSize = 0;
//      if (primaryAdjustmentsList != null) {
//        primaryListSize = primaryAdjustmentsList.size();
//      }
//      int secondaryListSize = 0;
//      if (secondaryAdjustmentsList != null) {
//        secondaryListSize = secondaryAdjustmentsList.size();
//      }
//      int maxAdjustments = Math.max(primaryListSize, secondaryListSize);
//      for (int rowCounter = 1; rowCounter < maxAdjustments; rowCounter++) {
//      
//        // Create an output row.  (Each service line will get at least one output row.)
//        QuickViewServiceLineOhiOutputRow nextOutputRow = new QuickViewServiceLineOhiOutputRow();
//        
//        // Put the primary info into the output row
//        if (primaryListSize >= rowCounter + 1) {
//          EwvServiceLineOhiAdjtmt nextPrimaryAdjustmentDb = primaryAdjustmentsList.get(rowCounter);
//          ServiceLineOhiAdjustment nextPrimaryAdjustment
//              = createServiceLineOhiAdjustment(primaryPayerId, nextPrimaryAdjustmentDb);
//          nextOutputRow.setPrimaryServiceLineAdjustment(nextPrimaryAdjustment);
//        }
//        
//        // Put the secondary info into the output row
//        if (secondaryListSize >= rowCounter + 1) {
//          EwvServiceLineOhiAdjtmt nextSecondaryAdjustmentDb = secondaryAdjustmentsList.get(rowCounter);
//          ServiceLineOhiAdjustment nextSecondaryAdjustment
//              = createServiceLineOhiAdjustment(secondaryPayerId, nextSecondaryAdjustmentDb);
//          nextOutputRow.setSecondaryServiceLineAdjustment(nextSecondaryAdjustment);
//        }
//          
//        returnList.add(nextOutputRow);
//      }
//    } // End of checking whether there are ohis for this service line.
//  } // End of service lines
//  
//    return returnList;
//  }

    private ServiceLineOhiAdjustment createServiceLineOhiAdjustment(EwvServiceLineOhiAdjtmt adjustment)
    {
//      // first go with the payer id passed in
//      String payerId = ohiPayerId;
//      if (ohiPayerId == null || ohiPayerId.length() == 0) {
//        payerId = adjustment.getPrimaryPayerId();
//      }
      
      String quantity = adjustment.getQuantity();
      String descriptionLong = adjustment.getDescription();
      String descriptionShort = Ewv2Utils.shorten(descriptionLong, 15);
      String reasonCode = adjustment.getSvcLineAdjustmtReasonCode();
      
      String coAmount = adjustment.getCoAmt();
      String crAmount = adjustment.getCrAmt();
      String oaAmount = adjustment.getOaAmt();
      String piAmount = adjustment.getPiAmt();
      String prAmount = adjustment.getPrAmt();
      
      ServiceLineOhiAdjustment ret = new ServiceLineOhiAdjustment(
//          payerId, 
          descriptionLong, 
          descriptionShort,
          reasonCode,
          quantity, 
          coAmount,
          crAmount,
          oaAmount,
          piAmount, 
          prAmount);
      
      if (coAmount != null) {
          ret.setAdjustmentType(AdjustmentType.CO);
      } else if (crAmount != null) {
          ret.setAdjustmentType(AdjustmentType.CR);
      } else if (oaAmount != null) {
        ret.setAdjustmentType(AdjustmentType.OA);
      } else if (piAmount != null){
        ret.setAdjustmentType(AdjustmentType.PI);
      } else if (prAmount != null) {
        ret.setAdjustmentType(AdjustmentType.PR);
      }
    
      return ret;
    }

    private EwvServiceLineOhi determineServiceLineOhiForPayer
              (Set<EwvServiceLineOhi> serviceLineOhiSet, String payerId, Log log) {
  
      EwvServiceLineOhi returnOhi = null;
      
      // Loop through the service line ohis, looking for one that "matches" the payer id
      for (EwvServiceLineOhi nextOhi : serviceLineOhiSet) {
      
        // First, look in the payer_id column of service line ohis
        if (payerId != null && payerId.equals(nextOhi.getPayerId())) {
          returnOhi = nextOhi;
        }
        
        // If nothing is there, check the adjustments' primary_payer_id field.  This is
        //   deprecated for new data, but older data might still have it there.
        else {
          Set<EwvServiceLineOhiAdjtmt> adjustments = nextOhi.getEwvServiceLineOhiAdjtmts();
          if (adjustments != null && adjustments.size() > 0) {
            String payerIdFromAdjustments = null;
            for (EwvServiceLineOhiAdjtmt nextAdjustment : adjustments) {
              String nextPayerId = nextAdjustment.getPrimaryPayerId();
              if (nextPayerId != null && !nextPayerId.isEmpty()) {
                payerIdFromAdjustments = nextPayerId;
              }
            }
            
            if (payerIdFromAdjustments != null && !payerIdFromAdjustments.isEmpty()) {
              if (payerIdFromAdjustments.equals(payerId)) {
                returnOhi = nextOhi;
              }
            }
          }
        }
      }
      
      return returnOhi;
    }

    public boolean getHasMultiplePayersWithSamePayerId(EwvClaim claimBean, Log log) {
      boolean returnValue = false;
      
      // Get the EWV_Claim_Ohi records
      Set<EwvClaimOhi> claimOhiSet = claimBean.getEwvClaimOhis();
      
      // Loop through them, getting out their payer id values
      List<String> payerIdList = new ArrayList<String>();
      for (EwvClaimOhi claimOhi : claimOhiSet) {
        payerIdList.add(claimOhi.getOhiPayerId());
      }
      
      // Find out whether there are duplicates
      returnValue = Ewv2Utils.hasNonNullDuplicates(payerIdList);
      
      return returnValue;
    }
    
    public boolean getHasMultiplePrimaryPayers(EwvClaim claimBean, Log log) {
      // Get the EWV_Claim_Ohi records
      Set<EwvClaimOhi> claimOhiSet = claimBean.getEwvClaimOhis();
      
      // Count how many of them are marked as Primary
      int primaryOhiCounter = 0;
      for (EwvClaimOhi claimOhi : claimOhiSet) {
        if (claimOhi.getPaymentSequenceIndicator().equals("P")) {
          primaryOhiCounter += 1;
        }
      }
      
      return (primaryOhiCounter > 1);
    }
    
    public boolean getHasMultipleSecondaryPayers(EwvClaim claimBean, Log log) {
      // Get the EWV_Claim_Ohi records
      Set<EwvClaimOhi> claimOhiSet = claimBean.getEwvClaimOhis();
      
      // Count how many of them are marked as Primary
      int secondaryOhiCounter = 0;
      for (EwvClaimOhi claimOhi : claimOhiSet) {
        if (claimOhi.getPaymentSequenceIndicator().equals("S")) {
          secondaryOhiCounter += 1;
        }
      }
      
      return (secondaryOhiCounter > 1);
    }
    
    public boolean getHasMultipleAdjudicationsFromSamePayer(EwvClaim claimBean, Log log) {
      boolean returnValue = false;
      List<String> adjudicationList;
      
      // Loop through the service lines
      Set<EwvServiceLine> serviceLineSet = claimBean.getEwvServiceLines();
      for (EwvServiceLine serviceLine : serviceLineSet) {
      
      
        // For each service line, put together a list of its adjudications
        Set<EwvServiceLineOhi> serviceLineOhiSet = serviceLine.getEwvServiceLineOhis();
        adjudicationList = new ArrayList<String>();
        for (EwvServiceLineOhi serviceLineOhi : serviceLineOhiSet) {
          adjudicationList.add(serviceLineOhi.getPayerId());
        }
        
        // Find out whether there are duplicates
        boolean dupsExist = Ewv2Utils.hasNonNullDuplicates(adjudicationList);
        if (dupsExist) {
          returnValue = true;
        }
      }
      
      return returnValue;
    }
    
  private String determinePaymentSequenceIndicator(String payerId,
                                                   Set<EwvClaimOhi> claimOhiSet,
                                                   Log log) {
    // If no matches are found, display hyphens
    // This should never happen, though.
    String returnVal = "---";
    
    // Go through the claimOhiSet, looking for a match
    // We can assume no more than one match, because that scenario is checked earlier,
    //   and this procedure is not called if there are multiple payers with the same payer id
    //   (Instead, a ? appears)
    for (EwvClaimOhi nextClaimOhi : claimOhiSet) {
      if (payerId.equals(nextClaimOhi.getOhiPayerId())) {
        returnVal = nextClaimOhi.getPaymentSequenceIndicator();
        break;
      }
    }
  
  return returnVal;
  }

  private String determinePayerId(EwvServiceLineOhi inOhi, Log log) {
    String returnValue = null;
    
    //log.info("In determinePayerId with service line ohi: " + inOhi.getGuid());
    
    // First, use the new location, on the service line ohi, where it should be
    returnValue = inOhi.getPayerId();
    if (returnValue != null) {
      return returnValue;
    }
    
    // If that does not work, look through its adjustments.
    // The Phase I code actually took the last adjustment's value, so we finish the loop here
    //log.info("Trying adjustments");
    Set<EwvServiceLineOhiAdjtmt> serviceLineOhiAdjustments = inOhi.getEwvServiceLineOhiAdjtmts();
    for (EwvServiceLineOhiAdjtmt nextAdjustment : serviceLineOhiAdjustments) {
      if (nextAdjustment.getPrimaryPayerId() != null) {
        returnValue = nextAdjustment.getPrimaryPayerId();
      }
    }
    if (returnValue != null) {
      //log.info("Found payer id in adjustments");
      return returnValue;
    }
    
    // If that still does not work, try the other service line ohis
    //log.info("Trying all adjustments");
    EwvServiceLine serviceLine = inOhi.getEwvServiceLine();
    Set<EwvServiceLineOhi> allOhis = serviceLine.getEwvServiceLineOhis();
    //log.info("Size of service line ohis: " + allOhis.size());
    for (EwvServiceLineOhi nextOhi : allOhis) {
      //log.info("Trying ohi with guid: " + nextOhi.getGuid());
      Set<EwvServiceLineOhiAdjtmt> allAdjustmentsThisOhi = nextOhi.getEwvServiceLineOhiAdjtmts();
      for (EwvServiceLineOhiAdjtmt nextAdjustment : allAdjustmentsThisOhi) {
        //log.info("Trying adjustment with guid: " + nextAdjustment.getGuid());
        if (nextAdjustment.getPrimaryPayerId() != null) {
          returnValue = nextAdjustment.getPrimaryPayerId();
        }
      }
    }
    if (returnValue != null) {
      //log.info("Found payer id in other service line ohi adjustments");
      return returnValue;
    }
    
    // If that still does not work, try the other service lines
    //log.info("Trying all service lines");
    EwvClaim claim = serviceLine.getEwvClaim();
    Set<EwvServiceLine> allServiceLines = claim.getEwvServiceLines();
    for (EwvServiceLine nextServiceLine : allServiceLines) {
      //log.info("Trying service line with guid: " + nextServiceLine.getGuid());
      Set<EwvServiceLineOhi> thisServiceLineOhis = nextServiceLine.getEwvServiceLineOhis();
      for (EwvServiceLineOhi nextOhi : thisServiceLineOhis) {
        //log.info("Trying ohi with guid: " + nextOhi.getGuid());
        Set<EwvServiceLineOhiAdjtmt> allAdjustmentsThisOhi = nextOhi.getEwvServiceLineOhiAdjtmts();
        for (EwvServiceLineOhiAdjtmt nextAdjustment : allAdjustmentsThisOhi) {
          //log.info("Trying adjustment with guid: " + nextAdjustment.getGuid());
          if (nextAdjustment.getPrimaryPayerId() != null) {
            returnValue = nextAdjustment.getPrimaryPayerId();
            //log.info("Found payer id in other service line adjustments");
          }
        }
      }
    }
    
    return returnValue;
  }
}
